<?php

namespace App\Http\Controllers;

use App\Models\Supplier;
use App\Models\PurchaseInvoice;
use App\Models\PurchaseOrder;
use Illuminate\Http\Request;
use Carbon\Carbon;
use Illuminate\Support\Facades\DB;

class SupplierReportController extends Controller
{
    /**
     * Display supplier reports dashboard
     */
    public function index()
    {
        return view('system-reports.supplier.index');
    }

    /**
     * Supplier summary report
     */
    public function summary(Request $request)
    {
        $totalSuppliers = Supplier::count();
        $activeSuppliers = Supplier::where('is_active', true)->count();
        $inactiveSuppliers = Supplier::where('is_active', false)->count();

        $supplierStats = Supplier::selectRaw('
                COUNT(*) as total_suppliers,
                COUNT(CASE WHEN is_active = 1 THEN 1 END) as active_suppliers,
                COUNT(CASE WHEN is_active = 0 THEN 1 END) as inactive_suppliers
            ')
            ->first();

        $topSuppliers = Supplier::leftJoin('purchase_invoices', 'suppliers.id', '=', 'purchase_invoices.supplier_id')
            ->selectRaw('
                suppliers.id,
                suppliers.name,
                suppliers.email,
                suppliers.phone,
                COUNT(purchase_invoices.id) as total_invoices,
                COALESCE(SUM(purchase_invoices.total_amount), 0) as total_purchases
            ')
            ->groupBy('suppliers.id', 'suppliers.name', 'suppliers.email', 'suppliers.phone')
            ->orderByDesc('total_purchases')
            ->limit(10)
            ->get();

        return view('system-reports.supplier.summary', compact('supplierStats', 'topSuppliers'));
    }

    /**
     * Supplier performance report
     */
    public function performance(Request $request)
    {
        $startDate = $request->get('start_date', now()->startOfMonth());
        $endDate = $request->get('end_date', now()->endOfMonth());

        $supplierPerformance = Supplier::leftJoin('purchase_orders', 'suppliers.id', '=', 'purchase_orders.supplier_id')
            ->leftJoin('purchase_invoices', 'suppliers.id', '=', 'purchase_invoices.supplier_id')
            ->whereBetween('purchase_invoices.invoice_date', [$startDate, $endDate])
            ->selectRaw('
                suppliers.id,
                suppliers.name,
                suppliers.email,
                suppliers.phone,
                suppliers.created_at as registration_date,
                COUNT(DISTINCT purchase_orders.id) as total_orders,
                COUNT(DISTINCT purchase_invoices.id) as total_invoices,
                COALESCE(SUM(purchase_invoices.total_amount), 0) as total_purchases,
                COALESCE(AVG(purchase_invoices.total_amount), 0) as avg_order_value,
                MAX(purchase_invoices.invoice_date) as last_purchase_date,
                DATEDIFF(NOW(), MAX(purchase_invoices.invoice_date)) as days_since_last_purchase,
                COUNT(CASE WHEN purchase_orders.status = "completed" THEN 1 END) as completed_orders,
                COUNT(CASE WHEN purchase_orders.status = "cancelled" THEN 1 END) as cancelled_orders,
                CASE 
                    WHEN COUNT(purchase_orders.id) > 0 
                    THEN (COUNT(CASE WHEN purchase_orders.status = "completed" THEN 1 END) / COUNT(purchase_orders.id) * 100)
                    ELSE 0 
                END as completion_rate
            ')
            ->groupBy('suppliers.id', 'suppliers.name', 'suppliers.email', 'suppliers.phone', 'suppliers.created_at')
            ->orderByDesc('total_purchases')
            ->paginate(15);

        return view('system-reports.supplier.performance', compact('supplierPerformance', 'startDate', 'endDate'));
    }

    /**
     * Supplier payment analysis
     */
    public function paymentAnalysis(Request $request)
    {
        $startDate = $request->get('start_date', now()->startOfMonth());
        $endDate = $request->get('end_date', now()->endOfMonth());

        $paymentAnalysis = Supplier::leftJoin('purchase_invoices', 'suppliers.id', '=', 'purchase_invoices.supplier_id')
            ->whereBetween('purchase_invoices.invoice_date', [$startDate, $endDate])
            ->selectRaw('
                suppliers.id,
                suppliers.name,
                suppliers.payment_terms,
                COUNT(purchase_invoices.id) as total_invoices,
                SUM(purchase_invoices.total_amount) as total_amount,
                SUM(purchase_invoices.paid_amount) as total_paid,
                SUM(purchase_invoices.total_amount - purchase_invoices.paid_amount) as outstanding_balance,
                0 as avg_payment_days,
                COUNT(CASE WHEN purchase_invoices.payment_status = "paid" THEN 1 END) as paid_invoices,
                COUNT(CASE WHEN purchase_invoices.payment_status = "overdue" THEN 1 END) as overdue_invoices
            ')
            ->groupBy('suppliers.id', 'suppliers.name', 'suppliers.payment_terms')
            ->orderByDesc('outstanding_balance')
            ->paginate(15);

        $paymentStats = DB::select("
            SELECT 
                SUM(total_amount) as total_purchases,
                SUM(paid_amount) as total_paid,
                SUM(total_amount - paid_amount) as total_outstanding,
                0 as avg_payment_days
            FROM purchase_invoices 
            WHERE invoice_date BETWEEN ? AND ?
        ", [$startDate, $endDate])[0];

        return view('system-reports.supplier.payment-analysis', compact('paymentAnalysis', 'paymentStats', 'startDate', 'endDate'));
    }

    /**
     * Supplier quality report
     */
    public function quality(Request $request)
    {
        $startDate = $request->get('start_date', now()->startOfMonth());
        $endDate = $request->get('end_date', now()->endOfMonth());

        // This would require additional tables for quality metrics, returns, etc.
        $qualityMetrics = Supplier::leftJoin('purchase_orders', 'suppliers.id', '=', 'purchase_orders.supplier_id')
            ->whereBetween('purchase_orders.order_date', [$startDate, $endDate])
            ->selectRaw('
                suppliers.id,
                suppliers.name,
                COUNT(purchase_orders.id) as total_orders,
                0 as avg_delay_days,
                0 as on_time_deliveries,
                0 as late_deliveries,
                CASE
                    WHEN COUNT(purchase_orders.id) > 0
                    THEN (COUNT(CASE WHEN purchase_orders.status = "received" THEN 1 END) / COUNT(purchase_orders.id) * 100)
                    ELSE 0
                END as on_time_percentage
            ')
            ->groupBy('suppliers.id', 'suppliers.name')
            ->orderByDesc('on_time_percentage')
            ->paginate(15);

        return view('system-reports.supplier.quality', compact('qualityMetrics', 'startDate', 'endDate'));
    }

    /**
     * Supplier comparison report
     */
    public function comparison(Request $request)
    {
        $startDate = $request->get('start_date', now()->startOfMonth());
        $endDate = $request->get('end_date', now()->endOfMonth());
        $metric = $request->get('metric', 'total_purchases');

        $comparison = Supplier::leftJoin('purchase_invoices', 'suppliers.id', '=', 'purchase_invoices.supplier_id')
            ->whereBetween('purchase_invoices.invoice_date', [$startDate, $endDate])
            ->selectRaw('
                suppliers.id,
                suppliers.name,
                suppliers.country,
                suppliers.payment_terms,
                COUNT(purchase_invoices.id) as total_invoices,
                SUM(purchase_invoices.total_amount) as total_purchases,
                AVG(purchase_invoices.total_amount) as avg_order_value,
                MIN(purchase_invoices.total_amount) as min_order_value,
                MAX(purchase_invoices.total_amount) as max_order_value
            ')
            ->groupBy('suppliers.id', 'suppliers.name', 'suppliers.country', 'suppliers.payment_terms')
            ->orderByDesc($metric)
            ->limit(20)
            ->get();

        return view('system-reports.supplier.comparison', compact('comparison', 'startDate', 'endDate', 'metric'));
    }

    /**
     * Supplier geographic analysis
     */
    public function geographic(Request $request)
    {
        $geographicData = Supplier::leftJoin('purchase_invoices', 'suppliers.id', '=', 'purchase_invoices.supplier_id')
            ->selectRaw('
                suppliers.country,
                suppliers.city,
                COUNT(DISTINCT suppliers.id) as supplier_count,
                COUNT(purchase_invoices.id) as total_invoices,
                SUM(purchase_invoices.total_amount) as total_purchases,
                AVG(purchase_invoices.total_amount) as avg_order_value
            ')
            ->groupBy('suppliers.country', 'suppliers.city')
            ->orderByDesc('total_purchases')
            ->get()
            ->groupBy('country');

        return view('system-reports.supplier.geographic', compact('geographicData'));
    }

    /**
     * Supplier aging report
     */
    public function aging(Request $request)
    {
        $asOfDate = $request->get('as_of_date', now()->format('Y-m-d'));

        $agingData = DB::select("
            SELECT 
                suppliers.id,
                suppliers.name,
                SUM(purchase_invoices.total_amount - purchase_invoices.paid_amount) as total_outstanding,
                SUM(CASE WHEN DATEDIFF(?, purchase_invoices.due_date) <= 30 THEN (purchase_invoices.total_amount - purchase_invoices.paid_amount) ELSE 0 END) as current_amount,
                SUM(CASE WHEN DATEDIFF(?, purchase_invoices.due_date) BETWEEN 31 AND 60 THEN (purchase_invoices.total_amount - purchase_invoices.paid_amount) ELSE 0 END) as days_31_60,
                SUM(CASE WHEN DATEDIFF(?, purchase_invoices.due_date) BETWEEN 61 AND 90 THEN (purchase_invoices.total_amount - purchase_invoices.paid_amount) ELSE 0 END) as days_61_90,
                SUM(CASE WHEN DATEDIFF(?, purchase_invoices.due_date) > 90 THEN (purchase_invoices.total_amount - purchase_invoices.paid_amount) ELSE 0 END) as days_over_90
            FROM suppliers
            LEFT JOIN purchase_invoices ON suppliers.id = purchase_invoices.supplier_id
            WHERE purchase_invoices.payment_status != 'paid'
            GROUP BY suppliers.id, suppliers.name
            HAVING total_outstanding > 0
            ORDER BY total_outstanding DESC
        ", [$asOfDate, $asOfDate, $asOfDate, $asOfDate]);

        return view('system-reports.supplier.aging', compact('agingData', 'asOfDate'));
    }

    /**
     * Export supplier report
     */
    public function export(Request $request)
    {
        $type = $request->get('type', 'summary');
        $format = $request->get('format', 'excel');
        
        // Implementation for export functionality
        return redirect()->back()->with('success', 'تم تصدير التقرير بنجاح');
    }
}