<?php

namespace App\Http\Controllers;

use App\Models\Supplier;
use App\Models\PurchaseInvoice;
use App\Models\PaymentVoucher;
use App\Models\JournalEntry;
use App\Models\CashTransaction;
use App\Models\BankAccount;
use Illuminate\Http\Request;
use Illuminate\View\View;
use Illuminate\Http\RedirectResponse;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;

class SupplierPaymentController extends Controller
{
    /**
     * عرض قائمة مدفوعات الموردين
     */
    public function index(Request $request): View
    {
        $query = PaymentVoucher::with(['supplier', 'user'])
            ->whereNotNull('supplier_id')
            ->latest('payment_date');

        // فلترة حسب المورد
        if ($request->filled('supplier_id')) {
            $query->where('supplier_id', $request->supplier_id);
        }

        // فلترة حسب طريقة الدفع
        if ($request->filled('payment_method')) {
            $query->where('payment_method', $request->payment_method);
        }

        // فلترة حسب التاريخ
        if ($request->filled('date_from')) {
            $query->whereDate('payment_date', '>=', $request->date_from);
        }
        
        if ($request->filled('date_to')) {
            $query->whereDate('payment_date', '<=', $request->date_to);
        }

        // فلترة حسب المبلغ
        if ($request->filled('amount_from')) {
            $query->where('amount', '>=', $request->amount_from);
        }
        
        if ($request->filled('amount_to')) {
            $query->where('amount', '<=', $request->amount_to);
        }

        // البحث النصي
        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function($q) use ($search) {
                $q->where('description', 'like', "%{$search}%")
                  ->orWhere('reference', 'like', "%{$search}%")
                  ->orWhereHas('supplier', function($sq) use ($search) {
                      $sq->where('name', 'like', "%{$search}%");
                  });
            });
        }

        $payments = $query->paginate(15)->withQueryString();
        $suppliers = Supplier::active()->get();

        // إحصائيات سريعة
        $stats = [
            'total_payments' => PaymentVoucher::whereNotNull('supplier_id')->count(),
            'total_amount' => PaymentVoucher::whereNotNull('supplier_id')->sum('amount'),
            'this_month_payments' => PaymentVoucher::whereNotNull('supplier_id')
                ->whereMonth('payment_date', now()->month)
                ->whereYear('payment_date', now()->year)
                ->sum('amount'),
            'pending_payments' => $this->getPendingPaymentsAmount(),
        ];

        return view('suppliers.payments.index', compact('payments', 'suppliers', 'stats'));
    }

    /**
     * عرض نموذج إنشاء دفعة جديدة
     */
    public function create(Request $request): View
    {
        $supplierId = $request->get('supplier_id');
        $supplier = null;
        $outstandingInvoices = collect();

        if ($supplierId) {
            $supplier = Supplier::findOrFail($supplierId);
            $outstandingInvoices = $supplier->purchaseInvoices()
                ->where('payment_status', '!=', 'paid')
                ->where('status', '!=', 'cancelled')
                ->where('remaining_amount', '>', 0)
                ->orderBy('due_date')
                ->get();
        }

        $suppliers = Supplier::active()->where('current_balance', '>', 0)->get();
        $bankAccounts = BankAccount::active()->get();

        return view('suppliers.payments.create', compact('supplier', 'suppliers', 'outstandingInvoices', 'bankAccounts'));
    }

    /**
     * حفظ دفعة جديدة
     */
    public function store(Request $request): RedirectResponse
    {
        $validated = $request->validate([
            'supplier_id' => 'required|exists:suppliers,id',
            'payment_date' => 'required|date|before_or_equal:today',
            'payment_method' => 'required|in:cash,bank_transfer,check,credit_card',
            'bank_account_id' => 'required_if:payment_method,bank_transfer|nullable|exists:bank_accounts,id',
            'check_number' => 'required_if:payment_method,check|nullable|string|max:50',
            'check_date' => 'required_if:payment_method,check|nullable|date',
            'reference' => 'nullable|string|max:100',
            'description' => 'required|string|max:255',
            'currency' => 'required|string|max:3',
            'exchange_rate' => 'required|numeric|min:0.0001',
            'notes' => 'nullable|string',
            'payment_allocation' => 'required|in:manual,automatic',
            'invoices' => 'required_if:payment_allocation,manual|array',
            'invoices.*.invoice_id' => 'required_if:payment_allocation,manual|exists:purchase_invoices,id',
            'invoices.*.amount' => 'required_if:payment_allocation,manual|numeric|min:0.01',
            'total_amount' => 'required|numeric|min:0.01',
        ]);

        DB::transaction(function () use ($validated) {
            $supplier = Supplier::findOrFail($validated['supplier_id']);

            // التحقق من أن المبلغ لا يتجاوز المديونية
            if ($validated['total_amount'] > $supplier->current_balance) {
                throw new \Exception('المبلغ يتجاوز إجمالي المديونية المستحقة');
            }

            // إنشاء سند الدفع
            $paymentVoucher = PaymentVoucher::create([
                'supplier_id' => $validated['supplier_id'],
                'supplier_name' => $supplier->name,
                'payment_date' => $validated['payment_date'],
                'payment_method' => $validated['payment_method'],
                'bank_account_id' => $validated['bank_account_id'],
                'check_number' => $validated['check_number'],
                'check_date' => $validated['check_date'],
                'reference' => $validated['reference'],
                'description' => $validated['description'],
                'amount' => $validated['total_amount'],
                'currency' => $validated['currency'],
                'exchange_rate' => $validated['exchange_rate'],
                'notes' => $validated['notes'],
                'user_id' => auth()->id(),
                'company_id' => 1, // يمكن تعديلها حسب الحاجة
            ]);

            // توزيع المبلغ على الفواتير
            if ($validated['payment_allocation'] === 'manual') {
                $this->allocatePaymentManually($paymentVoucher, $validated['invoices']);
            } else {
                $this->allocatePaymentAutomatically($paymentVoucher, $supplier);
            }

            // تحديث رصيد المورد
            $supplier->updateBalance($validated['total_amount'], 'subtract');

            // إنشاء قيد محاسبي
            $this->createJournalEntry($paymentVoucher);

            // إنشاء حركة نقدية أو بنكية
            $this->createCashOrBankTransaction($paymentVoucher);
        });

        return redirect()->route('suppliers.payments.index')
            ->with('success', 'تم تسجيل الدفعة بنجاح');
    }

    /**
     * عرض تفاصيل الدفعة
     */
    public function show(PaymentVoucher $payment): View
    {
        $payment->load(['supplier', 'user', 'bankAccount']);
        
        // جلب الفواتير المرتبطة بهذه الدفعة
        $allocatedInvoices = $this->getPaymentAllocations($payment);
        
        return view('suppliers.payments.show', compact('payment', 'allocatedInvoices'));
    }

    /**
     * حذف دفعة
     */
    public function destroy(PaymentVoucher $payment): RedirectResponse
    {
        // التحقق من إمكانية الحذف (خلال 24 ساعة فقط)
        if ($payment->created_at->diffInHours(now()) > 24) {
            return redirect()->back()->with('error', 'لا يمكن حذف الدفعة بعد مرور 24 ساعة');
        }

        DB::transaction(function () use ($payment) {
            // عكس تأثير الدفعة على الفواتير
            $this->reversePaymentAllocations($payment);

            // استرداد رصيد المورد
            $payment->supplier->updateBalance($payment->amount, 'add');

            // حذف القيد المحاسبي
            if ($payment->journal_entry_id) {
                JournalEntry::find($payment->journal_entry_id)?->delete();
            }

            // حذف الحركة النقدية/البنكية
            CashTransaction::where('reference_type', 'payment_voucher')
                ->where('reference_id', $payment->id)
                ->delete();

            // حذف سند الدفع
            $payment->delete();
        });

        return redirect()->route('suppliers.payments.index')
            ->with('success', 'تم حذف الدفعة بنجاح');
    }

    /**
     * عرض الفواتير المستحقة لمورد
     */
    public function getOutstandingInvoices(Request $request, Supplier $supplier)
    {
        $invoices = $supplier->purchaseInvoices()
            ->where('payment_status', '!=', 'paid')
            ->where('status', '!=', 'cancelled')
            ->where('remaining_amount', '>', 0)
            ->orderBy('due_date')
            ->get()
            ->map(function ($invoice) {
                return [
                    'id' => $invoice->id,
                    'invoice_number' => $invoice->invoice_number,
                    'invoice_date' => $invoice->invoice_date->format('Y-m-d'),
                    'due_date' => $invoice->due_date->format('Y-m-d'),
                    'total_amount' => $invoice->total_amount,
                    'paid_amount' => $invoice->paid_amount,
                    'remaining_amount' => $invoice->remaining_amount,
                    'days_overdue' => $invoice->is_overdue ? $invoice->days_overdue : 0,
                    'is_overdue' => $invoice->is_overdue,
                ];
            });

        return response()->json($invoices);
    }

    /**
     * تقرير مدفوعات المورد
     */
    public function supplierPaymentReport(Request $request, Supplier $supplier): View
    {
        $dateFrom = $request->get('date_from', now()->startOfYear()->format('Y-m-d'));
        $dateTo = $request->get('date_to', now()->format('Y-m-d'));

        $payments = $supplier->paymentVouchers()
            ->whereBetween('payment_date', [$dateFrom, $dateTo])
            ->with(['user'])
            ->orderBy('payment_date', 'desc')
            ->paginate(20);

        $summary = [
            'total_payments' => $payments->total(),
            'total_amount' => $supplier->paymentVouchers()
                ->whereBetween('payment_date', [$dateFrom, $dateTo])
                ->sum('amount'),
            'average_payment' => $payments->count() > 0 ? 
                $supplier->paymentVouchers()
                    ->whereBetween('payment_date', [$dateFrom, $dateTo])
                    ->avg('amount') : 0,
        ];

        return view('suppliers.payments.supplier-report', compact('supplier', 'payments', 'summary', 'dateFrom', 'dateTo'));
    }

    /**
     * تقرير تحليل المدفوعات
     */
    public function paymentAnalysis(Request $request): View
    {
        $dateFrom = $request->get('date_from', now()->startOfYear()->format('Y-m-d'));
        $dateTo = $request->get('date_to', now()->format('Y-m-d'));

        // المدفوعات حسب الطريقة
        $paymentsByMethod = PaymentVoucher::whereNotNull('supplier_id')
            ->whereBetween('payment_date', [$dateFrom, $dateTo])
            ->selectRaw('payment_method, COUNT(*) as count, SUM(amount) as total')
            ->groupBy('payment_method')
            ->get();

        // المدفوعات الشهرية
        $monthlyPayments = PaymentVoucher::whereNotNull('supplier_id')
            ->whereBetween('payment_date', [$dateFrom, $dateTo])
            ->selectRaw('YEAR(payment_date) as year, MONTH(payment_date) as month, SUM(amount) as total')
            ->groupByRaw('YEAR(payment_date), MONTH(payment_date)')
            ->orderByRaw('YEAR(payment_date), MONTH(payment_date)')
            ->get();

        // أكبر الموردين من حيث المدفوعات
        $topSuppliers = PaymentVoucher::with('supplier')
            ->whereNotNull('supplier_id')
            ->whereBetween('payment_date', [$dateFrom, $dateTo])
            ->selectRaw('supplier_id, SUM(amount) as total_paid, COUNT(*) as payment_count')
            ->groupBy('supplier_id')
            ->orderByDesc('total_paid')
            ->take(10)
            ->get();

        return view('suppliers.payments.analysis', compact(
            'paymentsByMethod', 'monthlyPayments', 'topSuppliers', 'dateFrom', 'dateTo'
        ));
    }

    /**
     * توزيع الدفعة يدوياً على الفواتير
     */
    private function allocatePaymentManually(PaymentVoucher $payment, array $invoices): void
    {
        foreach ($invoices as $invoiceData) {
            $invoice = PurchaseInvoice::find($invoiceData['invoice_id']);
            if ($invoice && $invoiceData['amount'] > 0) {
                $invoice->addPayment($invoiceData['amount']);
                
                // حفظ تفاصيل التوزيع (يمكن إنشاء جدول منفصل لهذا)
                // PaymentAllocation::create([...]);
            }
        }
    }

    /**
     * توزيع الدفعة تلقائياً على الفواتير (الأقدم أولاً)
     */
    private function allocatePaymentAutomatically(PaymentVoucher $payment, Supplier $supplier): void
    {
        $remainingAmount = $payment->amount;
        
        $invoices = $supplier->purchaseInvoices()
            ->where('payment_status', '!=', 'paid')
            ->where('status', '!=', 'cancelled')
            ->where('remaining_amount', '>', 0)
            ->orderBy('due_date')
            ->get();

        foreach ($invoices as $invoice) {
            if ($remainingAmount <= 0) break;

            $paymentAmount = min($remainingAmount, $invoice->remaining_amount);
            $invoice->addPayment($paymentAmount);
            $remainingAmount -= $paymentAmount;
        }
    }

    /**
     * إنشاء قيد محاسبي للدفعة
     */
    private function createJournalEntry(PaymentVoucher $payment): void
    {
        $journalEntry = JournalEntry::create([
            'entry_number' => JournalEntry::generateNextEntryNumber(),
            'entry_date' => $payment->payment_date,
            'description' => "دفعة للمورد {$payment->supplier->name} - {$payment->description}",
            'status' => 'posted',
        ]);

        // مدين: حسابات دائنة (تقليل المديونية)
        $journalEntry->journalEntryLines()->create([
            'account_id' => 7, // حسابات دائنة
            'description' => "دفعة للمورد {$payment->supplier->name}",
            'debit_amount' => $payment->amount,
            'credit_amount' => 0,
            'supplier_id' => $payment->supplier_id,
        ]);

        // دائن: نقدية أو بنك
        $creditAccountId = $payment->payment_method === 'cash' ? 1 : 2; // نقدية أو بنك
        $journalEntry->journalEntryLines()->create([
            'account_id' => $creditAccountId,
            'description' => "دفعة بـ" . $this->getPaymentMethodLabel($payment->payment_method),
            'debit_amount' => 0,
            'credit_amount' => $payment->amount,
        ]);

        $payment->update(['journal_entry_id' => $journalEntry->id]);
    }

    /**
     * إنشاء حركة نقدية أو بنكية
     */
    private function createCashOrBankTransaction(PaymentVoucher $payment): void
    {
        if ($payment->payment_method === 'cash') {
            CashTransaction::create([
                'transaction_date' => $payment->payment_date,
                'type' => 'outgoing',
                'amount' => $payment->amount,
                'description' => "دفعة للمورد {$payment->supplier->name}",
                'reference_type' => 'payment_voucher',
                'reference_id' => $payment->id,
                'account_id' => 1, // حساب النقدية
                'supplier_id' => $payment->supplier_id,
            ]);
        } elseif ($payment->payment_method === 'bank_transfer' && $payment->bank_account_id) {
            // يمكن إنشاء حركة بنكية هنا
        }
    }

    /**
     * الحصول على تسمية طريقة الدفع
     */
    private function getPaymentMethodLabel(string $method): string
    {
        $labels = [
            'cash' => 'نقداً',
            'bank_transfer' => 'حوالة بنكية',
            'check' => 'شيك',
            'credit_card' => 'بطاقة ائتمان',
        ];

        return $labels[$method] ?? $method;
    }

    /**
     * الحصول على مبلغ المدفوعات المعلقة
     */
    private function getPendingPaymentsAmount(): float
    {
        return PurchaseInvoice::where('payment_status', '!=', 'paid')
            ->where('status', '!=', 'cancelled')
            ->sum('remaining_amount');
    }

    /**
     * الحصول على تفاصيل توزيع الدفعة على الفواتير
     */
    private function getPaymentAllocations(PaymentVoucher $payment): array
    {
        // هذه دالة مؤقتة - يمكن تحسينها بإنشاء جدول منفصل لتوزيع المدفوعات
        return [];
    }

    /**
     * عكس تأثير توزيع الدفعة
     */
    private function reversePaymentAllocations(PaymentVoucher $payment): void
    {
        // منطق عكس تأثير الدفعة على الفواتير
        // يحتاج لتطوير حسب التصميم المطلوب
    }
}