<?php

namespace App\Http\Controllers;

use App\Models\Supplier;
use App\Models\PurchaseInvoice;
use App\Models\SupplierDebtSchedule;
use App\Models\SupplierPaymentAllocation;
use App\Models\PaymentVoucher;
use Illuminate\Http\Request;
use Illuminate\Http\RedirectResponse;
use Illuminate\View\View;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;

class SupplierDebtController extends Controller
{
    /**
     * عرض لوحة تحكم إدارة الديون
     */
    public function dashboard(): View
    {
        $stats = [
            // إجماليات الديون
            'total_debt' => Supplier::sum('current_balance'),
            'overdue_debt' => PurchaseInvoice::overdue()->sum('remaining_amount'),
            'suppliers_with_debt' => Supplier::where('current_balance', '>', 0)->count(),
            
            // المجدولات
            'overdue_schedules' => SupplierDebtSchedule::overdue()->count(),
            'due_soon_schedules' => SupplierDebtSchedule::dueSoon(7)->count(),
            'total_penalty_amount' => SupplierDebtSchedule::sum('penalty_amount'),
            
            // إحصائيات شهرية
            'monthly_payments' => PaymentVoucher::whereNotNull('supplier_id')
                ->whereMonth('payment_date', now()->month)
                ->sum('amount'),
            'monthly_new_debt' => PurchaseInvoice::whereMonth('invoice_date', now()->month)
                ->sum('remaining_amount'),
        ];

        // الموردين الأكثر مديونية
        $topDebtors = Supplier::where('current_balance', '>', 0)
            ->orderByDesc('current_balance')
            ->take(10)
            ->get();

        // الفواتير المتأخرة
        $overdueInvoices = PurchaseInvoice::with(['supplier'])
            ->overdue()
            ->orderBy('due_date')
            ->take(20)
            ->get();

        // المجدولات المستحقة قريباً
        $upcomingSchedules = SupplierDebtSchedule::with(['supplier', 'purchaseInvoice'])
            ->dueSoon(7)
            ->orderBy('schedule_date')
            ->take(15)
            ->get();

        // رسم بياني لتطور الديون (آخر 12 شهر)
        $debtEvolution = $this->getDebtEvolutionData();

        // رسم بياني لتوزيع الديون حسب المدة
        $ageDistribution = $this->getDebtAgeDistribution();

        return view('suppliers.debt.dashboard', compact(
            'stats', 'topDebtors', 'overdueInvoices', 'upcomingSchedules',
            'debtEvolution', 'ageDistribution'
        ));
    }

    /**
     * عرض كشف مديونية مورد
     */
    public function supplierStatement(Request $request, Supplier $supplier): View
    {
        $dateFrom = $request->get('date_from', now()->startOfYear()->format('Y-m-d'));
        $dateTo = $request->get('date_to', now()->format('Y-m-d'));

        // الفواتير غير المدفوعة
        $outstandingInvoices = $supplier->purchaseInvoices()
            ->where('payment_status', '!=', 'paid')
            ->where('status', '!=', 'cancelled')
            ->whereBetween('invoice_date', [$dateFrom, $dateTo])
            ->orderBy('due_date')
            ->get();

        // المدفوعات في الفترة
        $payments = PaymentVoucher::where('supplier_id', $supplier->id)
            ->whereBetween('payment_date', [$dateFrom, $dateTo])
            ->orderBy('payment_date', 'desc')
            ->get();

        // الجدولة الحالية
        $schedules = SupplierDebtSchedule::where('supplier_id', $supplier->id)
            ->where('status', '!=', 'paid')
            ->orderBy('schedule_date')
            ->get();

        // تجميع الديون حسب العمر
        $ageGroups = [
            'current' => $outstandingInvoices->where('is_overdue', false)->sum('remaining_amount'),
            '1_30_days' => $outstandingInvoices->where('days_overdue', '>=', 1)->where('days_overdue', '<=', 30)->sum('remaining_amount'),
            '31_60_days' => $outstandingInvoices->where('days_overdue', '>=', 31)->where('days_overdue', '<=', 60)->sum('remaining_amount'),
            '61_90_days' => $outstandingInvoices->where('days_overdue', '>=', 61)->where('days_overdue', '<=', 90)->sum('remaining_amount'),
            'over_90_days' => $outstandingInvoices->where('days_overdue', '>', 90)->sum('remaining_amount'),
        ];

        return view('suppliers.debt.statement', compact(
            'supplier', 'outstandingInvoices', 'payments', 'schedules', 
            'ageGroups', 'dateFrom', 'dateTo'
        ));
    }

    /**
     * إنشاء جدولة سداد
     */
    public function createSchedule(Request $request): View
    {
        $supplierId = $request->get('supplier_id');
        $supplier = null;
        $invoices = collect();

        if ($supplierId) {
            $supplier = Supplier::findOrFail($supplierId);
            $invoices = $supplier->purchaseInvoices()
                ->where('payment_status', '!=', 'paid')
                ->where('status', '!=', 'cancelled')
                ->where('remaining_amount', '>', 0)
                ->orderBy('due_date')
                ->get();
        }

        $suppliers = Supplier::where('current_balance', '>', 0)->get();

        return view('suppliers.debt.create-schedule', compact('supplier', 'suppliers', 'invoices'));
    }

    /**
     * حفظ جدولة سداد جديدة
     */
    public function storeSchedule(Request $request): RedirectResponse
    {
        $validated = $request->validate([
            'supplier_id' => 'required|exists:suppliers,id',
            'purchase_invoice_id' => 'required|exists:purchase_invoices,id',
            'schedule_date' => 'required|date|after:today',
            'scheduled_amount' => 'required|numeric|min:0.01',
            'payment_deadline' => 'nullable|date|after:schedule_date',
            'notes' => 'nullable|string',
        ]);

        DB::transaction(function () use ($validated) {
            SupplierDebtSchedule::create($validated);
        });

        return redirect()->route('suppliers.debt.dashboard')
            ->with('success', 'تم إنشاء جدولة السداد بنجاح');
    }

    /**
     * إرسال تذكيرات الدفع
     */
    public function sendReminders(Request $request): RedirectResponse
    {
        $scheduleIds = $request->get('schedule_ids', []);
        $count = 0;

        if (empty($scheduleIds)) {
            // إرسال تذكيرات لكل المجدولات المستحقة قريباً
            $schedules = SupplierDebtSchedule::dueSoon(3)->get();
        } else {
            $schedules = SupplierDebtSchedule::whereIn('id', $scheduleIds)->get();
        }

        foreach ($schedules as $schedule) {
            $schedule->sendReminder();
            $count++;
        }

        return redirect()->back()
            ->with('success', "تم إرسال {$count} تذكير بنجاح");
    }

    /**
     * تطبيق غرامات التأخير
     */
    public function applyPenalties(Request $request): RedirectResponse
    {
        $overdueSchedules = SupplierDebtSchedule::overdue()
            ->where('status', '!=', 'overdue')
            ->get();

        $count = 0;
        $totalPenalty = 0;

        foreach ($overdueSchedules as $schedule) {
            $schedule->markAsOverdue();
            $totalPenalty += $schedule->penalty_amount;
            $count++;
        }

        return redirect()->back()
            ->with('success', "تم تطبيق {$count} غرامة بإجمالي " . number_format($totalPenalty, 2) . ' ر.س');
    }

    /**
     * تقرير الديون المتأخرة
     */
    public function overdueReport(Request $request): View
    {
        $query = PurchaseInvoice::with(['supplier'])
            ->overdue();

        // فلترة حسب المورد
        if ($request->filled('supplier_id')) {
            $query->where('supplier_id', $request->supplier_id);
        }

        // فلترة حسب المدة
        if ($request->filled('overdue_days')) {
            $days = (int) $request->overdue_days;
            $query->where('due_date', '<=', now()->subDays($days));
        }

        $overdueInvoices = $query->orderBy('due_date')->paginate(20);

        $suppliers = Supplier::has('purchaseInvoices')->get();
        
        $summary = [
            'total_overdue' => PurchaseInvoice::overdue()->count(),
            'total_amount' => PurchaseInvoice::overdue()->sum('remaining_amount'),
            'avg_days_overdue' => PurchaseInvoice::overdue()->avg(DB::raw('DATEDIFF(NOW(), due_date)')),
            'affected_suppliers' => PurchaseInvoice::overdue()->distinct('supplier_id')->count(),
        ];

        return view('suppliers.debt.overdue-report', compact(
            'overdueInvoices', 'suppliers', 'summary'
        ));
    }

    /**
     * تقرير جدولة المدفوعات
     */
    public function scheduleReport(Request $request): View
    {
        $query = SupplierDebtSchedule::with(['supplier', 'purchaseInvoice']);

        // فلترة حسب الحالة
        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        // فلترة حسب التاريخ
        if ($request->filled('date_from')) {
            $query->where('schedule_date', '>=', $request->date_from);
        }
        if ($request->filled('date_to')) {
            $query->where('schedule_date', '<=', $request->date_to);
        }

        $schedules = $query->orderBy('schedule_date')->paginate(20);

        $summary = [
            'total_scheduled' => SupplierDebtSchedule::count(),
            'total_amount' => SupplierDebtSchedule::sum('scheduled_amount'),
            'overdue_count' => SupplierDebtSchedule::overdue()->count(),
            'overdue_amount' => SupplierDebtSchedule::overdue()->sum('scheduled_amount'),
            'total_penalties' => SupplierDebtSchedule::sum('penalty_amount'),
        ];

        return view('suppliers.debt.schedule-report', compact(
            'schedules', 'summary'
        ));
    }

    /**
     * تصفية الديون المشكوك فيها
     */
    public function badDebtManagement(): View
    {
        $suppliers = Supplier::where('current_balance', '>', 0)
            ->get()
            ->map(function ($supplier) {
                $overdueAmount = $supplier->purchaseInvoices()
                    ->overdue()
                    ->sum('remaining_amount');
                    
                $avgDaysOverdue = $supplier->purchaseInvoices()
                    ->overdue()
                    ->avg(DB::raw('DATEDIFF(NOW(), due_date)')) ?? 0;

                $riskScore = $this->calculateRiskScore($supplier, $overdueAmount, $avgDaysOverdue);

                return [
                    'supplier' => $supplier,
                    'total_debt' => $supplier->current_balance,
                    'overdue_amount' => $overdueAmount,
                    'avg_days_overdue' => round($avgDaysOverdue),
                    'risk_score' => $riskScore,
                    'risk_level' => $this->getRiskLevel($riskScore),
                ];
            })
            ->sortByDesc('risk_score');

        return view('suppliers.debt.bad-debt-management', compact('suppliers'));
    }

    /**
     * حساب معامل المخاطر للمورد
     */
    private function calculateRiskScore(Supplier $supplier, float $overdueAmount, float $avgDaysOverdue): int
    {
        $score = 0;

        // المبلغ المتأخر (40% من النتيجة)
        $debtRatio = $supplier->current_balance > 0 ? ($overdueAmount / $supplier->current_balance) : 0;
        $score += $debtRatio * 40;

        // متوسط أيام التأخير (30% من النتيجة)
        $score += min($avgDaysOverdue / 30, 1) * 30;

        // تجاوز حد الائتمان (20% من النتيجة)
        if ($supplier->exceedsCreditLimit()) {
            $score += 20;
        }

        // عدد الفواتير المتأخرة (10% من النتيجة)
        $overdueCount = $supplier->purchaseInvoices()->overdue()->count();
        $score += min($overdueCount / 5, 1) * 10;

        return min(100, round($score));
    }

    /**
     * تحديد مستوى المخاطر
     */
    private function getRiskLevel(int $score): string
    {
        if ($score >= 80) return 'critical';
        if ($score >= 60) return 'high';
        if ($score >= 40) return 'medium';
        return 'low';
    }

    /**
     * بيانات تطور الديون لآخر 12 شهر
     */
    private function getDebtEvolutionData(): array
    {
        $months = [];
        $data = [];

        for ($i = 11; $i >= 0; $i--) {
            $date = now()->subMonths($i);
            $months[] = $date->format('M Y');

            $debt = PurchaseInvoice::where('payment_status', '!=', 'paid')
                ->where('status', '!=', 'cancelled')
                ->whereDate('invoice_date', '<=', $date->endOfMonth())
                ->sum('remaining_amount');

            $data[] = round($debt, 2);
        }

        return compact('months', 'data');
    }

    /**
     * بيانات توزيع أعمار الديون
     */
    private function getDebtAgeDistribution(): array
    {
        $invoices = PurchaseInvoice::where('payment_status', '!=', 'paid')
            ->where('status', '!=', 'cancelled')
            ->get();

        $distribution = [
            'current' => 0,
            '1_30_days' => 0,
            '31_60_days' => 0,
            '61_90_days' => 0,
            'over_90_days' => 0,
        ];

        foreach ($invoices as $invoice) {
            $amount = $invoice->remaining_amount;
            
            if (!$invoice->is_overdue) {
                $distribution['current'] += $amount;
            } elseif ($invoice->days_overdue <= 30) {
                $distribution['1_30_days'] += $amount;
            } elseif ($invoice->days_overdue <= 60) {
                $distribution['31_60_days'] += $amount;
            } elseif ($invoice->days_overdue <= 90) {
                $distribution['61_90_days'] += $amount;
            } else {
                $distribution['over_90_days'] += $amount;
            }
        }

        return $distribution;
    }
}