<?php

namespace App\Http\Controllers;

use App\Models\Supplier;
use Illuminate\Http\Request;
use Illuminate\Http\RedirectResponse;
use Illuminate\View\View;

class SupplierController extends Controller
{
    /**
     * عرض قائمة الموردين
     */
    public function index(Request $request): View
    {
        $query = Supplier::with('purchaseOrders');

        // Search filter
        if ($request->filled('search')) {
            $search = $request->get('search');
            $query->where(function($q) use ($search) {
                $q->where('name', 'like', "%{$search}%")
                  ->orWhere('email', 'like', "%{$search}%");
            });
        }

        // Status filter
        if ($request->filled('is_active')) {
            $query->where('is_active', $request->get('is_active'));
        }

        // Payment terms filter
        if ($request->filled('payment_terms')) {
            $query->where('payment_terms', $request->get('payment_terms'));
        }

        $suppliers = $query->latest()->paginate(10)->withQueryString();
        
        // Calculate statistics
        $stats = [
            'total_suppliers' => Supplier::count(),
            'active_suppliers' => Supplier::where('is_active', true)->count(),
            'high_risk_suppliers' => Supplier::where('current_balance', '>', \DB::raw('credit_limit * 0.9'))->count(),
            'total_outstanding' => Supplier::sum('current_balance'),
        ];
        
        return view('suppliers.index', compact('suppliers', 'stats'));
    }

    /**
     * عرض نموذج إنشاء مورد جديد
     */
    public function create(): View
    {
        return view('suppliers.create');
    }

    /**
     * حفظ مورد جديد
     */
    public function store(Request $request): RedirectResponse
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'name_en' => 'nullable|string|max:255',
            'contact_person' => 'nullable|string|max:255',
            'email' => 'nullable|email|max:255',
            'phone' => 'nullable|string|max:20',
            'address' => 'nullable|string',
            'tax_number' => 'nullable|string|max:50',
            'commercial_registration' => 'nullable|string|max:50',
            'payment_terms' => 'nullable|string|max:255',
            'credit_limit' => 'nullable|numeric|min:0',
            'is_active' => 'boolean',
        ]);

        $supplier = Supplier::create($validated);

        return redirect()->route('suppliers.index')
            ->with('success', 'تم إنشاء المورد بنجاح');
    }

    /**
     * عرض تفاصيل المورد
     */
    public function show(Supplier $supplier): View
    {
        $supplier->load(['purchaseOrders' => function($query) {
            $query->latest()->take(10);
        }]);

        return view('suppliers.show', compact('supplier'));
    }

    /**
     * عرض نموذج تعديل المورد
     */
    public function edit(Supplier $supplier): View
    {
        return view('suppliers.edit', compact('supplier'));
    }

    /**
     * تحديث بيانات المورد
     */
    public function update(Request $request, Supplier $supplier): RedirectResponse
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'name_en' => 'nullable|string|max:255',
            'contact_person' => 'nullable|string|max:255',
            'email' => 'nullable|email|max:255',
            'phone' => 'nullable|string|max:20',
            'address' => 'nullable|string',
            'tax_number' => 'nullable|string|max:50',
            'commercial_registration' => 'nullable|string|max:50',
            'payment_terms' => 'nullable|string|max:255',
            'credit_limit' => 'nullable|numeric|min:0',
            'is_active' => 'boolean',
        ]);

        $supplier->update($validated);

        return redirect()->route('suppliers.index')
            ->with('success', 'تم تحديث المورد بنجاح');
    }

    /**
     * حذف المورد
     */
    public function destroy(Supplier $supplier): RedirectResponse
    {
        if ($supplier->purchaseOrders()->exists()) {
            return redirect()->route('suppliers.index')
                ->with('error', 'لا يمكن حذف المورد لأنه مرتبط بأوامر شراء');
        }

        $supplier->delete();

        return redirect()->route('suppliers.index')
            ->with('success', 'تم حذف المورد بنجاح');
    }

    /**
     * تفعيل/إلغاء تفعيل المورد
     */
    public function toggleStatus(Supplier $supplier): RedirectResponse
    {
        $supplier->update(['is_active' => !$supplier->is_active]);

        $message = $supplier->is_active ? 'تم تفعيل المورد' : 'تم إلغاء تفعيل المورد';

        return redirect()->back()->with('success', $message);
    }

    /**
     * تصدير قائمة الموردين
     */
    public function export(Request $request)
    {
        $suppliers = Supplier::when($request->search, function($query, $search) {
                $query->where(function($q) use ($search) {
                    $q->where('name', 'like', "%{$search}%")
                      ->orWhere('code', 'like', "%{$search}%")
                      ->orWhere('email', 'like', "%{$search}%");
                });
            })
            ->when($request->is_active, function($query, $status) {
                $query->where('is_active', $status);
            })
            ->get();

        $filename = 'suppliers_' . date('Y-m-d_H-i-s') . '.csv';
        
        return response()->streamDownload(function() use ($suppliers) {
            echo "الكود,الاسم,البريد الإلكتروني,الهاتف,حد الائتمان,الرصيد الحالي,الحالة\n";
            
            foreach ($suppliers as $supplier) {
                echo implode(',', [
                    $supplier->code ?? '',
                    '"' . $supplier->name . '"',
                    $supplier->email ?? '',
                    $supplier->phone ?? '',
                    $supplier->credit_limit ?? 0,
                    $supplier->current_balance ?? 0,
                    $supplier->is_active ? 'نشط' : 'غير نشط'
                ]) . "\n";
            }
        }, $filename, [
            'Content-Type' => 'text/csv',
            'Content-Disposition' => 'attachment; filename="' . $filename . '"',
        ]);
    }
}