<?php

namespace App\Http\Controllers;

use App\Models\Supplier;
use App\Models\PurchaseInvoice;
use App\Models\PurchaseOrder;
use App\Models\PaymentVoucher;
use Illuminate\Http\Request;
use Illuminate\View\View;
use Illuminate\Http\Response;
use Carbon\Carbon;
use DB;

class SupplierAnalyticsController extends Controller
{
    /**
     * لوحة تحكم تحليلات الموردين
     */
    public function dashboard(): View
    {
        $totalSuppliers = Supplier::count();
        $activeSuppliers = Supplier::active()->count();
        $suppliersWithDebt = Supplier::withBalance()->count();
        
        // إجمالي المديونية
        $totalDebt = Supplier::sum('current_balance');
        
        // إجمالي المشتريات هذا الشهر
        $monthlyPurchases = PurchaseInvoice::whereMonth('invoice_date', now()->month)
            ->whereYear('invoice_date', now()->year)
            ->where('status', '!=', 'cancelled')
            ->sum('total_amount');
            
        // إجمالي المشتريات هذا العام
        $yearlyPurchases = PurchaseInvoice::whereYear('invoice_date', now()->year)
            ->where('status', '!=', 'cancelled')
            ->sum('total_amount');
            
        // أكبر الموردين من حيث المشتريات
        $topSuppliersByPurchases = Supplier::with(['purchaseInvoices' => function($query) {
                $query->whereYear('invoice_date', now()->year)
                      ->where('status', '!=', 'cancelled');
            }])
            ->get()
            ->map(function ($supplier) {
                $supplier->total_purchases = $supplier->purchaseInvoices->sum('total_amount');
                return $supplier;
            })
            ->sortByDesc('total_purchases')
            ->take(10);
            
        // الموردين مع أعلى مديونية
        $topSuppliersByDebt = Supplier::where('current_balance', '>', 0)
            ->orderByDesc('current_balance')
            ->take(10)
            ->get();
            
        // الفواتير المتأخرة السداد
        $overdueInvoices = PurchaseInvoice::overdue()->count();
        $overdueAmount = PurchaseInvoice::overdue()->sum('remaining_amount');
        
        // المدفوعات هذا الشهر
        $monthlyPayments = PaymentVoucher::whereMonth('payment_date', now()->month)
            ->whereYear('payment_date', now()->year)
            ->whereNotNull('supplier_id')
            ->sum('amount');
            
        // إحصائيات أوامر الشراء
        $pendingOrders = PurchaseOrder::where('status', 'pending')->count();
        $approvedOrders = PurchaseOrder::where('status', 'approved')->count();
        
        // بيانات الرسم البياني للمشتريات الشهرية
        $monthlyPurchasesChart = $this->getMonthlyPurchasesData();
        
        // بيانات رسم تطور المديونية
        $debtEvolutionChart = $this->getDebtEvolutionData();
        
        return view('suppliers.analytics.dashboard', compact(
            'totalSuppliers', 'activeSuppliers', 'suppliersWithDebt', 'totalDebt',
            'monthlyPurchases', 'yearlyPurchases', 'topSuppliersByPurchases',
            'topSuppliersByDebt', 'overdueInvoices', 'overdueAmount', 'monthlyPayments',
            'pendingOrders', 'approvedOrders', 'monthlyPurchasesChart', 'debtEvolutionChart'
        ));
    }
    
    /**
     * تقرير أداء الموردين
     */
    public function performance(Request $request): View
    {
        $dateFrom = $request->get('date_from', now()->startOfYear()->format('Y-m-d'));
        $dateTo = $request->get('date_to', now()->format('Y-m-d'));
        
        $suppliers = Supplier::with(['purchaseInvoices' => function($query) use ($dateFrom, $dateTo) {
                $query->whereBetween('invoice_date', [$dateFrom, $dateTo])
                      ->where('status', '!=', 'cancelled');
            }, 'purchaseOrders' => function($query) use ($dateFrom, $dateTo) {
                $query->whereBetween('order_date', [$dateFrom, $dateTo]);
            }])
            ->get()
            ->map(function ($supplier) {
                // حساب مؤشرات الأداء
                $totalPurchases = $supplier->purchaseInvoices->sum('total_amount');
                $totalInvoices = $supplier->purchaseInvoices->count();
                $totalOrders = $supplier->purchaseOrders->count();
                $averageInvoiceValue = $totalInvoices > 0 ? $totalPurchases / $totalInvoices : 0;
                
                // حساب معدل التأخير في التسليم
                $onTimeDeliveries = $supplier->purchaseOrders->where('status', 'received')->count();
                $deliveryRate = $totalOrders > 0 ? ($onTimeDeliveries / $totalOrders) * 100 : 0;
                
                // حساب متوسط أيام السداد
                $paidInvoices = $supplier->purchaseInvoices->where('payment_status', 'paid');
                $averagePaymentDays = 0;
                if ($paidInvoices->count() > 0) {
                    $totalDays = $paidInvoices->sum(function($invoice) {
                        return $invoice->invoice_date->diffInDays($invoice->updated_at);
                    });
                    $averagePaymentDays = $totalDays / $paidInvoices->count();
                }
                
                return [
                    'supplier' => $supplier,
                    'total_purchases' => $totalPurchases,
                    'total_invoices' => $totalInvoices,
                    'total_orders' => $totalOrders,
                    'average_invoice_value' => $averageInvoiceValue,
                    'delivery_rate' => $deliveryRate,
                    'average_payment_days' => $averagePaymentDays,
                    'current_balance' => $supplier->current_balance,
                    'credit_utilization' => $supplier->credit_limit > 0 ? ($supplier->current_balance / $supplier->credit_limit) * 100 : 0,
                ];
            })
            ->sortByDesc('total_purchases');
            
        return view('suppliers.analytics.performance', compact('suppliers', 'dateFrom', 'dateTo'));
    }
    
    /**
     * تقرير أعمار الديون
     */
    public function agingReport(Request $request): View
    {
        $asOfDate = $request->get('as_of_date', now()->format('Y-m-d'));
        
        $agingData = PurchaseInvoice::with('supplier')
            ->where('payment_status', '!=', 'paid')
            ->where('status', '!=', 'cancelled')
            ->whereDate('due_date', '<=', $asOfDate)
            ->get()
            ->groupBy('supplier_id')
            ->map(function ($invoices) use ($asOfDate) {
                $supplier = $invoices->first()->supplier;
                $asOfDateCarbon = Carbon::parse($asOfDate);
                
                $aging = [
                    'supplier' => $supplier,
                    'current' => 0,
                    '1_30_days' => 0,
                    '31_60_days' => 0,
                    '61_90_days' => 0,
                    'over_90_days' => 0,
                    'total' => 0,
                ];
                
                foreach ($invoices as $invoice) {
                    $daysOverdue = $asOfDateCarbon->diffInDays($invoice->due_date, false);
                    $amount = $invoice->remaining_amount;
                    
                    if ($daysOverdue <= 0) {
                        $aging['current'] += $amount;
                    } elseif ($daysOverdue <= 30) {
                        $aging['1_30_days'] += $amount;
                    } elseif ($daysOverdue <= 60) {
                        $aging['31_60_days'] += $amount;
                    } elseif ($daysOverdue <= 90) {
                        $aging['61_90_days'] += $amount;
                    } else {
                        $aging['over_90_days'] += $amount;
                    }
                    
                    $aging['total'] += $amount;
                }
                
                return $aging;
            })
            ->sortByDesc('total');
            
        // حساب المجاميع الإجمالية
        $totals = [
            'current' => $agingData->sum('current'),
            '1_30_days' => $agingData->sum('1_30_days'),
            '31_60_days' => $agingData->sum('31_60_days'),
            '61_90_days' => $agingData->sum('61_90_days'),
            'over_90_days' => $agingData->sum('over_90_days'),
            'total' => $agingData->sum('total'),
        ];
        
        return view('suppliers.analytics.aging', compact('agingData', 'totals', 'asOfDate'));
    }
    
    /**
     * تقرير مقارنة الموردين
     */
    public function comparison(Request $request): View
    {
        $supplierIds = $request->get('supplier_ids', []);
        $dateFrom = $request->get('date_from', now()->startOfYear()->format('Y-m-d'));
        $dateTo = $request->get('date_to', now()->format('Y-m-d'));
        
        if (empty($supplierIds)) {
            // اختيار أفضل 5 موردين تلقائياً
            $supplierIds = Supplier::withSum(['purchaseInvoices' => function($query) use ($dateFrom, $dateTo) {
                    $query->whereBetween('invoice_date', [$dateFrom, $dateTo])
                          ->where('status', '!=', 'cancelled');
                }], 'total_amount')
                ->orderByDesc('purchase_invoices_sum_total_amount')
                ->take(5)
                ->pluck('id')
                ->toArray();
        }
        
        $suppliers = Supplier::whereIn('id', $supplierIds)
            ->with(['purchaseInvoices' => function($query) use ($dateFrom, $dateTo) {
                $query->whereBetween('invoice_date', [$dateFrom, $dateTo])
                      ->where('status', '!=', 'cancelled');
            }])
            ->get();
            
        $comparison = $suppliers->map(function ($supplier) {
            $invoices = $supplier->purchaseInvoices;
            $totalAmount = $invoices->sum('total_amount');
            $invoiceCount = $invoices->count();
            $averageAmount = $invoiceCount > 0 ? $totalAmount / $invoiceCount : 0;
            $paidInvoices = $invoices->where('payment_status', 'paid');
            $paymentRate = $invoiceCount > 0 ? ($paidInvoices->count() / $invoiceCount) * 100 : 0;
            
            return [
                'supplier' => $supplier,
                'total_amount' => $totalAmount,
                'invoice_count' => $invoiceCount,
                'average_amount' => $averageAmount,
                'payment_rate' => $paymentRate,
                'current_balance' => $supplier->current_balance,
            ];
        });
        
        $allSuppliers = Supplier::active()->get();
        
        return view('suppliers.analytics.comparison', compact('comparison', 'allSuppliers', 'supplierIds', 'dateFrom', 'dateTo'));
    }
    
    /**
     * تقرير كشف حساب المورد
     */
    public function statement(Request $request, Supplier $supplier): View
    {
        $dateFrom = $request->get('date_from', now()->startOfYear()->format('Y-m-d'));
        $dateTo = $request->get('date_to', now()->format('Y-m-d'));
        
        // جلب الفواتير والمدفوعات
        $invoices = $supplier->purchaseInvoices()
            ->whereBetween('invoice_date', [$dateFrom, $dateTo])
            ->orderBy('invoice_date')
            ->get();
            
        $payments = PaymentVoucher::where('supplier_id', $supplier->id)
            ->whereBetween('payment_date', [$dateFrom, $dateTo])
            ->orderBy('payment_date')
            ->get();
            
        // دمج المعاملات وترتيبها حسب التاريخ
        $transactions = collect();
        
        foreach ($invoices as $invoice) {
            $transactions->push([
                'date' => $invoice->invoice_date,
                'type' => 'invoice',
                'reference' => $invoice->invoice_number,
                'description' => "فاتورة شراء #{$invoice->invoice_number}",
                'debit' => $invoice->total_amount,
                'credit' => 0,
                'balance' => 0,
                'details' => $invoice,
            ]);
        }
        
        foreach ($payments as $payment) {
            $transactions->push([
                'date' => $payment->payment_date,
                'type' => 'payment',
                'reference' => $payment->voucher_number ?? 'P-' . $payment->id,
                'description' => "دفعة - {$payment->description}",
                'debit' => 0,
                'credit' => $payment->amount,
                'balance' => 0,
                'details' => $payment,
            ]);
        }
        
        // ترتيب المعاملات حسب التاريخ وحساب الرصيد المتراكم
        $transactions = $transactions->sortBy('date');
        $runningBalance = 0;
        
        $transactions = $transactions->map(function ($transaction) use (&$runningBalance) {
            $runningBalance += ($transaction['debit'] - $transaction['credit']);
            $transaction['balance'] = $runningBalance;
            return $transaction;
        });
        
        // حساب المجاميع
        $totals = [
            'total_debits' => $transactions->sum('debit'),
            'total_credits' => $transactions->sum('credit'),
            'closing_balance' => $runningBalance,
        ];
        
        return view('suppliers.analytics.statement', compact('supplier', 'transactions', 'totals', 'dateFrom', 'dateTo'));
    }
    
    /**
     * تقرير مراقبة حدود الائتمان
     */
    public function creditMonitoring(): View
    {
        $suppliers = Supplier::where('credit_limit', '>', 0)
            ->where('is_active', true)
            ->get()
            ->map(function ($supplier) {
                $creditUtilization = $supplier->credit_limit > 0 ? ($supplier->current_balance / $supplier->credit_limit) * 100 : 0;
                $availableCredit = max(0, $supplier->credit_limit - $supplier->current_balance);
                
                $riskLevel = 'low';
                if ($creditUtilization >= 90) {
                    $riskLevel = 'critical';
                } elseif ($creditUtilization >= 75) {
                    $riskLevel = 'high';
                } elseif ($creditUtilization >= 50) {
                    $riskLevel = 'medium';
                }
                
                return [
                    'supplier' => $supplier,
                    'credit_utilization' => $creditUtilization,
                    'available_credit' => $availableCredit,
                    'risk_level' => $riskLevel,
                    'exceeds_limit' => $supplier->current_balance > $supplier->credit_limit,
                ];
            })
            ->sortByDesc('credit_utilization');
            
        return view('suppliers.analytics.credit-monitoring', compact('suppliers'));
    }
    
    /**
     * بيانات المشتريات الشهرية للرسم البياني
     */
    private function getMonthlyPurchasesData(): array
    {
        $months = [];
        $data = [];
        
        for ($i = 11; $i >= 0; $i--) {
            $date = now()->subMonths($i);
            $months[] = $date->format('M Y');
            
            $amount = PurchaseInvoice::whereYear('invoice_date', $date->year)
                ->whereMonth('invoice_date', $date->month)
                ->where('status', '!=', 'cancelled')
                ->sum('total_amount');
                
            $data[] = round($amount, 2);
        }
        
        return compact('months', 'data');
    }
    
    /**
     * بيانات تطور المديونية للرسم البياني
     */
    private function getDebtEvolutionData(): array
    {
        $months = [];
        $data = [];
        
        for ($i = 11; $i >= 0; $i--) {
            $date = now()->subMonths($i);
            $months[] = $date->format('M Y');
            
            // حساب المديونية في نهاية الشهر
            $debt = PurchaseInvoice::where('payment_status', '!=', 'paid')
                ->where('status', '!=', 'cancelled')
                ->whereDate('invoice_date', '<=', $date->endOfMonth())
                ->sum('remaining_amount');
                
            $data[] = round($debt, 2);
        }
        
        return compact('months', 'data');
    }
    
    /**
     * تصدير التقارير
     */
    public function export(Request $request)
    {
        $reportType = $request->get('type');
        $format = $request->get('format', 'excel');
        
        switch ($reportType) {
            case 'performance':
                return $this->exportPerformanceReport($request, $format);
            case 'aging':
                return $this->exportAgingReport($request, $format);
            case 'comparison':
                return $this->exportComparisonReport($request, $format);
            case 'statement':
                return $this->exportStatementReport($request, $format);
            case 'credit-monitoring':
                return $this->exportCreditMonitoringReport($request, $format);
            default:
                return redirect()->back()->with('error', 'نوع التقرير غير مدعوم');
        }
    }
    
    private function exportPerformanceReport(Request $request, string $format)
    {
        // منطق تصدير تقرير الأداء
        // يمكن تنفيذه باستخدام Laravel Excel أو مكتبات أخرى
    }
    
    private function exportAgingReport(Request $request, string $format)
    {
        // منطق تصدير تقرير أعمار الديون
    }
    
    private function exportComparisonReport(Request $request, string $format)
    {
        // منطق تصدير تقرير المقارنة
    }
    
    private function exportStatementReport(Request $request, string $format)
    {
        // منطق تصدير كشف الحساب
    }
    
    private function exportCreditMonitoringReport(Request $request, string $format)
    {
        // منطق تصدير تقرير مراقبة الائتمان
    }
}