<?php

namespace App\Http\Controllers;

use App\Models\SavedReport;
use App\Models\FiscalYear;
use App\Models\CostCenter;
use App\Models\Account;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Storage;
use Carbon\Carbon;

class SavedReportController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth');
        $this->middleware('accounting.permissions:view_reports')->only(['index', 'show']);
        $this->middleware('accounting.permissions:create_reports')->only(['create', 'store']);
        $this->middleware('accounting.permissions:edit_reports')->only(['edit', 'update']);
        $this->middleware('accounting.permissions:delete_reports')->only(['destroy']);
        $this->middleware('accounting.permissions:generate_reports')->only(['generate', 'download']);
    }

    /**
     * Display a listing of saved reports
     */
    public function index(Request $request)
    {
        $query = SavedReport::with(['creator', 'updater']);

        // البحث
        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('name', 'like', "%{$search}%")
                  ->orWhere('description', 'like', "%{$search}%")
                  ->orWhere('report_type', 'like', "%{$search}%");
            });
        }

        // فلترة حسب نوع التقرير
        if ($request->filled('report_type')) {
            $query->where('report_type', $request->report_type);
        }

        // فلترة حسب التنسيق
        if ($request->filled('format')) {
            $query->where('format', $request->format);
        }

        // فلترة المجدولة فقط
        if ($request->boolean('scheduled_only')) {
            $query->scheduled();
        }

        // فلترة النشطة فقط
        if ($request->boolean('active_only')) {
            $query->active();
        }

        // فلترة العامة فقط
        if ($request->boolean('public_only')) {
            $query->public();
        }

        // فلترة حسب المنشئ
        if ($request->filled('created_by')) {
            $query->where('created_by', $request->created_by);
        }

        $savedReports = $query->orderBy('created_at', 'desc')->paginate(15);

        // البيانات المساعدة
        $reportTypes = [
            'trial_balance' => 'ميزان المراجعة',
            'balance_sheet' => 'الميزانية العمومية',
            'income_statement' => 'قائمة الدخل',
            'cash_flow' => 'التدفق النقدي',
            'general_ledger' => 'دفتر الأستاذ العام',
            'account_statement' => 'كشف حساب',
            'cost_center_report' => 'تقرير مركز التكلفة',
            'budget_report' => 'تقرير الميزانية',
        ];

        $formats = ['excel', 'pdf', 'csv', 'html'];
        $creators = \App\Models\User::orderBy('name')->get();

        // إحصائيات
        $stats = [
            'total' => SavedReport::count(),
            'scheduled' => SavedReport::scheduled()->count(),
            'active' => SavedReport::active()->count(),
            'due_generation' => SavedReport::dueForGeneration()->count(),
        ];

        return view('saved-reports.index', compact(
            'savedReports', 'reportTypes', 'formats', 'creators', 'stats'
        ));
    }

    /**
     * Show the form for creating a new saved report
     */
    public function create()
    {
        $reportTypes = [
            'trial_balance' => 'ميزان المراجعة',
            'balance_sheet' => 'الميزانية العمومية',
            'income_statement' => 'قائمة الدخل',
            'cash_flow' => 'التدفق النقدي',
            'general_ledger' => 'دفتر الأستاذ العام',
            'account_statement' => 'كشف حساب',
            'cost_center_report' => 'تقرير مركز التكلفة',
            'budget_report' => 'تقرير الميزانية',
        ];

        $formats = ['excel', 'pdf', 'csv', 'html'];
        $scheduleFrequencies = ['daily', 'weekly', 'monthly', 'quarterly', 'annually'];

        $fiscalYears = FiscalYear::orderBy('start_date', 'desc')->get();
        $costCenters = CostCenter::where('is_active', true)->orderBy('name')->get();
        $accounts = Account::where('is_active', true)->orderBy('code')->get();

        return view('saved-reports.create', compact(
            'reportTypes', 'formats', 'scheduleFrequencies', 
            'fiscalYears', 'costCenters', 'accounts'
        ));
    }

    /**
     * Store a newly created saved report
     */
    public function store(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'report_type' => 'required|string',
            'description' => 'nullable|string|max:1000',
            'format' => 'required|in:excel,pdf,csv,html',
            'parameters' => 'required|array',
            'filters' => 'nullable|array',
            'is_scheduled' => 'boolean',
            'schedule_frequency' => 'nullable|required_if:is_scheduled,true|in:daily,weekly,monthly,quarterly,annually',
            'schedule_config' => 'nullable|array',
            'recipients' => 'nullable|array',
            'recipients.*' => 'email',
            'is_public' => 'boolean',
        ]);

        DB::beginTransaction();
        
        try {
            $savedReport = SavedReport::create([
                'name' => $request->name,
                'report_type' => $request->report_type,
                'description' => $request->description,
                'format' => $request->format,
                'parameters' => $request->parameters,
                'filters' => $request->filters ?? [],
                'is_scheduled' => $request->boolean('is_scheduled'),
                'schedule_frequency' => $request->schedule_frequency,
                'schedule_config' => $request->schedule_config ?? [],
                'recipients' => $request->recipients ?? [],
                'is_public' => $request->boolean('is_public'),
                'is_active' => true,
                'created_by' => auth()->id(),
            ]);

            // حساب التوقيت التالي للتوليد إذا كان مجدولاً
            if ($savedReport->is_scheduled) {
                $savedReport->update([
                    'next_generation_at' => $this->calculateNextGeneration($request->schedule_frequency)
                ]);
            }

            DB::commit();

            return redirect()->route('saved-reports.show', $savedReport)
                           ->with('success', 'تم إنشاء التقرير المحفوظ بنجاح');

        } catch (\Exception $e) {
            DB::rollback();
            return back()->withErrors(['error' => 'حدث خطأ: ' . $e->getMessage()])
                        ->withInput();
        }
    }

    /**
     * Display the specified saved report
     */
    public function show(SavedReport $savedReport)
    {
        $savedReport->load(['creator', 'updater']);

        // تحقق من الصلاحيات
        if (!$savedReport->is_public && $savedReport->created_by !== auth()->id() && !auth()->user()->hasPermission('view_all_reports')) {
            abort(403, 'غير مصرح لك بعرض هذا التقرير');
        }

        return view('saved-reports.show', compact('savedReport'));
    }

    /**
     * Show the form for editing the saved report
     */
    public function edit(SavedReport $savedReport)
    {
        // تحقق من الصلاحيات
        if ($savedReport->created_by !== auth()->id() && !auth()->user()->hasPermission('edit_all_reports')) {
            return redirect()->route('saved-reports.show', $savedReport)
                           ->with('error', 'غير مصرح لك بتعديل هذا التقرير');
        }

        $reportTypes = [
            'trial_balance' => 'ميزان المراجعة',
            'balance_sheet' => 'الميزانية العمومية',
            'income_statement' => 'قائمة الدخل',
            'cash_flow' => 'التدفق النقدي',
            'general_ledger' => 'دفتر الأستاذ العام',
            'account_statement' => 'كشف حساب',
            'cost_center_report' => 'تقرير مركز التكلفة',
            'budget_report' => 'تقرير الميزانية',
        ];

        $formats = ['excel', 'pdf', 'csv', 'html'];
        $scheduleFrequencies = ['daily', 'weekly', 'monthly', 'quarterly', 'annually'];

        $fiscalYears = FiscalYear::orderBy('start_date', 'desc')->get();
        $costCenters = CostCenter::where('is_active', true)->orderBy('name')->get();
        $accounts = Account::where('is_active', true)->orderBy('code')->get();

        return view('saved-reports.edit', compact(
            'savedReport', 'reportTypes', 'formats', 'scheduleFrequencies',
            'fiscalYears', 'costCenters', 'accounts'
        ));
    }

    /**
     * Update the specified saved report
     */
    public function update(Request $request, SavedReport $savedReport)
    {
        // تحقق من الصلاحيات
        if ($savedReport->created_by !== auth()->id() && !auth()->user()->hasPermission('edit_all_reports')) {
            return redirect()->route('saved-reports.show', $savedReport)
                           ->with('error', 'غير مصرح لك بتعديل هذا التقرير');
        }

        $request->validate([
            'name' => 'required|string|max:255',
            'report_type' => 'required|string',
            'description' => 'nullable|string|max:1000',
            'format' => 'required|in:excel,pdf,csv,html',
            'parameters' => 'required|array',
            'filters' => 'nullable|array',
            'is_scheduled' => 'boolean',
            'schedule_frequency' => 'nullable|required_if:is_scheduled,true|in:daily,weekly,monthly,quarterly,annually',
            'schedule_config' => 'nullable|array',
            'recipients' => 'nullable|array',
            'recipients.*' => 'email',
            'is_public' => 'boolean',
            'is_active' => 'boolean',
        ]);

        DB::beginTransaction();
        
        try {
            $savedReport->update([
                'name' => $request->name,
                'report_type' => $request->report_type,
                'description' => $request->description,
                'format' => $request->format,
                'parameters' => $request->parameters,
                'filters' => $request->filters ?? [],
                'is_scheduled' => $request->boolean('is_scheduled'),
                'schedule_frequency' => $request->schedule_frequency,
                'schedule_config' => $request->schedule_config ?? [],
                'recipients' => $request->recipients ?? [],
                'is_public' => $request->boolean('is_public'),
                'is_active' => $request->boolean('is_active', true),
                'updated_by' => auth()->id(),
            ]);

            // تحديث التوقيت التالي للتوليد
            if ($savedReport->is_scheduled) {
                $savedReport->update([
                    'next_generation_at' => $this->calculateNextGeneration($request->schedule_frequency)
                ]);
            } else {
                $savedReport->update(['next_generation_at' => null]);
            }

            DB::commit();

            return redirect()->route('saved-reports.show', $savedReport)
                           ->with('success', 'تم تحديث التقرير المحفوظ بنجاح');

        } catch (\Exception $e) {
            DB::rollback();
            return back()->withErrors(['error' => 'حدث خطأ: ' . $e->getMessage()])
                        ->withInput();
        }
    }

    /**
     * Generate the saved report
     */
    public function generate(SavedReport $savedReport)
    {
        // تحقق من الصلاحيات
        if (!$savedReport->is_public && $savedReport->created_by !== auth()->id() && !auth()->user()->hasPermission('generate_all_reports')) {
            return redirect()->route('saved-reports.show', $savedReport)
                           ->with('error', 'غير مصرح لك بتوليد هذا التقرير');
        }

        if (!$savedReport->is_active) {
            return redirect()->route('saved-reports.show', $savedReport)
                           ->with('error', 'التقرير غير نشط');
        }

        try {
            $filePath = $savedReport->generate();

            return redirect()->route('saved-reports.show', $savedReport)
                           ->with('success', 'تم توليد التقرير بنجاح');

        } catch (\Exception $e) {
            return redirect()->route('saved-reports.show', $savedReport)
                           ->with('error', 'حدث خطأ أثناء توليد التقرير: ' . $e->getMessage());
        }
    }

    /**
     * Download the generated report
     */
    public function download(SavedReport $savedReport)
    {
        // تحقق من الصلاحيات
        if (!$savedReport->is_public && $savedReport->created_by !== auth()->id() && !auth()->user()->hasPermission('download_all_reports')) {
            abort(403, 'غير مصرح لك بتحميل هذا التقرير');
        }

        if (!$savedReport->file_path || !Storage::exists($savedReport->file_path)) {
            return redirect()->route('saved-reports.show', $savedReport)
                           ->with('error', 'ملف التقرير غير موجود. يرجى توليد التقرير أولاً');
        }

        try {
            return $savedReport->download();

        } catch (\Exception $e) {
            return redirect()->route('saved-reports.show', $savedReport)
                           ->with('error', 'حدث خطأ أثناء تحميل التقرير: ' . $e->getMessage());
        }
    }

    /**
     * Toggle report active status
     */
    public function toggleActive(SavedReport $savedReport)
    {
        // تحقق من الصلاحيات
        if ($savedReport->created_by !== auth()->id() && !auth()->user()->hasPermission('edit_all_reports')) {
            return redirect()->route('saved-reports.show', $savedReport)
                           ->with('error', 'غير مصرح لك بتعديل هذا التقرير');
        }

        $savedReport->update(['is_active' => !$savedReport->is_active]);

        $status = $savedReport->is_active ? 'تم تفعيل' : 'تم إلغاء تفعيل';
        
        return redirect()->route('saved-reports.show', $savedReport)
                       ->with('success', $status . ' التقرير بنجاح');
    }

    /**
     * Duplicate the saved report
     */
    public function duplicate(SavedReport $savedReport)
    {
        try {
            $newReport = $savedReport->replicate();
            $newReport->name = $savedReport->name . ' (نسخة)';
            $newReport->file_path = null;
            $newReport->file_size = null;
            $newReport->last_generated_at = null;
            $newReport->next_generation_at = $savedReport->is_scheduled 
                ? $this->calculateNextGeneration($savedReport->schedule_frequency)
                : null;
            $newReport->created_by = auth()->id();
            $newReport->updated_by = null;
            $newReport->save();

            return redirect()->route('saved-reports.show', $newReport)
                           ->with('success', 'تم نسخ التقرير بنجاح');

        } catch (\Exception $e) {
            return redirect()->route('saved-reports.show', $savedReport)
                           ->with('error', 'حدث خطأ أثناء نسخ التقرير: ' . $e->getMessage());
        }
    }

    /**
     * Generate all due reports
     */
    public function generateDue()
    {
        try {
            $generated = SavedReport::generateDueReports();

            return redirect()->route('saved-reports.index')
                           ->with('success', "تم توليد {$generated} تقرير مجدول");

        } catch (\Exception $e) {
            return redirect()->route('saved-reports.index')
                           ->with('error', 'حدث خطأ أثناء توليد التقارير: ' . $e->getMessage());
        }
    }

    /**
     * Clean up old report files
     */
    public function cleanup(Request $request)
    {
        $request->validate([
            'days_to_keep' => 'required|integer|min:1|max:365',
            'confirm' => 'required|accepted',
        ]);

        try {
            $cleaned = SavedReport::cleanupOldFiles($request->days_to_keep);

            return redirect()->route('saved-reports.index')
                           ->with('success', "تم تنظيف {$cleaned} ملف قديم");

        } catch (\Exception $e) {
            return redirect()->route('saved-reports.index')
                           ->with('error', 'حدث خطأ أثناء تنظيف الملفات: ' . $e->getMessage());
        }
    }

    /**
     * Show scheduled reports dashboard
     */
    public function scheduled()
    {
        $scheduledReports = SavedReport::scheduled()
            ->active()
            ->with(['creator'])
            ->orderBy('next_generation_at')
            ->get();

        $dueReports = SavedReport::dueForGeneration()
            ->with(['creator'])
            ->orderBy('next_generation_at')
            ->get();

        $stats = [
            'total_scheduled' => $scheduledReports->count(),
            'due_now' => $dueReports->count(),
            'due_today' => SavedReport::dueForGeneration()
                ->whereDate('next_generation_at', today())
                ->count(),
            'due_week' => SavedReport::dueForGeneration()
                ->whereBetween('next_generation_at', [now(), now()->addWeek()])
                ->count(),
        ];

        return view('saved-reports.scheduled', compact('scheduledReports', 'dueReports', 'stats'));
    }

    /**
     * Remove the specified saved report
     */
    public function destroy(SavedReport $savedReport)
    {
        // تحقق من الصلاحيات
        if ($savedReport->created_by !== auth()->id() && !auth()->user()->hasPermission('delete_all_reports')) {
            return redirect()->route('saved-reports.index')
                           ->with('error', 'غير مصرح لك بحذف هذا التقرير');
        }

        try {
            DB::beginTransaction();

            // حذف الملف إذا كان موجوداً
            $savedReport->deleteFile();

            // حذف التقرير
            $savedReport->delete();

            DB::commit();

            return redirect()->route('saved-reports.index')
                           ->with('success', 'تم حذف التقرير المحفوظ بنجاح');

        } catch (\Exception $e) {
            DB::rollback();
            return redirect()->route('saved-reports.index')
                           ->with('error', 'حدث خطأ أثناء حذف التقرير: ' . $e->getMessage());
        }
    }

    /**
     * Calculate next generation time based on frequency
     */
    private function calculateNextGeneration($frequency)
    {
        $now = now();
        
        return match($frequency) {
            'daily' => $now->addDay(),
            'weekly' => $now->addWeek(),
            'monthly' => $now->addMonth(),
            'quarterly' => $now->addMonths(3),
            'annually' => $now->addYear(),
            default => null
        };
    }
}