<?php

namespace App\Http\Controllers;

use App\Models\Invoice;
use App\Models\Customer;
use App\Models\Product;
use Illuminate\Http\Request;
use Carbon\Carbon;
use Illuminate\Support\Facades\DB;

class SalesReportController extends Controller
{
    /**
     * Display sales reports dashboard
     */
    public function index()
    {
        return view('system-reports.sales.index');
    }

    /**
     * Sales summary report
     */
    public function summary(Request $request)
    {
        $startDate = $request->get('start_date', now()->startOfMonth());
        $endDate = $request->get('end_date', now()->endOfMonth());

        $salesData = Invoice::whereBetween('issue_date', [$startDate, $endDate])
            ->selectRaw('
                COUNT(*) as total_invoices,
                SUM(total_amount) as total_sales,
                SUM(tax_amount) as total_tax,
                AVG(total_amount) as average_sale
            ')
            ->first();

        $monthlySales = Invoice::whereBetween('issue_date', [$startDate, $endDate])
            ->selectRaw('YEAR(issue_date) as year, MONTH(issue_date) as month, SUM(total_amount) as total')
            ->groupBy('year', 'month')
            ->orderBy('year')
            ->orderBy('month')
            ->get();

        return view('system-reports.sales.summary', compact('salesData', 'monthlySales', 'startDate', 'endDate'));
    }

    /**
     * Sales by customer report
     */
    public function byCustomer(Request $request)
    {
        $startDate = $request->get('start_date', now()->startOfMonth());
        $endDate = $request->get('end_date', now()->endOfMonth());

        $customerSales = Customer::leftJoin('invoices', 'customers.id', '=', 'invoices.customer_id')
            ->whereBetween('invoices.issue_date', [$startDate, $endDate])
            ->selectRaw('
                customers.id,
                customers.name,
                customers.email,
                COUNT(invoices.id) as total_invoices,
                COALESCE(SUM(invoices.total_amount), 0) as total_sales
            ')
            ->groupBy('customers.id', 'customers.name', 'customers.email')
            ->orderByDesc('total_sales')
            ->paginate(15);

        return view('system-reports.sales.by-customer', compact('customerSales', 'startDate', 'endDate'));
    }

    /**
     * Sales by product report
     */
    public function byProduct(Request $request)
    {
        $startDate = $request->get('start_date', now()->startOfMonth());
        $endDate = $request->get('end_date', now()->endOfMonth());

        $productSales = Product::leftJoin('invoice_items', 'products.id', '=', 'invoice_items.product_id')
            ->leftJoin('invoices', 'invoice_items.invoice_id', '=', 'invoices.id')
            ->whereBetween('invoices.issue_date', [$startDate, $endDate])
            ->selectRaw('
                products.id,
                products.name,
                products.price,
                COALESCE(SUM(invoice_items.quantity), 0) as total_quantity,
                COALESCE(SUM(invoice_items.line_total), 0) as total_sales
            ')
            ->groupBy('products.id', 'products.name', 'products.price')
            ->orderByDesc('total_sales')
            ->paginate(15);

        return view('system-reports.sales.by-product', compact('productSales', 'startDate', 'endDate'));
    }

    /**
     * Daily sales report
     */
    public function daily(Request $request)
    {
        $date = $request->get('date', now()->format('Y-m-d'));

        $dailySales = Invoice::whereDate('issue_date', $date)
            ->selectRaw('
                COUNT(*) as total_invoices,
                SUM(total_amount) as total_sales,
                SUM(tax_amount) as total_tax,
                AVG(total_amount) as average_sale,
                HOUR(created_at) as hour
            ')
            ->groupBy('hour')
            ->orderBy('hour')
            ->get();

        $totalDaily = Invoice::whereDate('issue_date', $date)
            ->selectRaw('
                COUNT(*) as total_invoices,
                SUM(total_amount) as total_sales,
                SUM(tax_amount) as total_tax
            ')
            ->first();

        return view('system-reports.sales.daily', compact('dailySales', 'totalDaily', 'date'));
    }

    /**
     * Export sales report
     */
    public function export(Request $request)
    {
        $type = $request->get('type', 'summary');
        $format = $request->get('format', 'excel');
        
        // Implementation for export functionality
        return redirect()->back()->with('success', 'تم تصدير التقرير بنجاح');
    }
}