<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Invoice;
use App\Models\Product;
use App\Models\InvoiceItem;
use Carbon\Carbon;
use Illuminate\Support\Facades\DB;

class RestaurantReportsController extends Controller
{
    /**
     * عرض تقارير المطعم
     */
    public function index(Request $request)
    {
        $dateFrom = $request->get('date_from', Carbon::today()->format('Y-m-d'));
        $dateTo = $request->get('date_to', Carbon::today()->format('Y-m-d'));

        // إحصائيات يومية
        $dailyStats = $this->getDailyStats($dateFrom, $dateTo);
        
        // أكثر المنتجات مبيعاً
        $topProducts = $this->getTopSellingProducts($dateFrom, $dateTo);
        
        // إحصائيات حسب الفئة
        $categoryStats = $this->getCategoryStats($dateFrom, $dateTo);
        
        // إحصائيات أنواع الطلبات
        $orderTypeStats = $this->getOrderTypeStats($dateFrom, $dateTo);
        
        // إحصائيات الساعات
        $hourlyStats = $this->getHourlyStats($dateFrom, $dateTo);

        return view('restaurant.reports.index', compact(
            'dailyStats',
            'topProducts', 
            'categoryStats',
            'orderTypeStats',
            'hourlyStats',
            'dateFrom',
            'dateTo'
        ));
    }

    /**
     * الإحصائيات اليومية
     */
    private function getDailyStats($dateFrom, $dateTo)
    {
        return Invoice::whereBetween('issue_date', [$dateFrom, $dateTo])
            ->select([
                DB::raw('DATE(issue_date) as date'),
                DB::raw('COUNT(*) as total_orders'),
                DB::raw('SUM(total_amount) as total_sales'),
                DB::raw('AVG(total_amount) as average_order'),
                DB::raw('SUM(paid_amount) as total_paid'),
                DB::raw('SUM(total_amount - paid_amount) as total_remaining')
            ])
            ->groupBy('date')
            ->orderBy('date')
            ->get();
    }

    /**
     * أكثر المنتجات مبيعاً
     */
    private function getTopSellingProducts($dateFrom, $dateTo, $limit = 10)
    {
        return InvoiceItem::join('invoices', 'invoice_items.invoice_id', '=', 'invoices.id')
            ->join('products', 'invoice_items.product_id', '=', 'products.id')
            ->whereBetween('invoices.issue_date', [$dateFrom, $dateTo])
            ->select([
                'products.name',
                'products.category',
                DB::raw('SUM(invoice_items.quantity) as total_quantity'),
                DB::raw('SUM(invoice_items.quantity * invoice_items.unit_price) as total_revenue'),
                DB::raw('COUNT(DISTINCT invoices.id) as order_count')
            ])
            ->groupBy('products.id', 'products.name', 'products.category')
            ->orderBy('total_quantity', 'desc')
            ->limit($limit)
            ->get();
    }

    /**
     * إحصائيات الفئات
     */
    private function getCategoryStats($dateFrom, $dateTo)
    {
        return InvoiceItem::join('invoices', 'invoice_items.invoice_id', '=', 'invoices.id')
            ->join('products', 'invoice_items.product_id', '=', 'products.id')
            ->whereBetween('invoices.issue_date', [$dateFrom, $dateTo])
            ->select([
                'products.category',
                DB::raw('COUNT(*) as item_count'),
                DB::raw('SUM(invoice_items.quantity) as total_quantity'),
                DB::raw('SUM(invoice_items.quantity * invoice_items.unit_price) as total_revenue'),
                DB::raw('AVG(invoice_items.unit_price) as avg_price')
            ])
            ->groupBy('products.category')
            ->orderBy('total_revenue', 'desc')
            ->get()
            ->map(function ($item) {
                $item->category_name = $this->getCategoryName($item->category);
                return $item;
            });
    }

    /**
     * إحصائيات أنواع الطلبات
     */
    private function getOrderTypeStats($dateFrom, $dateTo)
    {
        // نحتاج إضافة حقل order_type للجدول
        // هذا مثال لما يمكن أن يكون
        return collect([
            [
                'type' => 'dine_in',
                'name' => 'تناول في المطعم',
                'count' => Invoice::whereBetween('issue_date', [$dateFrom, $dateTo])->count() * 0.6,
                'revenue' => Invoice::whereBetween('issue_date', [$dateFrom, $dateTo])->sum('total_amount') * 0.6
            ],
            [
                'type' => 'takeaway',
                'name' => 'طلب خارجي',
                'count' => Invoice::whereBetween('issue_date', [$dateFrom, $dateTo])->count() * 0.3,
                'revenue' => Invoice::whereBetween('issue_date', [$dateFrom, $dateTo])->sum('total_amount') * 0.3
            ],
            [
                'type' => 'delivery',
                'name' => 'توصيل',
                'count' => Invoice::whereBetween('issue_date', [$dateFrom, $dateTo])->count() * 0.1,
                'revenue' => Invoice::whereBetween('issue_date', [$dateFrom, $dateTo])->sum('total_amount') * 0.1
            ]
        ]);
    }

    /**
     * إحصائيات الساعات
     */
    private function getHourlyStats($dateFrom, $dateTo)
    {
        return Invoice::whereBetween('issue_date', [$dateFrom, $dateTo])
            ->select([
                DB::raw('HOUR(created_at) as hour'),
                DB::raw('COUNT(*) as order_count'),
                DB::raw('SUM(total_amount) as revenue')
            ])
            ->groupBy('hour')
            ->orderBy('hour')
            ->get()
            ->map(function ($item) {
                $item->hour_display = sprintf('%02d:00', $item->hour);
                return $item;
            });
    }

    /**
     * ترجمة أسماء الفئات
     */
    private function getCategoryName($category)
    {
        $categories = [
            'meals' => 'الوجبات الرئيسية',
            'appetizers' => 'المقبلات',
            'beverages' => 'المشروبات',
            'desserts' => 'الحلويات'
        ];

        return $categories[$category] ?? $category;
    }

    /**
     * تصدير التقرير
     */
    public function export(Request $request)
    {
        $dateFrom = $request->get('date_from', Carbon::today()->format('Y-m-d'));
        $dateTo = $request->get('date_to', Carbon::today()->format('Y-m-d'));
        
        $data = [
            'dailyStats' => $this->getDailyStats($dateFrom, $dateTo),
            'topProducts' => $this->getTopSellingProducts($dateFrom, $dateTo),
            'categoryStats' => $this->getCategoryStats($dateFrom, $dateTo),
        ];

        return response()->json($data);
    }

    /**
     * احصائيات لوحة التحكم
     */
    public function dashboard()
    {
        $today = Carbon::today();
        
        $todayStats = [
            'orders_count' => Invoice::whereDate('issue_date', $today)->count(),
            'revenue' => Invoice::whereDate('issue_date', $today)->sum('total_amount'),
            'avg_order' => Invoice::whereDate('issue_date', $today)->avg('total_amount') ?: 0,
            'items_sold' => InvoiceItem::join('invoices', 'invoice_items.invoice_id', '=', 'invoices.id')
                ->whereDate('invoices.issue_date', $today)
                ->sum('invoice_items.quantity')
        ];

        $weekStats = [
            'orders_count' => Invoice::whereBetween('issue_date', [$today->copy()->subDays(6), $today])->count(),
            'revenue' => Invoice::whereBetween('issue_date', [$today->copy()->subDays(6), $today])->sum('total_amount'),
        ];

        $topProductsToday = $this->getTopSellingProducts($today, $today, 5);

        return response()->json([
            'today' => $todayStats,
            'week' => $weekStats,
            'top_products' => $topProductsToday
        ]);
    }
}