<?php

namespace App\Http\Controllers;

use App\Models\ReceiptVoucher;
use App\Models\Company;
use App\Models\Customer;
use App\Models\Invoice;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class ReceiptVoucherController extends Controller
{
    /**
     * عرض قائمة سندات القبض
     */
    public function index(Request $request)
    {
        $query = ReceiptVoucher::with(['company', 'customer', 'user', 'invoice'])
            ->latest('receipt_date');

        // فلترة حسب الشركة
        if ($request->filled('company_id')) {
            $query->where('company_id', $request->company_id);
        }

        // فلترة حسب العميل
        if ($request->filled('customer_id')) {
            $query->where('customer_id', $request->customer_id);
        }

        // فلترة حسب الحالة
        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        // فلترة حسب طريقة الدفع
        if ($request->filled('payment_method')) {
            $query->where('payment_method', $request->payment_method);
        }

        // فلترة حسب التاريخ
        if ($request->filled('date_from')) {
            $query->whereDate('receipt_date', '>=', $request->date_from);
        }

        if ($request->filled('date_to')) {
            $query->whereDate('receipt_date', '<=', $request->date_to);
        }

        // البحث
        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('voucher_number', 'like', "%{$search}%")
                  ->orWhere('reference_number', 'like', "%{$search}%")
                  ->orWhere('description', 'like', "%{$search}%")
                  ->orWhereHas('customer', function ($customerQuery) use ($search) {
                      $customerQuery->where('name', 'like', "%{$search}%");
                  });
            });
        }

        $vouchers = $query->paginate(15);

        // بيانات إضافية للفلاتر
        $companies = Company::all();
        $customers = Customer::all();

        // إحصائيات
        $stats = [
            'total_amount' => ReceiptVoucher::active()->sum('amount'),
            'total_count' => ReceiptVoucher::active()->count(),
            'pending_count' => ReceiptVoucher::where('status', 'pending')->count(),
            'cancelled_count' => ReceiptVoucher::where('status', 'cancelled')->count(),
        ];

        return view('receipt-vouchers.index', compact('vouchers', 'companies', 'customers', 'stats'));
    }

    /**
     * عرض نموذج إنشاء سند قبض جديد
     */
    public function create()
    {
        $companies = Company::all();
        $customers = Customer::all();
        $invoices = Invoice::unpaid()->with('customer')->get();

        return view('receipt-vouchers.create', compact('companies', 'customers', 'invoices'));
    }

    /**
     * حفظ سند قبض جديد
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'company_id' => 'required|exists:companies,id',
            'customer_id' => 'required|exists:customers,id',
            'receipt_date' => 'required|date',
            'amount' => 'required|numeric|min:0.01',
            'payment_method' => 'required|in:cash,bank_transfer,check,credit_card,other',
            'reference_number' => 'nullable|string|max:255',
            'description' => 'required|string|max:500',
            'notes' => 'nullable|string|max:1000',
            'invoice_id' => 'nullable|exists:invoices,id',
            'account_code' => 'nullable|string|max:50',
            'status' => 'required|in:active,pending'
        ]);

        $validated['user_id'] = Auth::id();

        try {
            DB::beginTransaction();

            $voucher = ReceiptVoucher::create($validated);

            // إذا كان السند مرتبط بفاتورة، قم بتحديث المبلغ المدفوع
            if ($voucher->invoice_id) {
                $invoice = Invoice::find($voucher->invoice_id);
                if ($invoice) {
                    $invoice->addPayment($voucher->amount, $voucher->payment_method, "سند قبض رقم: {$voucher->voucher_number}");
                }
            }

            DB::commit();

            return redirect()->route('receipt-vouchers.show', $voucher)
                ->with('success', 'تم إنشاء سند القبض بنجاح');

        } catch (\Exception $e) {
            DB::rollback();
            return back()->withInput()
                ->with('error', 'حدث خطأ أثناء إنشاء سند القبض: ' . $e->getMessage());
        }
    }

    /**
     * عرض تفاصيل سند القبض
     */
    public function show(ReceiptVoucher $receiptVoucher)
    {
        $receiptVoucher->load(['company', 'customer', 'user', 'invoice']);
        
        return view('receipt-vouchers.show', compact('receiptVoucher'));
    }

    /**
     * عرض نموذج تحرير سند القبض
     */
    public function edit(ReceiptVoucher $receiptVoucher)
    {
        // لا يمكن تحرير السندات الملغية
        if ($receiptVoucher->status === 'cancelled') {
            return redirect()->route('receipt-vouchers.show', $receiptVoucher)
                ->with('error', 'لا يمكن تحرير سند قبض ملغي');
        }

        $companies = Company::all();
        $customers = Customer::all();
        $invoices = Invoice::unpaid()->with('customer')->get();

        return view('receipt-vouchers.edit', compact('receiptVoucher', 'companies', 'customers', 'invoices'));
    }

    /**
     * تحديث سند القبض
     */
    public function update(Request $request, ReceiptVoucher $receiptVoucher)
    {
        // لا يمكن تحديث السندات الملغية
        if ($receiptVoucher->status === 'cancelled') {
            return redirect()->route('receipt-vouchers.show', $receiptVoucher)
                ->with('error', 'لا يمكن تحديث سند قبض ملغي');
        }

        $validated = $request->validate([
            'company_id' => 'required|exists:companies,id',
            'customer_id' => 'required|exists:customers,id',
            'receipt_date' => 'required|date',
            'amount' => 'required|numeric|min:0.01',
            'payment_method' => 'required|in:cash,bank_transfer,check,credit_card,other',
            'reference_number' => 'nullable|string|max:255',
            'description' => 'required|string|max:500',
            'notes' => 'nullable|string|max:1000',
            'invoice_id' => 'nullable|exists:invoices,id',
            'account_code' => 'nullable|string|max:50',
            'status' => 'required|in:active,pending'
        ]);

        try {
            DB::beginTransaction();

            $oldAmount = $receiptVoucher->amount;
            $oldInvoiceId = $receiptVoucher->invoice_id;

            $receiptVoucher->update($validated);

            // إذا تغيرت الفاتورة المرتبطة أو المبلغ، قم بتحديث المدفوعات
            if ($oldInvoiceId !== $receiptVoucher->invoice_id || $oldAmount !== $receiptVoucher->amount) {
                // إلغاء الدفعة من الفاتورة القديمة
                if ($oldInvoiceId) {
                    $oldInvoice = Invoice::find($oldInvoiceId);
                    if ($oldInvoice) {
                        $oldInvoice->paid_amount -= $oldAmount;
                        $oldInvoice->updatePaymentStatus();
                        $oldInvoice->save();
                    }
                }

                // إضافة الدفعة للفاتورة الجديدة
                if ($receiptVoucher->invoice_id) {
                    $newInvoice = Invoice::find($receiptVoucher->invoice_id);
                    if ($newInvoice) {
                        $newInvoice->addPayment($receiptVoucher->amount, $receiptVoucher->payment_method, "سند قبض رقم: {$receiptVoucher->voucher_number}");
                    }
                }
            }

            DB::commit();

            return redirect()->route('receipt-vouchers.show', $receiptVoucher)
                ->with('success', 'تم تحديث سند القبض بنجاح');

        } catch (\Exception $e) {
            DB::rollback();
            return back()->withInput()
                ->with('error', 'حدث خطأ أثناء تحديث سند القبض: ' . $e->getMessage());
        }
    }

    /**
     * إلغاء سند القبض
     */
    public function destroy(ReceiptVoucher $receiptVoucher)
    {
        try {
            DB::beginTransaction();

            // إذا كان مرتبط بفاتورة، قم بإلغاء الدفعة
            if ($receiptVoucher->invoice_id && $receiptVoucher->status === 'active') {
                $invoice = Invoice::find($receiptVoucher->invoice_id);
                if ($invoice) {
                    $invoice->paid_amount -= $receiptVoucher->amount;
                    $invoice->updatePaymentStatus();
                    $invoice->save();
                }
            }

            // تحديث حالة السند إلى ملغي بدلاً من الحذف
            $receiptVoucher->update(['status' => 'cancelled']);

            DB::commit();

            return redirect()->route('receipt-vouchers.index')
                ->with('success', 'تم إلغاء سند القبض بنجاح');

        } catch (\Exception $e) {
            DB::rollback();
            return back()->with('error', 'حدث خطأ أثناء إلغاء سند القبض: ' . $e->getMessage());
        }
    }

    /**
     * طباعة سند القبض
     */
    public function print(ReceiptVoucher $receiptVoucher)
    {
        $receiptVoucher->load(['company', 'customer', 'user', 'invoice']);
        
        return view('receipt-vouchers.print', compact('receiptVoucher'));
    }

    /**
     * تصدير تقرير سندات القبض
     */
    public function export(Request $request)
    {
        $query = ReceiptVoucher::with(['company', 'customer', 'user'])
            ->latest('receipt_date');

        // تطبيق الفلاتر
        if ($request->filled('company_id')) {
            $query->where('company_id', $request->company_id);
        }

        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        if ($request->filled('date_from')) {
            $query->whereDate('receipt_date', '>=', $request->date_from);
        }

        if ($request->filled('date_to')) {
            $query->whereDate('receipt_date', '<=', $request->date_to);
        }

        $vouchers = $query->get();

        $filename = 'receipt_vouchers_' . now()->format('Y-m-d_H-i-s') . '.csv';

        $headers = [
            'Content-Type' => 'text/csv',
            'Content-Disposition' => "attachment; filename={$filename}",
        ];

        $callback = function() use ($vouchers) {
            $file = fopen('php://output', 'w');
            
            // العناوين
            fputcsv($file, [
                'رقم السند',
                'الشركة',
                'العميل',
                'التاريخ',
                'المبلغ',
                'طريقة الدفع',
                'الرقم المرجعي',
                'الوصف',
                'الحالة',
                'المستخدم'
            ]);

            // البيانات
            foreach ($vouchers as $voucher) {
                fputcsv($file, [
                    $voucher->voucher_number,
                    $voucher->company->name,
                    $voucher->customer->name,
                    $voucher->receipt_date->format('Y-m-d'),
                    $voucher->amount,
                    $voucher->getPaymentMethodText(),
                    $voucher->reference_number,
                    $voucher->description,
                    $voucher->getStatusText(),
                    $voucher->user->name
                ]);
            }

            fclose($file);
        };

        return response()->stream($callback, 200, $headers);
    }
}