<?php

namespace App\Http\Controllers;

use App\Models\PurchaseInvoice;
use App\Models\Supplier;
use App\Models\Product;
use App\Models\PurchaseOrder;
use Illuminate\Http\Request;
use Carbon\Carbon;
use Illuminate\Support\Facades\DB;

class PurchaseReportController extends Controller
{
    /**
     * Display purchase reports dashboard
     */
    public function index()
    {
        return view('system-reports.purchase.index');
    }

    /**
     * Purchase summary report
     */
    public function summary(Request $request)
    {
        $startDate = $request->get('start_date', now()->startOfMonth());
        $endDate = $request->get('end_date', now()->endOfMonth());

        $purchaseData = PurchaseInvoice::whereBetween('invoice_date', [$startDate, $endDate])
            ->selectRaw('
                COUNT(*) as total_invoices,
                SUM(total_amount) as total_purchases,
                SUM(tax_amount) as total_tax,
                AVG(total_amount) as average_purchase
            ')
            ->first();

        $monthlyPurchases = PurchaseInvoice::whereBetween('invoice_date', [$startDate, $endDate])
            ->selectRaw('YEAR(invoice_date) as year, MONTH(invoice_date) as month, SUM(total_amount) as total')
            ->groupBy('year', 'month')
            ->orderBy('year')
            ->orderBy('month')
            ->get();

        return view('system-reports.purchase.summary', compact('purchaseData', 'monthlyPurchases', 'startDate', 'endDate'));
    }

    /**
     * Purchases by supplier report
     */
    public function bySupplier(Request $request)
    {
        $startDate = $request->get('start_date', now()->startOfMonth());
        $endDate = $request->get('end_date', now()->endOfMonth());

        $supplierPurchases = Supplier::leftJoin('purchase_invoices', 'suppliers.id', '=', 'purchase_invoices.supplier_id')
            ->whereBetween('purchase_invoices.invoice_date', [$startDate, $endDate])
            ->selectRaw('
                suppliers.id,
                suppliers.name,
                suppliers.email,
                suppliers.phone,
                COUNT(purchase_invoices.id) as total_invoices,
                COALESCE(SUM(purchase_invoices.total_amount), 0) as total_purchases
            ')
            ->groupBy('suppliers.id', 'suppliers.name', 'suppliers.email', 'suppliers.phone')
            ->orderByDesc('total_purchases')
            ->paginate(15);

        return view('system-reports.purchase.by-supplier', compact('supplierPurchases', 'startDate', 'endDate'));
    }

    /**
     * Purchase orders report
     */
    public function orders(Request $request)
    {
        $startDate = $request->get('start_date', now()->startOfMonth());
        $endDate = $request->get('end_date', now()->endOfMonth());
        $status = $request->get('status', 'all');

        $query = PurchaseOrder::with('supplier')
            ->whereBetween('order_date', [$startDate, $endDate]);

        if ($status !== 'all') {
            $query->where('status', $status);
        }

        $purchaseOrders = $query->orderByDesc('order_date')->paginate(15);

        $orderStats = PurchaseOrder::whereBetween('order_date', [$startDate, $endDate])
            ->selectRaw('
                COUNT(*) as total_orders,
                SUM(total_amount) as total_amount,
                COUNT(CASE WHEN status = "pending" THEN 1 END) as pending_orders,
                COUNT(CASE WHEN status = "approved" THEN 1 END) as approved_orders,
                COUNT(CASE WHEN status = "completed" THEN 1 END) as completed_orders,
                COUNT(CASE WHEN status = "cancelled" THEN 1 END) as cancelled_orders
            ')
            ->first();

        return view('system-reports.purchase.orders', compact('purchaseOrders', 'orderStats', 'startDate', 'endDate', 'status'));
    }

    /**
     * Purchase performance report
     */
    public function performance(Request $request)
    {
        $startDate = $request->get('start_date', now()->startOfMonth());
        $endDate = $request->get('end_date', now()->endOfMonth());

        // Average time from order to delivery
        $performanceData = DB::select("
            SELECT 
                suppliers.name as supplier_name,
                COUNT(purchase_orders.id) as total_orders,
                AVG(DATEDIFF(purchase_invoices.invoice_date, purchase_orders.order_date)) as avg_delivery_days,
                COUNT(CASE WHEN purchase_orders.status = 'completed' THEN 1 END) as completed_orders,
                (COUNT(CASE WHEN purchase_orders.status = 'completed' THEN 1 END) / COUNT(purchase_orders.id) * 100) as completion_rate
            FROM suppliers
            LEFT JOIN purchase_orders ON suppliers.id = purchase_orders.supplier_id
            LEFT JOIN purchase_invoices ON purchase_orders.id = purchase_invoices.purchase_order_id
            WHERE purchase_orders.order_date BETWEEN ? AND ?
            GROUP BY suppliers.id, suppliers.name
            ORDER BY completion_rate DESC
        ", [$startDate, $endDate]);

        return view('system-reports.purchase.performance', compact('performanceData', 'startDate', 'endDate'));
    }

    /**
     * Purchase trends report
     */
    public function trends(Request $request)
    {
        $months = $request->get('months', 12);
        $startDate = now()->subMonths($months)->startOfMonth();
        $endDate = now()->endOfMonth();

        $monthlyTrends = PurchaseInvoice::whereBetween('invoice_date', [$startDate, $endDate])
            ->selectRaw('
                YEAR(invoice_date) as year,
                MONTH(invoice_date) as month,
                COUNT(*) as total_invoices,
                SUM(total_amount) as total_amount
            ')
            ->groupBy('year', 'month')
            ->orderBy('year')
            ->orderBy('month')
            ->get();

        $categoryTrends = DB::table('purchase_invoice_details')
            ->join('purchase_invoices', 'purchase_invoice_details.purchase_invoice_id', '=', 'purchase_invoices.id')
            ->join('items', 'purchase_invoice_details.item_id', '=', 'items.id')
            ->whereBetween('purchase_invoices.invoice_date', [$startDate, $endDate])
            ->selectRaw('
                items.category,
                SUM(purchase_invoice_details.quantity) as total_quantity,
                SUM(purchase_invoice_details.total_amount) as total_amount
            ')
            ->groupBy('items.category')
            ->orderByDesc('total_amount')
            ->get();

        return view('system-reports.purchase.trends', compact('monthlyTrends', 'categoryTrends', 'months'));
    }

    /**
     * Export purchase report
     */
    public function export(Request $request)
    {
        $type = $request->get('type', 'summary');
        $format = $request->get('format', 'excel');
        
        // Implementation for export functionality
        return redirect()->back()->with('success', 'تم تصدير التقرير بنجاح');
    }
}