<?php

namespace App\Http\Controllers;

use App\Models\PurchaseOrder;
use App\Models\PurchaseOrderItem;
use App\Models\Supplier;
use App\Models\Product;
use Illuminate\Http\Request;
use Illuminate\Http\RedirectResponse;
use Illuminate\View\View;
use Illuminate\Support\Facades\DB;

class PurchaseOrderController extends Controller
{
    /**
     * عرض قائمة أوامر الشراء
     */
    public function index(Request $request): View
    {
        $query = PurchaseOrder::with(['supplier', 'user'])
            ->latest();

        // فلترة حسب الحالة
        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        // فلترة حسب المورد
        if ($request->filled('supplier_id')) {
            $query->where('supplier_id', $request->supplier_id);
        }

        $purchaseOrders = $query->paginate(10);
        $suppliers = Supplier::active()->get();

        return view('purchase-orders.index', compact('purchaseOrders', 'suppliers'));
    }

    /**
     * عرض نموذج إنشاء أمر شراء جديد
     */
    public function create(): View
    {
        $suppliers = Supplier::active()->get();
        $products = Product::active()->get();
        
        return view('purchase-orders.create', compact('suppliers', 'products'));
    }

    /**
     * حفظ أمر شراء جديد
     */
    public function store(Request $request): RedirectResponse
    {
        $validated = $request->validate([
            'supplier_id' => 'required|exists:suppliers,id',
            'order_date' => 'required|date',
            'expected_delivery_date' => 'nullable|date|after:order_date',
            'notes' => 'nullable|string',
            'items' => 'required|array|min:1',
            'items.*.product_id' => 'required|exists:products,id',
            'items.*.quantity' => 'required|integer|min:1',
            'items.*.unit_price' => 'required|numeric|min:0',
            'items.*.tax_rate' => 'required|numeric|min:0|max:100',
        ]);

        DB::transaction(function () use ($validated) {
            // إنشاء رقم أمر الشراء
            $poNumber = 'PO-' . date('Y') . '-' . str_pad(
                PurchaseOrder::whereYear('created_at', date('Y'))->count() + 1,
                6, '0', STR_PAD_LEFT
            );

            // حساب المجاميع
            $subtotal = 0;
            $taxAmount = 0;
            
            foreach ($validated['items'] as $item) {
                $itemSubtotal = $item['quantity'] * $item['unit_price'];
                $itemTax = $itemSubtotal * ($item['tax_rate'] / 100);
                
                $subtotal += $itemSubtotal;
                $taxAmount += $itemTax;
            }

            // إنشاء أمر الشراء
            $purchaseOrder = PurchaseOrder::create([
                'supplier_id' => $validated['supplier_id'],
                'user_id' => auth()->id(),
                'po_number' => $poNumber,
                'order_date' => $validated['order_date'],
                'expected_delivery_date' => $validated['expected_delivery_date'],
                'subtotal' => $subtotal,
                'tax_amount' => $taxAmount,
                'total_amount' => $subtotal + $taxAmount,
                'notes' => $validated['notes'],
                'status' => 'draft',
            ]);

            // إنشاء عناصر أمر الشراء
            foreach ($validated['items'] as $item) {
                $itemSubtotal = $item['quantity'] * $item['unit_price'];
                $itemTax = $itemSubtotal * ($item['tax_rate'] / 100);

                PurchaseOrderItem::create([
                    'purchase_order_id' => $purchaseOrder->id,
                    'product_id' => $item['product_id'],
                    'quantity' => $item['quantity'],
                    'unit_price' => $item['unit_price'],
                    'tax_rate' => $item['tax_rate'],
                    'total_amount' => $itemSubtotal + $itemTax,
                ]);
            }
        });

        return redirect()->route('purchase-orders.index')
            ->with('success', 'تم إنشاء أمر الشراء بنجاح');
    }

    /**
     * عرض تفاصيل أمر الشراء
     */
    public function show(PurchaseOrder $purchaseOrder): View
    {
        $purchaseOrder->load(['supplier', 'user', 'items.product', 'receipts']);

        return view('purchase-orders.show', compact('purchaseOrder'));
    }

    /**
     * تغيير حالة أمر الشراء
     */
    public function updateStatus(Request $request, PurchaseOrder $purchaseOrder): RedirectResponse
    {
        $validated = $request->validate([
            'status' => 'required|in:pending,approved,sent,cancelled',
        ]);

        $purchaseOrder->update(['status' => $validated['status']]);

        return redirect()->back()
            ->with('success', 'تم تحديث حالة أمر الشراء بنجاح');
    }

    /**
     * حذف أمر الشراء
     */
    public function destroy(PurchaseOrder $purchaseOrder): RedirectResponse
    {
        if (!in_array($purchaseOrder->status, ['draft', 'cancelled'])) {
            return redirect()->route('purchase-orders.index')
                ->with('error', 'لا يمكن حذف أمر الشراء في هذه الحالة');
        }

        $purchaseOrder->delete();

        return redirect()->route('purchase-orders.index')
            ->with('success', 'تم حذف أمر الشراء بنجاح');
    }
}