<?php

namespace App\Http\Controllers;

use App\Models\PurchaseInvoice;
use App\Models\PurchaseInvoiceDetail;
use App\Models\Supplier;
use App\Models\Item;
use App\Models\CostCenter;
use Illuminate\Http\Request;
use Illuminate\Http\RedirectResponse;
use Illuminate\View\View;
use Illuminate\Support\Facades\DB;

class PurchaseInvoiceController extends Controller
{
    /**
     * عرض قائمة فواتير الشراء
     */
    public function index(Request $request): View
    {
        $query = PurchaseInvoice::with(['supplier', 'costCenter'])
            ->latest('invoice_date');

        // فلترة حسب الحالة
        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        // فلترة حسب حالة الدفع
        if ($request->filled('payment_status')) {
            $query->where('payment_status', $request->payment_status);
        }

        // فلترة حسب المورد
        if ($request->filled('supplier_id')) {
            $query->where('supplier_id', $request->supplier_id);
        }

        // فلترة حسب التاريخ
        if ($request->filled('date_from')) {
            $query->whereDate('invoice_date', '>=', $request->date_from);
        }
        
        if ($request->filled('date_to')) {
            $query->whereDate('invoice_date', '<=', $request->date_to);
        }

        // البحث النصي
        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function($q) use ($search) {
                $q->where('invoice_number', 'like', "%{$search}%")
                  ->orWhere('supplier_invoice_number', 'like', "%{$search}%")
                  ->orWhereHas('supplier', function($sq) use ($search) {
                      $sq->where('name', 'like', "%{$search}%");
                  });
            });
        }

        $purchaseInvoices = $query->paginate(15)->withQueryString();
        $suppliers = Supplier::active()->get();

        // إحصائيات سريعة
        $stats = [
            'total_invoices' => PurchaseInvoice::count(),
            'pending_invoices' => PurchaseInvoice::where('status', 'pending')->count(),
            'overdue_invoices' => PurchaseInvoice::overdue()->count(),
            'total_amount' => PurchaseInvoice::where('status', '!=', 'cancelled')->sum('total_amount'),
            'unpaid_amount' => PurchaseInvoice::where('payment_status', '!=', 'paid')->sum('remaining_amount'),
        ];

        return view('purchase-invoices.index', compact('purchaseInvoices', 'suppliers', 'stats'));
    }

    /**
     * عرض نموذج إنشاء فاتورة شراء جديدة
     */
    public function create(): View
    {
        $suppliers = Supplier::active()->get();
        $items = Item::active()->get();
        $costCenters = CostCenter::active()->get();
        
        $invoiceNumber = PurchaseInvoice::generateNextInvoiceNumber();
        
        return view('purchase-invoices.create', compact('suppliers', 'items', 'costCenters', 'invoiceNumber'));
    }

    /**
     * حفظ فاتورة شراء جديدة
     */
    public function store(Request $request): RedirectResponse
    {
        $validated = $request->validate([
            'supplier_id' => 'required|exists:suppliers,id',
            'supplier_invoice_number' => 'nullable|string|max:50',
            'invoice_date' => 'required|date',
            'due_date' => 'required|date|after_or_equal:invoice_date',
            'cost_center_id' => 'nullable|exists:cost_centers,id',
            'reference' => 'nullable|string|max:100',
            'payment_method' => 'required|in:cash,bank_transfer,check,credit_card,credit',
            'payment_terms' => 'nullable|string|max:100',
            'currency' => 'required|string|max:3',
            'exchange_rate' => 'required|numeric|min:0.0001',
            'discount_percentage' => 'nullable|numeric|min:0|max:100',
            'discount_amount' => 'nullable|numeric|min:0',
            'tax_percentage' => 'nullable|numeric|min:0|max:100',
            'shipping_amount' => 'nullable|numeric|min:0',
            'other_charges' => 'nullable|numeric|min:0',
            'delivery_address' => 'nullable|string',
            'delivery_date' => 'nullable|date',
            'notes' => 'nullable|string',
            'terms_conditions' => 'nullable|string',
            'items' => 'required|array|min:1',
            'items.*.item_id' => 'required|exists:items,id',
            'items.*.description' => 'nullable|string|max:255',
            'items.*.quantity' => 'required|numeric|min:0.001',
            'items.*.unit' => 'required|string|max:20',
            'items.*.unit_price' => 'required|numeric|min:0',
            'items.*.discount_percentage' => 'nullable|numeric|min:0|max:100',
            'items.*.discount_amount' => 'nullable|numeric|min:0',
            'items.*.tax_percentage' => 'nullable|numeric|min:0|max:100',
            'items.*.expiry_date' => 'nullable|date',
            'items.*.batch_number' => 'nullable|string|max:50',
            'items.*.notes' => 'nullable|string',
        ]);

        DB::transaction(function () use ($validated) {
            $invoice = PurchaseInvoice::create([
                'invoice_number' => PurchaseInvoice::generateNextInvoiceNumber(),
                'supplier_invoice_number' => $validated['supplier_invoice_number'],
                'invoice_date' => $validated['invoice_date'],
                'due_date' => $validated['due_date'],
                'supplier_id' => $validated['supplier_id'],
                'cost_center_id' => $validated['cost_center_id'],
                'reference' => $validated['reference'],
                'status' => 'draft',
                'payment_status' => 'unpaid',
                'payment_method' => $validated['payment_method'],
                'payment_terms' => $validated['payment_terms'],
                'currency' => $validated['currency'],
                'exchange_rate' => $validated['exchange_rate'],
                'discount_percentage' => $validated['discount_percentage'] ?? 0,
                'discount_amount' => $validated['discount_amount'] ?? 0,
                'tax_percentage' => $validated['tax_percentage'] ?? 15,
                'shipping_amount' => $validated['shipping_amount'] ?? 0,
                'other_charges' => $validated['other_charges'] ?? 0,
                'delivery_address' => $validated['delivery_address'],
                'delivery_date' => $validated['delivery_date'],
                'notes' => $validated['notes'],
                'terms_conditions' => $validated['terms_conditions'],
            ]);

            // إضافة تفاصيل الفاتورة
            foreach ($validated['items'] as $itemData) {
                PurchaseInvoiceDetail::create([
                    'purchase_invoice_id' => $invoice->id,
                    'item_id' => $itemData['item_id'],
                    'item_description' => $itemData['description'],
                    'quantity' => $itemData['quantity'],
                    'unit' => $itemData['unit'],
                    'unit_price' => $itemData['unit_price'],
                    'discount_percentage' => $itemData['discount_percentage'] ?? 0,
                    'discount_amount' => $itemData['discount_amount'] ?? 0,
                    'tax_percentage' => $itemData['tax_percentage'] ?? 15,
                    'expiry_date' => $itemData['expiry_date'],
                    'batch_number' => $itemData['batch_number'],
                    'notes' => $itemData['notes'],
                ]);
            }

            // حساب المجاميع
            $invoice->calculateTotals();
        });

        return redirect()->route('purchase-invoices.index')
            ->with('success', 'تم إنشاء فاتورة الشراء بنجاح');
    }

    /**
     * عرض تفاصيل فاتورة الشراء
     */
    public function show(PurchaseInvoice $purchaseInvoice): View
    {
        $purchaseInvoice->load(['supplier', 'costCenter', 'purchaseInvoiceDetails.item', 'journalEntry']);
        
        return view('purchase-invoices.show', compact('purchaseInvoice'));
    }

    /**
     * عرض نموذج تعديل فاتورة الشراء
     */
    public function edit(PurchaseInvoice $purchaseInvoice): View
    {
        // يمكن تعديل الفاتورة فقط في حالة المسودة
        if ($purchaseInvoice->status !== 'draft') {
            abort(403, 'لا يمكن تعديل هذه الفاتورة');
        }

        $purchaseInvoice->load('purchaseInvoiceDetails.item');
        $suppliers = Supplier::active()->get();
        $items = Item::active()->get();
        $costCenters = CostCenter::active()->get();
        
        return view('purchase-invoices.edit', compact('purchaseInvoice', 'suppliers', 'items', 'costCenters'));
    }

    /**
     * تحديث فاتورة الشراء
     */
    public function update(Request $request, PurchaseInvoice $purchaseInvoice): RedirectResponse
    {
        // يمكن تحديث الفاتورة فقط في حالة المسودة
        if ($purchaseInvoice->status !== 'draft') {
            return redirect()->back()->with('error', 'لا يمكن تعديل هذه الفاتورة');
        }

        $validated = $request->validate([
            'supplier_id' => 'required|exists:suppliers,id',
            'supplier_invoice_number' => 'nullable|string|max:50',
            'invoice_date' => 'required|date',
            'due_date' => 'required|date|after_or_equal:invoice_date',
            'cost_center_id' => 'nullable|exists:cost_centers,id',
            'reference' => 'nullable|string|max:100',
            'payment_method' => 'required|in:cash,bank_transfer,check,credit_card,credit',
            'payment_terms' => 'nullable|string|max:100',
            'currency' => 'required|string|max:3',
            'exchange_rate' => 'required|numeric|min:0.0001',
            'discount_percentage' => 'nullable|numeric|min:0|max:100',
            'discount_amount' => 'nullable|numeric|min:0',
            'tax_percentage' => 'nullable|numeric|min:0|max:100',
            'shipping_amount' => 'nullable|numeric|min:0',
            'other_charges' => 'nullable|numeric|min:0',
            'delivery_address' => 'nullable|string',
            'delivery_date' => 'nullable|date',
            'notes' => 'nullable|string',
            'terms_conditions' => 'nullable|string',
            'items' => 'required|array|min:1',
            'items.*.item_id' => 'required|exists:items,id',
            'items.*.description' => 'nullable|string|max:255',
            'items.*.quantity' => 'required|numeric|min:0.001',
            'items.*.unit' => 'required|string|max:20',
            'items.*.unit_price' => 'required|numeric|min:0',
            'items.*.discount_percentage' => 'nullable|numeric|min:0|max:100',
            'items.*.discount_amount' => 'nullable|numeric|min:0',
            'items.*.tax_percentage' => 'nullable|numeric|min:0|max:100',
            'items.*.expiry_date' => 'nullable|date',
            'items.*.batch_number' => 'nullable|string|max:50',
            'items.*.notes' => 'nullable|string',
        ]);

        DB::transaction(function () use ($validated, $purchaseInvoice) {
            // تحديث بيانات الفاتورة
            $purchaseInvoice->update([
                'supplier_invoice_number' => $validated['supplier_invoice_number'],
                'invoice_date' => $validated['invoice_date'],
                'due_date' => $validated['due_date'],
                'supplier_id' => $validated['supplier_id'],
                'cost_center_id' => $validated['cost_center_id'],
                'reference' => $validated['reference'],
                'payment_method' => $validated['payment_method'],
                'payment_terms' => $validated['payment_terms'],
                'currency' => $validated['currency'],
                'exchange_rate' => $validated['exchange_rate'],
                'discount_percentage' => $validated['discount_percentage'] ?? 0,
                'discount_amount' => $validated['discount_amount'] ?? 0,
                'tax_percentage' => $validated['tax_percentage'] ?? 15,
                'shipping_amount' => $validated['shipping_amount'] ?? 0,
                'other_charges' => $validated['other_charges'] ?? 0,
                'delivery_address' => $validated['delivery_address'],
                'delivery_date' => $validated['delivery_date'],
                'notes' => $validated['notes'],
                'terms_conditions' => $validated['terms_conditions'],
            ]);

            // حذف التفاصيل القديمة
            $purchaseInvoice->purchaseInvoiceDetails()->delete();

            // إضافة التفاصيل الجديدة
            foreach ($validated['items'] as $itemData) {
                PurchaseInvoiceDetail::create([
                    'purchase_invoice_id' => $purchaseInvoice->id,
                    'item_id' => $itemData['item_id'],
                    'item_description' => $itemData['description'],
                    'quantity' => $itemData['quantity'],
                    'unit' => $itemData['unit'],
                    'unit_price' => $itemData['unit_price'],
                    'discount_percentage' => $itemData['discount_percentage'] ?? 0,
                    'discount_amount' => $itemData['discount_amount'] ?? 0,
                    'tax_percentage' => $itemData['tax_percentage'] ?? 15,
                    'expiry_date' => $itemData['expiry_date'],
                    'batch_number' => $itemData['batch_number'],
                    'notes' => $itemData['notes'],
                ]);
            }

            // إعادة حساب المجاميع
            $purchaseInvoice->calculateTotals();
        });

        return redirect()->route('purchase-invoices.show', $purchaseInvoice)
            ->with('success', 'تم تحديث فاتورة الشراء بنجاح');
    }

    /**
     * حذف فاتورة الشراء
     */
    public function destroy(PurchaseInvoice $purchaseInvoice): RedirectResponse
    {
        // يمكن حذف الفاتورة فقط في حالة المسودة أو الإلغاء
        if (!in_array($purchaseInvoice->status, ['draft', 'cancelled'])) {
            return redirect()->back()->with('error', 'لا يمكن حذف هذه الفاتورة');
        }

        $purchaseInvoice->delete();

        return redirect()->route('purchase-invoices.index')
            ->with('success', 'تم حذف فاتورة الشراء بنجاح');
    }

    /**
     * اعتماد فاتورة الشراء
     */
    public function approve(PurchaseInvoice $purchaseInvoice): RedirectResponse
    {
        if ($purchaseInvoice->status !== 'pending') {
            return redirect()->back()->with('error', 'لا يمكن اعتماد هذه الفاتورة');
        }

        $purchaseInvoice->approve();

        return redirect()->back()->with('success', 'تم اعتماد فاتورة الشراء بنجاح');
    }

    /**
     * تسجيل استلام البضاعة
     */
    public function markAsReceived(Request $request, PurchaseInvoice $purchaseInvoice): RedirectResponse
    {
        if ($purchaseInvoice->status !== 'approved') {
            return redirect()->back()->with('error', 'لا يمكن تسجيل استلام هذه الفاتورة');
        }

        $validated = $request->validate([
            'received_by' => 'nullable|string|max:100',
            'items' => 'required|array',
            'items.*.id' => 'required|exists:purchase_invoice_details,id',
            'items.*.received_quantity' => 'required|numeric|min:0',
        ]);

        DB::transaction(function () use ($validated, $purchaseInvoice) {
            // تحديث الكميات المستلمة
            foreach ($validated['items'] as $itemData) {
                $detail = $purchaseInvoice->purchaseInvoiceDetails()->find($itemData['id']);
                if ($detail) {
                    $detail->update(['received_quantity' => $itemData['received_quantity']]);
                }
            }

            // تحديث حالة الفاتورة
            $purchaseInvoice->markAsReceived($validated['received_by']);
        });

        return redirect()->back()->with('success', 'تم تسجيل استلام البضاعة بنجاح');
    }

    /**
     * إلغاء فاتورة الشراء
     */
    public function cancel(PurchaseInvoice $purchaseInvoice): RedirectResponse
    {
        if (!in_array($purchaseInvoice->status, ['draft', 'pending', 'approved'])) {
            return redirect()->back()->with('error', 'لا يمكن إلغاء هذه الفاتورة');
        }

        $purchaseInvoice->cancel();

        return redirect()->back()->with('success', 'تم إلغاء فاتورة الشراء بنجاح');
    }

    /**
     * تسديد دفعة على الفاتورة
     */
    public function addPayment(Request $request, PurchaseInvoice $purchaseInvoice): RedirectResponse
    {
        $validated = $request->validate([
            'payment_amount' => 'required|numeric|min:0.01|max:' . $purchaseInvoice->remaining_amount,
            'payment_date' => 'required|date',
            'payment_method' => 'required|in:cash,bank_transfer,check,credit_card',
            'reference' => 'nullable|string|max:100',
            'notes' => 'nullable|string',
        ]);

        DB::transaction(function () use ($validated, $purchaseInvoice) {
            // إضافة الدفعة
            $purchaseInvoice->addPayment($validated['payment_amount']);

            // إنشاء سند صرف
            // يمكن إضافة منطق إنشاء سند الصرف هنا إذا لزم الأمر
        });

        return redirect()->back()->with('success', 'تم تسجيل الدفعة بنجاح');
    }

    /**
     * تغيير حالة الفاتورة
     */
    public function updateStatus(Request $request, PurchaseInvoice $purchaseInvoice): RedirectResponse
    {
        $validated = $request->validate([
            'status' => 'required|in:pending,approved,received,cancelled',
        ]);

        $oldStatus = $purchaseInvoice->status;
        
        switch ($validated['status']) {
            case 'pending':
                if ($oldStatus !== 'draft') {
                    return redirect()->back()->with('error', 'لا يمكن تحويل الفاتورة إلى معلقة');
                }
                $purchaseInvoice->update(['status' => 'pending']);
                break;
                
            case 'approved':
                if ($oldStatus !== 'pending') {
                    return redirect()->back()->with('error', 'لا يمكن اعتماد الفاتورة');
                }
                $purchaseInvoice->approve();
                break;
                
            case 'received':
                if ($oldStatus !== 'approved') {
                    return redirect()->back()->with('error', 'لا يمكن تسجيل استلام الفاتورة');
                }
                $purchaseInvoice->markAsReceived();
                break;
                
            case 'cancelled':
                $purchaseInvoice->cancel();
                break;
        }

        return redirect()->back()->with('success', 'تم تحديث حالة الفاتورة بنجاح');
    }

    /**
     * تصدير فواتير الشراء
     */
    public function export(Request $request)
    {
        // منطق تصدير البيانات
        // يمكن تنفيذه حسب الحاجة
    }

    /**
     * طباعة فاتورة الشراء
     */
    public function print(PurchaseInvoice $purchaseInvoice): View
    {
        $purchaseInvoice->load(['supplier', 'costCenter', 'purchaseInvoiceDetails.item']);
        
        return view('purchase-invoices.print', compact('purchaseInvoice'));
    }
}