<?php

namespace App\Http\Controllers;

use App\Models\Product;
use App\Models\InvoiceItem;
use App\Models\InventoryMovement;
use Illuminate\Http\Request;
use Carbon\Carbon;
use Illuminate\Support\Facades\DB;

class ProductReportController extends Controller
{
    /**
     * Display product reports dashboard
     */
    public function index()
    {
        return view('system-reports.product.index');
    }

    /**
     * Product performance report
     */
    public function performance(Request $request)
    {
        $startDate = $request->get('start_date', now()->startOfMonth());
        $endDate = $request->get('end_date', now()->endOfMonth());

        $productPerformance = Product::leftJoin('invoice_items', 'products.id', '=', 'invoice_items.product_id')
            ->leftJoin('invoices', 'invoice_items.invoice_id', '=', 'invoices.id')
            ->whereBetween('invoices.issue_date', [$startDate, $endDate])
            ->selectRaw('
                products.id,
                products.name,
                products.price,
                products.category,
                COALESCE(SUM(invoice_items.quantity), 0) as total_sold,
                COALESCE(SUM(invoice_items.line_total), 0) as total_revenue,
                COALESCE(AVG(invoice_items.unit_price), 0) as avg_selling_price,
                COUNT(DISTINCT invoices.id) as orders_count
            ')
            ->groupBy('products.id', 'products.name', 'products.price', 'products.category')
            ->orderByDesc('total_revenue')
            ->paginate(15);

        return view('system-reports.product.performance', compact('productPerformance', 'startDate', 'endDate'));
    }

    /**
     * Product inventory report
     */
    public function inventory(Request $request)
    {
        $products = Product::selectRaw('
                id,
                name,
                category,
                price,
                stock_quantity,
                min_stock,
                CASE
                    WHEN stock_quantity <= 0 THEN "نفذ"
                    WHEN stock_quantity <= min_stock THEN "منخفض"
                    WHEN stock_quantity <= min_stock * 2 THEN "متوسط"
                    ELSE "جيد"
                END as stock_status
            ')
            ->orderBy('stock_quantity')
            ->paginate(15);

        $inventoryStats = Product::selectRaw('
                COUNT(*) as total_products,
                COUNT(CASE WHEN stock_quantity <= 0 THEN 1 END) as out_of_stock,
                COUNT(CASE WHEN stock_quantity <= min_stock THEN 1 END) as low_stock,
                SUM(stock_quantity * price) as total_inventory_value,
                AVG(stock_quantity) as avg_stock_level
            ')
            ->first();

        return view('system-reports.product.inventory', compact('products', 'inventoryStats'));
    }

    /**
     * Product movement report
     */
    public function movements(Request $request)
    {
        $startDate = $request->get('start_date', now()->startOfMonth());
        $endDate = $request->get('end_date', now()->endOfMonth());
        $productId = $request->get('product_id');

        $query = InventoryMovement::with('product')
            ->whereBetween('created_at', [$startDate, $endDate]);

        if ($productId) {
            $query->where('product_id', $productId);
        }

        $movements = $query->orderByDesc('created_at')->paginate(15);

        $movementStats = InventoryMovement::whereBetween('created_at', [$startDate, $endDate])
            ->selectRaw('
                type as movement_type,
                COUNT(*) as count,
                SUM(quantity) as total_quantity
            ')
            ->groupBy('type')
            ->get();

        $products = Product::select('id', 'name')->get();

        return view('system-reports.product.movements', compact('movements', 'movementStats', 'products', 'startDate', 'endDate', 'productId'));
    }

    /**
     * Product profitability report
     */
    public function profitability(Request $request)
    {
        $startDate = $request->get('start_date', now()->startOfMonth());
        $endDate = $request->get('end_date', now()->endOfMonth());

        $productProfitability = Product::leftJoin('invoice_items', 'products.id', '=', 'invoice_items.product_id')
            ->leftJoin('invoices', 'invoice_items.invoice_id', '=', 'invoices.id')
            ->whereBetween('invoices.issue_date', [$startDate, $endDate])
            ->selectRaw('
                products.id,
                products.name,
                products.price as cost_price,
                COALESCE(SUM(invoice_items.quantity), 0) as total_sold,
                COALESCE(SUM(invoice_items.line_total), 0) as total_revenue,
                COALESCE(SUM(invoice_items.quantity * products.price), 0) as total_cost,
                COALESCE(SUM(invoice_items.line_total) - SUM(invoice_items.quantity * products.price), 0) as total_profit,
                CASE
                    WHEN SUM(invoice_items.line_total) > 0
                    THEN ((SUM(invoice_items.line_total) - SUM(invoice_items.quantity * products.price)) / SUM(invoice_items.line_total)) * 100
                    ELSE 0
                END as profit_margin
            ')
            ->groupBy('products.id', 'products.name', 'products.price')
            ->havingRaw('total_sold > 0')
            ->orderByDesc('total_profit')
            ->paginate(15);

        return view('system-reports.product.profitability', compact('productProfitability', 'startDate', 'endDate'));
    }

    /**
     * Product category analysis
     */
    public function categoryAnalysis(Request $request)
    {
        $startDate = $request->get('start_date', now()->startOfMonth());
        $endDate = $request->get('end_date', now()->endOfMonth());

        $categoryAnalysis = Product::leftJoin('invoice_items', 'products.id', '=', 'invoice_items.product_id')
            ->leftJoin('invoices', 'invoice_items.invoice_id', '=', 'invoices.id')
            ->whereBetween('invoices.issue_date', [$startDate, $endDate])
            ->selectRaw('
                products.category,
                COUNT(DISTINCT products.id) as product_count,
                COALESCE(SUM(invoice_items.quantity), 0) as total_sold,
                COALESCE(SUM(invoice_items.line_total), 0) as total_revenue,
                COALESCE(AVG(invoice_items.unit_price), 0) as avg_price
            ')
            ->groupBy('products.category')
            ->orderByDesc('total_revenue')
            ->get();

        return view('system-reports.product.category-analysis', compact('categoryAnalysis', 'startDate', 'endDate'));
    }

    /**
     * Top selling products report
     */
    public function topSelling(Request $request)
    {
        $period = $request->get('period', 'month');
        $limit = $request->get('limit', 20);

        $dateRange = match($period) {
            'week' => [now()->startOfWeek(), now()->endOfWeek()],
            'month' => [now()->startOfMonth(), now()->endOfMonth()],
            'quarter' => [now()->startOfQuarter(), now()->endOfQuarter()],
            'year' => [now()->startOfYear(), now()->endOfYear()],
            default => [now()->startOfMonth(), now()->endOfMonth()]
        };

        $topProducts = Product::leftJoin('invoice_items', 'products.id', '=', 'invoice_items.product_id')
            ->leftJoin('invoices', 'invoice_items.invoice_id', '=', 'invoices.id')
            ->whereBetween('invoices.issue_date', $dateRange)
            ->selectRaw('
                products.id,
                products.name,
                products.category,
                products.price,
                COALESCE(SUM(invoice_items.quantity), 0) as total_sold,
                COALESCE(SUM(invoice_items.line_total), 0) as total_revenue
            ')
            ->groupBy('products.id', 'products.name', 'products.category', 'products.price')
            ->orderByDesc('total_sold')
            ->limit($limit)
            ->get();

        return view('system-reports.product.top-selling', compact('topProducts', 'period', 'limit'));
    }

    /**
     * Export product report
     */
    public function export(Request $request)
    {
        $type = $request->get('type', 'performance');
        $format = $request->get('format', 'excel');
        
        // Implementation for export functionality
        return redirect()->back()->with('success', 'تم تصدير التقرير بنجاح');
    }
}