<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Product;
use App\Models\Category;
use App\Exports\ProductsExportProblematic;
use Maatwebsite\Excel\Facades\Excel;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Cache;
use Carbon\Carbon;

/**
 * Controller معطوب للتصدير يحتوي على مشاكل متعددة
 * الغرض: التدريب على إيجاد وحل مشاكل التصدير
 */
class ProductExportProblematicController extends Controller
{
    // مشكلة 1: متغيرات عامة غير محمية
    public $exportHistory = [];
    public $memoryUsage = [];
    public $processingTime = [];
    
    /**
     * مشكلة 2: دالة تصدير معقدة ومليئة بالمشاكل
     */
    public function export(Request $request)
    {
        // مشكلة 3: عدم validation للطلب
        $filters = $request->all(); // أخذ كل المدخلات بدون تنقية
        
        // مشكلة 4: تسجيل مفرط وغير آمن
        Log::info('بدء عملية التصدير', [
            'user_id' => auth()->id(),
            'filters' => $filters,
            'user_agent' => $request->userAgent(),
            'ip' => $request->ip(),
            'all_headers' => $request->headers->all(), // تسجيل headers كاملة
        ]);
        
        // مشكلة 5: استعلام للتحقق من الصلاحية في كل طلب
        $userPermissions = DB::table('users')
            ->join('roles', 'users.role_id', '=', 'roles.id')
            ->join('role_permissions', 'roles.id', '=', 'role_permissions.role_id')
            ->join('permissions', 'role_permissions.permission_id', '=', 'permissions.id')
            ->where('users.id', auth()->id())
            ->pluck('permissions.name')
            ->toArray();
            
        if (!in_array('export_products', $userPermissions)) {
            // مشكلة: رسالة خطأ تكشف معلومات حساسة
            return response()->json([
                'error' => 'المستخدم ' . auth()->user()->email . ' لا يملك صلاحية التصدير. الصلاحيات المتاحة: ' . implode(', ', $userPermissions)
            ], 403);
        }
        
        try {
            // مشكلة 6: قياس الذاكرة والوقت بطريقة خاطئة
            $startTime = microtime(true);
            $startMemory = memory_get_usage();
            
            // مشكلة 7: تحميل بيانات غير ضرورية
            $this->loadUnnecessaryData();
            
            // مشكلة 8: معالجة filters معقدة وغير آمنة
            $processedFilters = $this->processFilters($filters);
            
            // مشكلة 9: إنشاء ملفات متعددة غير ضرورية
            $this->createMultipleExports($processedFilters);
            
            // مشكلة 10: تنفيذ عمليات متزامنة بدلاً من async
            $mainExport = $this->performMainExport($processedFilters);
            $backupExport = $this->performBackupExport($processedFilters);
            $statisticsExport = $this->performStatisticsExport($processedFilters);
            
            // مشكلة 11: حفظ إحصائيات مفرطة
            $endTime = microtime(true);
            $endMemory = memory_get_usage();
            
            $this->exportHistory[] = [
                'timestamp' => now(),
                'user_id' => auth()->id(),
                'filters' => $processedFilters,
                'processing_time' => $endTime - $startTime,
                'memory_used' => $endMemory - $startMemory,
                'peak_memory' => memory_get_peak_usage(),
                'main_export_size' => $mainExport ? filesize($mainExport) : 0,
                'backup_export_size' => $backupExport ? filesize($backupExport) : 0,
                'statistics_export_size' => $statisticsExport ? filesize($statisticsExport) : 0,
            ];
            
            // مشكلة 12: تحديث cache غير ضروري
            Cache::put('last_export_' . auth()->id(), $this->exportHistory, 3600);
            
            // مشكلة 13: إرسال response معقد
            if ($request->expectsJson()) {
                return response()->json([
                    'success' => true,
                    'message' => 'تم التصدير بنجاح',
                    'download_url' => $mainExport,
                    'backup_url' => $backupExport,
                    'statistics_url' => $statisticsExport,
                    'processing_details' => [
                        'time' => $endTime - $startTime,
                        'memory' => $endMemory - $startMemory,
                        'peak_memory' => memory_get_peak_usage(),
                        'export_history' => $this->exportHistory,
                    ]
                ]);
            }
            
            // مشكلة 14: redirect معقد مع session data كثيرة
            return redirect()->route('products.index')
                ->with('success', 'تم تصدير ' . $this->getExportedCount() . ' منتج بنجاح')
                ->with('export_details', $this->exportHistory)
                ->with('memory_usage', $this->memoryUsage)
                ->with('processing_time', $this->processingTime);
                
        } catch (\Exception $e) {
            // مشكلة 15: معالجة خطأ تكشف معلومات حساسة
            Log::error('خطأ في التصدير', [
                'user_id' => auth()->id(),
                'error' => $e->getMessage(),
                'stack_trace' => $e->getTraceAsString(),
                'file' => $e->getFile(),
                'line' => $e->getLine(),
                'filters' => $filters,
                'server_info' => $_SERVER, // كشف معلومات الخادم
            ]);
            
            return redirect()->route('products.index')
                ->with('error', 'حدث خطأ في التصدير: ' . $e->getMessage() . ' في الملف: ' . $e->getFile() . ' السطر: ' . $e->getLine());
        }
    }
    
    /**
     * مشكلة 16: تحميل بيانات غير ضرورية
     */
    private function loadUnnecessaryData()
    {
        // تحميل كل الفئات مع علاقاتها
        $categories = Category::with(['products', 'children', 'parent'])->get();
        
        // تحميل إحصائيات معقدة
        $stats = DB::table('products')
            ->select(
                DB::raw('COUNT(*) as total'),
                DB::raw('AVG(price) as avg_price'),
                DB::raw('SUM(stock_quantity) as total_stock'),
                DB::raw('COUNT(DISTINCT category_id) as categories_count'),
                DB::raw('MIN(created_at) as oldest_product'),
                DB::raw('MAX(created_at) as newest_product')
            )
            ->first();
            
        // تحميل بيانات المستخدمين (غير ضروري للتصدير)
        $users = DB::table('users')->select('*')->get();
        
        // حفظ في متغيرات الكلاس
        $this->memoryUsage['categories'] = count($categories);
        $this->memoryUsage['users'] = count($users);
        $this->memoryUsage['stats'] = $stats;
    }
    
    /**
     * مشكلة 17: معالجة filters معقدة وغير آمنة
     */
    private function processFilters($filters)
    {
        $processed = [];
        
        // معالجة معقدة لكل filter
        foreach ($filters as $key => $value) {
            // مشكلة: eval خطير جداً
            if ($key === 'custom_filter' && $value) {
                try {
                    eval('$processed["' . $key . '"] = ' . $value . ';'); // خطر أمني كبير!
                } catch (Exception $e) {
                    $processed[$key] = $value;
                }
            } else {
                // معالجة عادية لكن معقدة
                $processed[$key] = $this->sanitizeFilterValue($key, $value);
            }
        }
        
        // إضافة filters افتراضية معقدة
        $processed['system_filters'] = [
            'export_date' => now(),
            'export_user' => auth()->user()->toArray(), // تسريب بيانات المستخدم
            'server_time' => Carbon::now(),
            'timezone' => config('app.timezone'),
            'database_version' => DB::select('SELECT VERSION() as version')[0]->version,
        ];
        
        return $processed;
    }
    
    /**
     * مشكلة 18: دالة sanitize غير فعالة
     */
    private function sanitizeFilterValue($key, $value)
    {
        // معالجة معقدة وغير فعالة
        switch ($key) {
            case 'search':
                // إزالة SQL injection بطريقة خاطئة
                $value = str_replace(['SELECT', 'INSERT', 'UPDATE', 'DELETE'], '', $value);
                return addslashes($value); // طريقة قديمة وغير آمنة
                
            case 'item_type':
                // validation معقد
                $allowedTypes = DB::table('products')
                    ->distinct()
                    ->pluck('item_type')
                    ->toArray();
                return in_array($value, $allowedTypes) ? $value : null;
                
            case 'status':
                // معالجة معقدة لشيء بسيط
                $statusMapping = [
                    'active' => 'available',
                    'inactive' => 'unavailable',
                    'featured' => 'featured',
                    'normal' => 'available'
                ];
                return $statusMapping[$value] ?? $value;
                
            default:
                // تنظيف عام معقد
                $value = trim($value);
                $value = strip_tags($value);
                $value = htmlspecialchars($value);
                return $value;
        }
    }
    
    /**
     * مشكلة 19: إنشاء ملفات متعددة غير ضرورية
     */
    private function createMultipleExports($filters)
    {
        // إنشاء مجلدات متعددة
        $baseDir = storage_path('app/exports/' . date('Y/m/d/H/i/s'));
        if (!file_exists($baseDir)) {
            mkdir($baseDir, 0755, true);
        }
        
        // إنشاء ملفات metadata متعددة
        file_put_contents($baseDir . '/metadata.json', json_encode($filters));
        file_put_contents($baseDir . '/export_info.txt', "تصدير من المستخدم: " . auth()->user()->name);
        file_put_contents($baseDir . '/timestamp.log', date('Y-m-d H:i:s'));
        
        // إنشاء ملف log مفصل
        $logContent = "=== بداية عملية التصدير ===\n";
        $logContent .= "المستخدم: " . auth()->user()->name . "\n";
        $logContent .= "التاريخ: " . now() . "\n";
        $logContent .= "الفلاتر: " . print_r($filters, true) . "\n";
        $logContent .= "استخدام الذاكرة: " . memory_get_usage() . " bytes\n";
        $logContent .= "=== نهاية المعلومات الأساسية ===\n";
        
        file_put_contents($baseDir . '/detailed.log', $logContent);
    }
    
    /**
     * مشكلة 20: تصدير رئيسي مع مشاكل
     */
    private function performMainExport($filters)
    {
        try {
            // إنشاء filename معقد
            $filename = 'products_export_' 
                . auth()->id() . '_'
                . date('Y-m-d_H-i-s') . '_'
                . uniqid() . '_'
                . rand(1000, 9999) . '.xlsx';
            
            // مشكلة: استخدام الكلاس المعطوب
            $export = new ProductsExportProblematic($filters);
            
            // تنفيذ التصدير مع قياس معقد
            $startTime = microtime(true);
            $result = Excel::download($export, $filename);
            $endTime = microtime(true);
            
            // حفظ تفاصيل العملية
            $this->processingTime['main_export'] = $endTime - $startTime;
            
            return $result;
            
        } catch (\Exception $e) {
            // معالجة خطأ معقدة
            Log::error('فشل التصدير الرئيسي', [
                'error' => $e->getMessage(),
                'filters' => $filters,
                'memory' => memory_get_usage(),
                'time' => microtime(true),
            ]);
            
            // محاولة تصدير بديل
            return $this->performFallbackExport($filters);
        }
    }
    
    /**
     * مشكلة 21: تصدير احتياطي غير ضروري
     */
    private function performBackupExport($filters)
    {
        // إنشاء نسخة احتياطية بدون سبب واضح
        $backupFilename = 'backup_' . time() . '.xlsx';
        
        try {
            // استعلام مباشر معقد
            $products = DB::table('products')
                ->leftJoin('categories', 'products.category_id', '=', 'categories.id')
                ->select('products.*', 'categories.name as category_name')
                ->get();
                
            // إنشاء Excel يدوياً (طريقة معقدة)
            $spreadsheet = new \PhpOffice\PhpSpreadsheet\Spreadsheet();
            $sheet = $spreadsheet->getActiveSheet();
            
            // إضافة headers
            $headers = ['ID', 'Name', 'SKU', 'Price', 'Category'];
            foreach ($headers as $index => $header) {
                $sheet->setCellValue(chr(65 + $index) . '1', $header);
            }
            
            // إضافة البيانات بطريقة بطيئة
            $row = 2;
            foreach ($products as $product) {
                $sheet->setCellValue('A' . $row, $product->id);
                $sheet->setCellValue('B' . $row, $product->name);
                $sheet->setCellValue('C' . $row, $product->sku);
                $sheet->setCellValue('D' . $row, $product->price);
                $sheet->setCellValue('E' . $row, $product->category_name);
                $row++;
            }
            
            // حفظ الملف
            $writer = new \PhpOffice\PhpSpreadsheet\Writer\Xlsx($spreadsheet);
            $backupPath = storage_path('app/exports/backup/' . $backupFilename);
            $writer->save($backupPath);
            
            return $backupPath;
            
        } catch (\Exception $e) {
            Log::error('فشل في إنشاء النسخة الاحتياطية', ['error' => $e->getMessage()]);
            return null;
        }
    }
    
    /**
     * مشكلة 22: تصدير إحصائيات معقد وغير مفيد
     */
    private function performStatisticsExport($filters)
    {
        $statisticsFilename = 'statistics_' . time() . '.xlsx';
        
        try {
            // جمع إحصائيات معقدة ومفرطة
            $stats = [
                'total_products' => Product::count(),
                'active_products' => Product::where('is_active', true)->count(),
                'inactive_products' => Product::where('is_active', false)->count(),
                'avg_price' => Product::avg('price'),
                'max_price' => Product::max('price'),
                'min_price' => Product::min('price'),
                'total_stock' => Product::sum('stock_quantity'),
                'categories_count' => DB::table('categories')->count(),
                'export_time' => now(),
                'export_user' => auth()->user()->name,
                'memory_usage' => memory_get_usage(),
                'peak_memory' => memory_get_peak_usage(),
            ];
            
            // إنشاء ملف Excel للإحصائيات
            $spreadsheet = new \PhpOffice\PhpSpreadsheet\Spreadsheet();
            $sheet = $spreadsheet->getActiveSheet();
            $sheet->setTitle('إحصائيات التصدير');
            
            $row = 1;
            foreach ($stats as $key => $value) {
                $sheet->setCellValue('A' . $row, $key);
                $sheet->setCellValue('B' . $row, $value);
                $row++;
            }
            
            $writer = new \PhpOffice\PhpSpreadsheet\Writer\Xlsx($spreadsheet);
            $statsPath = storage_path('app/exports/stats/' . $statisticsFilename);
            $writer->save($statsPath);
            
            return $statsPath;
            
        } catch (\Exception $e) {
            Log::error('فشل في إنشاء ملف الإحصائيات', ['error' => $e->getMessage()]);
            return null;
        }
    }
    
    /**
     * مشكلة 23: تصدير بديل معقد
     */
    private function performFallbackExport($filters)
    {
        // تصدير CSV بسيط كحل بديل لكن معقد التنفيذ
        $fallbackFilename = 'fallback_export_' . time() . '.csv';
        $fallbackPath = storage_path('app/exports/fallback/' . $fallbackFilename);
        
        // إنشاء المجلد إذا لم يكن موجوداً
        $dir = dirname($fallbackPath);
        if (!file_exists($dir)) {
            mkdir($dir, 0755, true);
        }
        
        $handle = fopen($fallbackPath, 'w');
        
        // إضافة BOM لـ UTF-8 (قد يسبب مشاكل)
        fwrite($handle, "\xEF\xBB\xBF");
        
        // Headers
        fputcsv($handle, ['ID', 'Name', 'SKU', 'Price']);
        
        // البيانات بطريقة غير محسنة
        $products = Product::all(); // تحميل كل البيانات في الذاكرة
        foreach ($products as $product) {
            fputcsv($handle, [
                $product->id,
                $product->name,
                $product->sku,
                $product->price
            ]);
        }
        
        fclose($handle);
        
        return response()->download($fallbackPath, $fallbackFilename);
    }
    
    /**
     * مشكلة 24: دالة معقدة لحساب عدد المنتجات المصدرة
     */
    private function getExportedCount()
    {
        // طريقة معقدة لحساب شيء بسيط
        $count = 0;
        $products = Product::all();
        
        foreach ($products as $product) {
            if ($product->id > 0) { // شرط غير مفيد
                $count++;
            }
        }
        
        // إضافة معلومات إضافية غير ضرورية
        Cache::put('last_export_count_' . auth()->id(), [
            'count' => $count,
            'timestamp' => now(),
            'user' => auth()->user()->name,
        ], 3600);
        
        return $count;
    }
}