<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Product;
use App\Models\Category;
use App\Exports\ProductsExport;
use App\Imports\ProductsImport;
use Maatwebsite\Excel\Facades\Excel;
use Illuminate\Support\Facades\Storage;

class ProductController extends Controller
{
    /**
     * عرض قائمة المنتجات
     */
    public function index(Request $request)
    {
        $query = Product::query();
        
        // فلترة حسب نوع العنصر (item_type)
        if ($request->filled('item_type')) {
            $query->where('item_type', $request->item_type);
        }
        
        // فلترة حسب النوع القديم (type) للتوافق مع الإصدارات السابقة
        if ($request->filled('type')) {
            $query->where('type', $request->type);
        }
        
        // فلترة حسب الحالة
        if ($request->filled('status')) {
            switch ($request->status) {
                case 'available':
                    $query->where('is_active', true)->where('is_available', true);
                    break;
                case 'unavailable':
                    $query->where('is_active', true)->where('is_available', false);
                    break;
                case 'inactive':
                    $query->where('is_active', false);
                    break;
                case 'low_stock':
                    $query->whereColumn('stock_quantity', '<=', 'min_stock');
                    break;
            }
        }
        
        // فلترة حسب الفئة
        if ($request->filled('category_id')) {
            $query->where('category_id', $request->category_id);
        }
        
        // فلترة حسب المنتجات المميزة
        if ($request->filled('is_featured')) {
            $query->where('is_featured', $request->is_featured === 'true');
        }
        
        // البحث
        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function($q) use ($search) {
                $q->where('name', 'like', "%{$search}%")
                  ->orWhere('name_en', 'like', "%{$search}%")
                  ->orWhere('sku', 'like', "%{$search}%")
                  ->orWhere('barcode', 'like', "%{$search}%");
            });
        }
        
        $products = $query->orderBy('created_at', 'desc')->paginate(15);
        
        return view('products.index', compact('products'));
    }

    /**
     * عرض نموذج إنشاء منتج جديد
     */
    public function create()
    {
        $categories = Category::getSelectOptions();
        return view('products.create', compact('categories'));
    }

    /**
     * حفظ منتج جديد
     */
    public function store(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'name_en' => 'nullable|string|max:255',
            'description' => 'nullable|string',
            'sku' => 'required|string|max:255|unique:products,sku',
            'barcode' => 'nullable|string|max:255|unique:products,barcode',
            'type' => 'required|in:product,service',
            'category_id' => 'nullable|exists:categories,id',
            'price' => 'required|numeric|min:0',
            'tax_rate' => 'required|numeric|min:0|max:100',
            'stock_quantity' => 'required_if:type,product|integer|min:0',
            'min_stock' => 'required_if:type,product|integer|min:0',
            'unit' => 'required|string|max:50',
            'zatca_category_code' => 'nullable|string|max:50',
            'is_taxable' => 'boolean',
            'is_active' => 'boolean',
        ]);

        $data = $request->all();
        
        // حساب السعر مع الضريبة
        if ($request->is_taxable) {
            $data['price_with_tax'] = $data['price'] * (1 + ($data['tax_rate'] / 100));
        } else {
            $data['price_with_tax'] = $data['price'];
        }
        
        // للخدمات، جعل المخزون غير محدود
        if ($data['type'] === 'service') {
            $data['stock_quantity'] = 0;
            $data['min_stock'] = 0;
        }

        $product = Product::create($data);

        // إذا كان الطلب AJAX، أرجع JSON response
        if ($request->ajax() || $request->wantsJson()) {
            return response()->json([
                'success' => true,
                'message' => 'تم إنشاء المنتج بنجاح',
                'product' => $product
            ], 201);
        }

        return redirect()->route('products.index')
            ->with('success', 'تم إنشاء المنتج بنجاح');
    }

    /**
     * عرض تفاصيل المنتج
     */
    public function show(Product $product)
    {
        return view('products.show', compact('product'));
    }

    /**
     * عرض نموذج تعديل المنتج
     */
    public function edit(Product $product)
    {
        $categories = Category::getSelectOptions();
        return view('products.edit', compact('product', 'categories'));
    }

    /**
     * تحديث المنتج
     */
    public function update(Request $request, Product $product)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'name_en' => 'nullable|string|max:255',
            'description' => 'nullable|string',
            'sku' => 'required|string|max:255|unique:products,sku,' . $product->id,
            'barcode' => 'nullable|string|max:255|unique:products,barcode,' . $product->id,
            'type' => 'required|in:product,service',
            'category_id' => 'nullable|exists:categories,id',
            'price' => 'required|numeric|min:0',
            'tax_rate' => 'required|numeric|min:0|max:100',
            'stock_quantity' => 'required_if:type,product|integer|min:0',
            'min_stock' => 'required_if:type,product|integer|min:0',
            'unit' => 'required|string|max:50',
            'zatca_category_code' => 'nullable|string|max:50',
            'is_taxable' => 'boolean',
            'is_active' => 'boolean',
        ]);

        $data = $request->all();
        
        // حساب السعر مع الضريبة
        if ($request->is_taxable) {
            $data['price_with_tax'] = $data['price'] * (1 + ($data['tax_rate'] / 100));
        } else {
            $data['price_with_tax'] = $data['price'];
        }
        
        // للخدمات، جعل المخزون غير محدود
        if ($data['type'] === 'service') {
            $data['stock_quantity'] = 0;
            $data['min_stock'] = 0;
        }

        $product->update($data);

        return redirect()->route('products.index')
            ->with('success', 'تم تحديث المنتج بنجاح');
    }

    /**
     * حذف المنتج
     */
    public function destroy(Product $product)
    {
        // التحقق من عدم استخدام المنتج في فواتير
        if ($product->invoiceItems()->count() > 0) {
            return redirect()->route('products.index')
                ->with('error', 'لا يمكن حذف المنتج لأنه مستخدم في فواتير');
        }

        $product->delete();

        return redirect()->route('products.index')
            ->with('success', 'تم حذف المنتج بنجاح');
    }

    /**
     * تصدير المنتجات إلى Excel
     */
    public function export(Request $request)
    {
        try {
            // جمع جميع المرشحات المتاحة
            $filters = $request->only([
                'item_type',
                'status',
                'category_id',
                'is_featured',
                'search',
                'type' // للتوافق مع الإصدارات السابقة
            ]);
            
            // تنظيف المرشحات من القيم الفارغة
            $filters = array_filter($filters, function($value) {
                return !is_null($value) && $value !== '';
            });
            
            $filename = 'products_export_' . date('Y-m-d_H-i-s') . '.xlsx';
            
            // التحقق من وضع التدريب على المشاكل
            if ($request->has('use_problematic') && $request->get('use_problematic') === 'true') {
                // استخدام النسخة المعطوبة للتدريب
                return app(\App\Http\Controllers\ProductExportProblematicController::class)->export($request);
            }
            
            return Excel::download(new ProductsExport($filters), $filename);
            
        } catch (\Exception $e) {
            \Log::error('خطأ في تصدير المنتجات: ' . $e->getMessage(), [
                'filters' => $filters ?? [],
                'user_id' => auth()->id(),
                'stack_trace' => $e->getTraceAsString()
            ]);
            
            if ($request->ajax() || $request->wantsJson()) {
                return response()->json([
                    'success' => false,
                    'message' => 'حدث خطأ أثناء تصدير البيانات: ' . $e->getMessage()
                ], 500);
            }
            
            return redirect()->route('products.index')
                ->with('error', 'حدث خطأ أثناء تصدير البيانات: ' . $e->getMessage());
        }
    }

    /**
     * استيراد المنتجات من Excel
     */
    public function import(Request $request)
    {
        $request->validate([
            'excel_file' => 'required|file|mimes:xlsx,xls|max:10240', // 10MB max
            'preview_mode' => 'sometimes|boolean'
        ], [
            'excel_file.required' => 'يرجى تحديد ملف Excel',
            'excel_file.mimes' => 'يجب أن يكون الملف من نوع Excel (.xlsx أو .xls)',
            'excel_file.max' => 'حجم الملف يجب أن يكون أقل من 10 ميجابايت',
        ]);

        try {
            // Log the import attempt
            \Log::info('بدء عملية استيراد المنتجات', [
                'file_name' => $request->file('excel_file')->getClientOriginalName(),
                'file_size' => $request->file('excel_file')->getSize(),
                'user_id' => auth()->id(),
                'preview_mode' => $request->boolean('preview_mode')
            ]);

            // Generate unique session ID for this import
            $sessionId = uniqid('import_');
            $previewMode = $request->boolean('preview_mode');
            
            $import = new ProductsImport($previewMode, $sessionId);
            
            Excel::import($import, $request->file('excel_file'));
            
            $summary = $import->getImportSummary();
            
            // Mark import as completed
            if (!$previewMode) {
                $import->markAsCompleted();
            }
            
            // Build response message
            if ($previewMode) {
                $message = "تم تحليل الملف بنجاح - عرض معاينة";
            } else {
                $message = $this->buildImportMessage($summary);
            }

            // Log successful completion
            \Log::info('انتهت عملية استيراد المنتجات', $summary);

            if ($request->ajax() || $request->wantsJson()) {
                return response()->json([
                    'success' => true,
                    'message' => $message,
                    'session_id' => $sessionId,
                    'preview_mode' => $previewMode,
                    'summary' => $summary,
                    'imported' => $summary['imported'],
                    'updated' => $summary['updated'],
                    'skipped' => $summary['skipped'],
                    'errors' => $summary['errors'],
                    'success_rate' => $summary['success_rate'],
                    'preview_data' => $previewMode ? $summary['preview_data'] : null
                ]);
            }

            return redirect()->route('products.index')
                ->with('success', $message);
                
        } catch (\Maatwebsite\Excel\Validators\ValidationException $e) {
            $failures = $e->failures();
            $errorMessage = 'فشل في التحقق من البيانات:';
            
            foreach ($failures as $failure) {
                $errorMessage .= "\nالصف {$failure->row()}: " . implode(', ', $failure->errors());
            }

            \Log::error('خطأ في التحقق من بيانات الاستيراد', [
                'failures' => $failures,
                'file_name' => $request->file('excel_file')->getClientOriginalName()
            ]);

            if ($request->ajax() || $request->wantsJson()) {
                return response()->json([
                    'success' => false,
                    'message' => $errorMessage,
                    'failures' => $failures
                ], 422);
            }

            return redirect()->route('products.index')
                ->with('error', $errorMessage);
                
        } catch (\Exception $e) {
            $errorMessage = 'حدث خطأ أثناء استيراد البيانات: ' . $e->getMessage();

            \Log::error('خطأ عام في عملية الاستيراد', [
                'error_message' => $e->getMessage(),
                'file_name' => $request->file('excel_file')->getClientOriginalName(),
                'stack_trace' => $e->getTraceAsString(),
                'user_id' => auth()->id()
            ]);

            if ($request->ajax() || $request->wantsJson()) {
                return response()->json([
                    'success' => false,
                    'message' => $errorMessage
                ], 500);
            }

            return redirect()->route('products.index')
                ->with('error', $errorMessage);
        }
    }

    /**
     * Get import progress
     */
    public function getImportProgress(Request $request)
    {
        $sessionId = $request->get('session_id');
        
        if (!$sessionId) {
            return response()->json(['error' => 'Session ID is required'], 400);
        }
        
        $progress = ProductsImport::getImportProgress($sessionId);
        
        if (!$progress) {
            return response()->json(['error' => 'Session not found'], 404);
        }
        
        return response()->json($progress);
    }

    /**
     * Preview import data
     */
    public function previewImport(Request $request)
    {
        $request->validate([
            'excel_file' => 'required|file|mimes:xlsx,xls|max:10240'
        ]);

        try {
            $import = new ProductsImport(true); // Enable preview mode
            Excel::import($import, $request->file('excel_file'));
            
            $previewData = $import->getPreviewData();
            $summary = $import->getImportSummary();
            
            return response()->json([
                'success' => true,
                'preview_data' => array_slice($previewData, 0, 10), // Show first 10 rows
                'total_rows' => count($previewData),
                'summary' => $summary
            ]);
            
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'خطأ في معاينة الملف: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Build import success message
     */
    private function buildImportMessage(array $summary): string
    {
        $parts = [];
        
        if ($summary['imported'] > 0) {
            $parts[] = "تم إنشاء {$summary['imported']} منتج جديد";
        }
        
        if ($summary['updated'] > 0) {
            $parts[] = "تم تحديث {$summary['updated']} منتج موجود";
        }
        
        if ($summary['skipped'] > 0) {
            $parts[] = "تم تخطي {$summary['skipped']} صف";
        }
        
        if (empty($parts)) {
            return "تم معالجة ملف الاستيراد بنجاح";
        }
        
        $message = implode('، ', $parts);
        
        if ($summary['success_rate'] < 100) {
            $message .= " (معدل النجاح: {$summary['success_rate']}%)";
        }
        
        return $message;
    }

    /**
     * تحميل قالب Excel للاستيراد
     */
    public function exportTemplate()
    {
        try {
            $headers = [
                'الاسم',
                'الاسم بالإنجليزية',
                'الوصف',
                'الرمز (SKU)',
                'الباركود',
                'النوع',
                'نوع العنصر',
                'السعر',
                'معدل الضريبة (%)',
                'خاضع للضريبة',
                'الوحدة',
                'كمية المخزون',
                'الحد الأدنى للمخزون',
                'وقت التحضير (دقيقة)',
                'السعرات الحرارية',
                'نشط',
                'متوفر',
                'مميز',
                'نباتي',
                'نباتي صرف',
                'خالي من الجلوتين',
                'حار'
            ];

            $sampleData = [
                [
                    'همبرغر لحم',
                    'Beef Burger',
                    'همبرغر لحم طازج مع الخضار',
                    'BURGER-001',
                    '1234567890123',
                    'منتج',
                    'طعام',
                    25.50,
                    15,
                    'نعم',
                    'قطعة',
                    100,
                    10,
                    15,
                    450,
                    'نعم',
                    'نعم',
                    'نعم',
                    'لا',
                    'لا',
                    'لا',
                    'لا'
                ],
                [
                    'عصير برتقال',
                    'Orange Juice',
                    'عصير برتقال طبيعي طازج',
                    'JUICE-001',
                    '1234567890124',
                    'منتج',
                    'مشروب',
                    12.00,
                    15,
                    'نعم',
                    'كوب',
                    50,
                    5,
                    5,
                    120,
                    'نعم',
                    'نعم',
                    'لا',
                    'نعم',
                    'نعم',
                    'نعم',
                    'لا'
                ]
            ];

            $data = collect([$headers])->merge($sampleData);

            $filename = 'products_template_' . date('Y-m-d') . '.xlsx';

            return Excel::download(new class($data) implements \Maatwebsite\Excel\Concerns\FromCollection, \Maatwebsite\Excel\Concerns\WithStyles, \Maatwebsite\Excel\Concerns\ShouldAutoSize {
                private $data;

                public function __construct($data)
                {
                    $this->data = $data;
                }

                public function collection()
                {
                    return $this->data;
                }

                public function styles(\PhpOffice\PhpSpreadsheet\Worksheet\Worksheet $sheet)
                {
                    return [
                        1 => [
                            'font' => ['bold' => true, 'size' => 12],
                            'fill' => [
                                'fillType' => \PhpOffice\PhpSpreadsheet\Style\Fill::FILL_SOLID,
                                'startColor' => ['rgb' => '4F46E5']
                            ],
                            'font' => ['color' => ['rgb' => 'FFFFFF'], 'bold' => true]
                        ]
                    ];
                }
            }, $filename);

        } catch (\Exception $e) {
            return redirect()->route('products.index')
                ->with('error', 'حدث خطأ أثناء تحميل القالب: ' . $e->getMessage());
        }
    }
}
