<?php

namespace App\Http\Controllers;

use App\Models\PurchaseOrder;
use App\Models\PurchaseInvoice;
use App\Models\GoodsReceipt;
use App\Models\Supplier;
use App\Models\Product;
use Illuminate\Http\Request;
use Illuminate\Http\RedirectResponse;
use Illuminate\View\View;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;

class ProcurementWorkflowController extends Controller
{
    /**
     * لوحة تحكم إدارة سير العمل
     */
    public function dashboard(): View
    {
        $stats = [
            // أوامر الشراء
            'pending_orders' => PurchaseOrder::where('status', 'pending')->count(),
            'approved_orders' => PurchaseOrder::where('status', 'approved')->count(),
            'completed_orders' => PurchaseOrder::where('status', 'completed')->count(),
            'cancelled_orders' => PurchaseOrder::where('status', 'cancelled')->count(),
            
            // الفواتير
            'draft_invoices' => PurchaseInvoice::where('status', 'draft')->count(),
            'pending_invoices' => PurchaseInvoice::where('status', 'pending')->count(),
            'approved_invoices' => PurchaseInvoice::where('status', 'approved')->count(),
            
            // استلام البضائع
            'pending_receipts' => GoodsReceipt::where('status', 'pending')->count(),
            'partial_receipts' => GoodsReceipt::where('status', 'partial')->count(),
            'completed_receipts' => GoodsReceipt::where('status', 'completed')->count(),
        ];

        // أوامر الشراء المعلقة الموافقة
        $pendingApprovalOrders = PurchaseOrder::with(['supplier', 'user'])
            ->where('status', 'pending')
            ->orderBy('order_date')
            ->take(10)
            ->get();

        // أوامر الشراء المتأخرة التسليم
        $overdueOrders = PurchaseOrder::with(['supplier'])
            ->where('status', 'approved')
            ->where('expected_delivery_date', '<', now())
            ->orderBy('expected_delivery_date')
            ->take(10)
            ->get();

        // تحليل الأداء الشهري
        $monthlyStats = $this->getMonthlyProcurementStats();

        // رسم بياني لتوزيع الأوامر حسب الحالة
        $orderStatusDistribution = $this->getOrderStatusDistribution();

        return view('procurement.workflow.dashboard', compact(
            'stats', 'pendingApprovalOrders', 'overdueOrders', 
            'monthlyStats', 'orderStatusDistribution'
        ));
    }

    /**
     * إدارة أوامر الشراء المتقدمة
     */
    public function orderManagement(Request $request): View
    {
        $query = PurchaseOrder::with(['supplier', 'user', 'items.product']);

        // فلترة حسب الحالة
        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        // فلترة حسب المورد
        if ($request->filled('supplier_id')) {
            $query->where('supplier_id', $request->supplier_id);
        }

        // فلترة حسب التاريخ
        if ($request->filled('date_from')) {
            $query->whereDate('order_date', '>=', $request->date_from);
        }
        if ($request->filled('date_to')) {
            $query->whereDate('order_date', '<=', $request->date_to);
        }

        // فلترة حسب القيمة
        if ($request->filled('amount_from')) {
            $query->where('total_amount', '>=', $request->amount_from);
        }
        if ($request->filled('amount_to')) {
            $query->where('total_amount', '<=', $request->amount_to);
        }

        // البحث النصي
        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function($q) use ($search) {
                $q->where('po_number', 'like', "%{$search}%")
                  ->orWhere('notes', 'like', "%{$search}%")
                  ->orWhereHas('supplier', function($sq) use ($search) {
                      $sq->where('name', 'like', "%{$search}%");
                  });
            });
        }

        $orders = $query->latest()->paginate(15)->withQueryString();
        $suppliers = Supplier::active()->get();

        return view('procurement.workflow.orders', compact('orders', 'suppliers'));
    }

    /**
     * موافقة مجمعة على أوامر الشراء
     */
    public function bulkApproveOrders(Request $request): RedirectResponse
    {
        $orderIds = $request->get('order_ids', []);
        
        if (empty($orderIds)) {
            return redirect()->back()->with('error', 'يرجى اختيار أوامر للموافقة عليها');
        }

        $count = 0;
        DB::transaction(function () use ($orderIds, &$count) {
            foreach ($orderIds as $orderId) {
                $order = PurchaseOrder::find($orderId);
                if ($order && $order->status === 'pending') {
                    $order->update([
                        'status' => 'approved',
                        'approved_by' => auth()->id(),
                        'approved_at' => now(),
                    ]);
                    $count++;
                }
            }
        });

        return redirect()->back()
            ->with('success', "تم الموافقة على {$count} أمر شراء");
    }

    /**
     * تتبع أوامر الشراء
     */
    public function trackOrder(PurchaseOrder $order): View
    {
        $order->load(['supplier', 'user', 'items.product', 'receipts.items']);
        
        // حساب نسبة التقدم
        $totalItems = $order->items->sum('quantity');
        $receivedItems = $order->receipts->flatMap->items->sum('received_quantity');
        $completionPercentage = $totalItems > 0 ? ($receivedItems / $totalItems) * 100 : 0;

        // الفواتير المرتبطة
        $relatedInvoices = PurchaseInvoice::where('supplier_id', $order->supplier_id)
            ->whereBetween('invoice_date', [
                $order->order_date->subDays(7),
                $order->expected_delivery_date->addDays(30)
            ])
            ->get();

        // تاريخ الأنشطة
        $timeline = $this->getOrderTimeline($order);

        return view('procurement.workflow.track-order', compact(
            'order', 'completionPercentage', 'relatedInvoices', 'timeline'
        ));
    }

    /**
     * تحليل الموردين المتقدم
     */
    public function supplierPerformance(Request $request): View
    {
        $dateFrom = $request->get('date_from', now()->startOfYear()->format('Y-m-d'));
        $dateTo = $request->get('date_to', now()->format('Y-m-d'));

        $suppliers = Supplier::with([
            'purchaseOrders' => function($query) use ($dateFrom, $dateTo) {
                $query->whereBetween('order_date', [$dateFrom, $dateTo]);
            },
            'purchaseInvoices' => function($query) use ($dateFrom, $dateTo) {
                $query->whereBetween('invoice_date', [$dateFrom, $dateTo])
                      ->where('status', '!=', 'cancelled');
            }
        ])->get()->map(function ($supplier) {
            $orders = $supplier->purchaseOrders;
            $invoices = $supplier->purchaseInvoices;
            
            // مؤشرات الأداء
            $onTimeDelivery = $orders->where('status', 'completed')
                ->where('actual_delivery_date', '<=', 'expected_delivery_date')
                ->count();
            $totalCompletedOrders = $orders->where('status', 'completed')->count();
            $onTimeRate = $totalCompletedOrders > 0 ? ($onTimeDelivery / $totalCompletedOrders) * 100 : 0;

            // معدل جودة المنتجات (افتراضي بناء على عدد المرتجعات)
            $qualityRate = 95; // يمكن حسابه من بيانات فعلية

            // متوسط وقت التسليم
            $avgDeliveryTime = $orders->where('status', 'completed')
                ->avg(function($order) {
                    return $order->order_date->diffInDays($order->actual_delivery_date ?? $order->expected_delivery_date);
                }) ?? 0;

            return [
                'supplier' => $supplier,
                'total_orders' => $orders->count(),
                'completed_orders' => $totalCompletedOrders,
                'total_value' => $invoices->sum('total_amount'),
                'on_time_rate' => round($onTimeRate, 1),
                'quality_rate' => $qualityRate,
                'avg_delivery_days' => round($avgDeliveryTime),
                'payment_terms_compliance' => $this->calculatePaymentCompliance($supplier),
            ];
        })->sortByDesc('total_value');

        return view('procurement.workflow.supplier-performance', compact(
            'suppliers', 'dateFrom', 'dateTo'
        ));
    }

    /**
     * تحسين المخزون والشراء
     */
    public function inventoryOptimization(): View
    {
        // المنتجات التي تحتاج إعادة طلب
        $reorderProducts = Product::where('current_stock', '<=', DB::raw('reorder_level'))
            ->where('is_active', true)
            ->with(['supplier'])
            ->get()
            ->map(function ($product) {
                $avgMonthlyUsage = $this->getAverageMonthlyUsage($product->id);
                $suggestedOrderQty = max($product->reorder_level * 2, $avgMonthlyUsage * 3);
                
                return [
                    'product' => $product,
                    'current_stock' => $product->current_stock,
                    'reorder_level' => $product->reorder_level,
                    'avg_monthly_usage' => $avgMonthlyUsage,
                    'suggested_order_qty' => $suggestedOrderQty,
                    'estimated_cost' => $suggestedOrderQty * $product->purchase_price,
                ];
            });

        // تحليل المنتجات بطيئة الحركة
        $slowMovingProducts = Product::whereHas('inventoryMovements', function($query) {
                $query->where('created_at', '>=', now()->subMonths(6));
            }, '<=', 2)
            ->where('current_stock', '>', 0)
            ->with(['supplier'])
            ->get();

        // توصيات الشراء الذكية
        $purchaseRecommendations = $this->generatePurchaseRecommendations();

        return view('procurement.workflow.inventory-optimization', compact(
            'reorderProducts', 'slowMovingProducts', 'purchaseRecommendations'
        ));
    }

    /**
     * إنشاء أمر شراء تلقائي من التوصيات
     */
    public function createAutoPurchaseOrder(Request $request): RedirectResponse
    {
        $validated = $request->validate([
            'supplier_id' => 'required|exists:suppliers,id',
            'products' => 'required|array',
            'products.*.product_id' => 'required|exists:products,id',
            'products.*.quantity' => 'required|numeric|min:1',
        ]);

        DB::transaction(function () use ($validated) {
            $supplier = Supplier::find($validated['supplier_id']);
            
            // إنشاء رقم أمر الشراء
            $poNumber = 'PO-' . date('Y') . '-' . str_pad(
                PurchaseOrder::whereYear('created_at', date('Y'))->count() + 1,
                6, '0', STR_PAD_LEFT
            );

            // إنشاء أمر الشراء
            $order = PurchaseOrder::create([
                'supplier_id' => $supplier->id,
                'user_id' => auth()->id(),
                'po_number' => $poNumber,
                'order_date' => now(),
                'expected_delivery_date' => now()->addDays($supplier->delivery_days ?? 14),
                'status' => 'draft',
                'notes' => 'أمر شراء تلقائي من نظام تحسين المخزون',
            ]);

            $subtotal = 0;
            $taxAmount = 0;

            // إضافة المنتجات
            foreach ($validated['products'] as $productData) {
                $product = Product::find($productData['product_id']);
                $quantity = $productData['quantity'];
                $unitPrice = $product->purchase_price ?? $product->cost_price ?? 0;
                $itemTotal = $quantity * $unitPrice;
                $itemTax = $itemTotal * 0.15; // 15% VAT
                
                $order->items()->create([
                    'product_id' => $product->id,
                    'quantity' => $quantity,
                    'unit_price' => $unitPrice,
                    'tax_rate' => 15,
                    'total_amount' => $itemTotal + $itemTax,
                ]);

                $subtotal += $itemTotal;
                $taxAmount += $itemTax;
            }

            // تحديث مجاميع أمر الشراء
            $order->update([
                'subtotal' => $subtotal,
                'tax_amount' => $taxAmount,
                'total_amount' => $subtotal + $taxAmount,
            ]);
        });

        return redirect()->route('purchase-orders.index')
            ->with('success', 'تم إنشاء أمر الشراء التلقائي بنجاح');
    }

    /**
     * الحصول على إحصائيات المشتريات الشهرية
     */
    private function getMonthlyProcurementStats(): array
    {
        $stats = [];
        
        for ($i = 5; $i >= 0; $i--) {
            $date = now()->subMonths($i);
            $monthKey = $date->format('M Y');
            
            $stats[$monthKey] = [
                'orders_created' => PurchaseOrder::whereMonth('order_date', $date->month)
                    ->whereYear('order_date', $date->year)
                    ->count(),
                'orders_completed' => PurchaseOrder::whereMonth('order_date', $date->month)
                    ->whereYear('order_date', $date->year)
                    ->where('status', 'completed')
                    ->count(),
                'total_value' => PurchaseInvoice::whereMonth('invoice_date', $date->month)
                    ->whereYear('invoice_date', $date->year)
                    ->sum('total_amount'),
            ];
        }
        
        return $stats;
    }

    /**
     * توزيع أوامر الشراء حسب الحالة
     */
    private function getOrderStatusDistribution(): array
    {
        return [
            'draft' => PurchaseOrder::where('status', 'draft')->count(),
            'pending' => PurchaseOrder::where('status', 'pending')->count(),
            'approved' => PurchaseOrder::where('status', 'approved')->count(),
            'sent' => PurchaseOrder::where('status', 'sent')->count(),
            'completed' => PurchaseOrder::where('status', 'completed')->count(),
            'cancelled' => PurchaseOrder::where('status', 'cancelled')->count(),
        ];
    }

    /**
     * الحصول على تاريخ أنشطة أمر الشراء
     */
    private function getOrderTimeline(PurchaseOrder $order): array
    {
        $timeline = [];
        
        // إنشاء الأمر
        $timeline[] = [
            'date' => $order->created_at,
            'event' => 'order_created',
            'title' => 'إنشاء أمر الشراء',
            'description' => "تم إنشاء أمر الشراء {$order->po_number}",
            'user' => $order->user->name ?? 'النظام',
        ];

        // الموافقة
        if ($order->approved_at) {
            $timeline[] = [
                'date' => $order->approved_at,
                'event' => 'order_approved',
                'title' => 'موافقة على الأمر',
                'description' => 'تم الموافقة على أمر الشراء',
                'user' => 'المدير',
            ];
        }

        // استلام البضائع
        foreach ($order->receipts as $receipt) {
            $timeline[] = [
                'date' => $receipt->created_at,
                'event' => 'goods_received',
                'title' => 'استلام بضاعة',
                'description' => "تم استلام جزء من الطلبية - إيصال #{$receipt->receipt_number}",
                'user' => $receipt->user->name ?? 'المستودع',
            ];
        }

        return collect($timeline)->sortBy('date')->values()->toArray();
    }

    /**
     * حساب متوسط الاستهلاك الشهري للمنتج
     */
    private function getAverageMonthlyUsage(int $productId): float
    {
        // هذا مثال - يمكن تحسينه حسب نظام إدارة المخزون
        return 100; // متوسط افتراضي
    }

    /**
     * حساب معدل الامتثال لشروط الدفع
     */
    private function calculatePaymentCompliance(Supplier $supplier): float
    {
        $invoices = $supplier->purchaseInvoices()
            ->where('payment_status', 'paid')
            ->whereNotNull('paid_date')
            ->get();

        if ($invoices->isEmpty()) return 0;

        $onTimePayments = $invoices->filter(function ($invoice) {
            return $invoice->paid_date <= $invoice->due_date;
        })->count();

        return ($onTimePayments / $invoices->count()) * 100;
    }

    /**
     * إنتاج توصيات الشراء الذكية
     */
    private function generatePurchaseRecommendations(): array
    {
        return [
            [
                'type' => 'bulk_discount',
                'title' => 'فرصة خصم كمية',
                'description' => 'يمكن توفير 15% عند شراء أكثر من 1000 وحدة من المنتج أ',
                'potential_savings' => 5000,
            ],
            [
                'type' => 'seasonal_demand',
                'title' => 'طلب موسمي متوقع',
                'description' => 'توقع زيادة في الطلب على منتجات التبريد خلال الشهرين القادمين',
                'recommended_action' => 'زيادة المخزون بنسبة 40%',
            ],
        ];
    }
}