<?php

namespace App\Http\Controllers;

use App\Models\Supplier;
use App\Models\PurchaseOrder;
use App\Models\PurchaseInvoice;
use App\Models\PaymentVoucher;
use App\Models\SupplierDebtSchedule;
use Illuminate\Http\Request;
use Illuminate\Http\Response;
use Illuminate\View\View;
use Maatwebsite\Excel\Facades\Excel;
use Barryvdh\DomPDF\Facade\Pdf;
use Carbon\Carbon;

class ProcurementReportController extends Controller
{
    /**
     * عرض صفحة التقارير الرئيسية
     */
    public function index(): View
    {
        return view('procurement.reports.index');
    }

    /**
     * تصدير التقارير المختلفة
     */
    public function export(Request $request)
    {
        $reportType = $request->input('report_type');
        $format = $request->input('format', 'excel');
        
        switch ($reportType) {
            case 'supplier-performance':
                return $this->exportSupplierPerformance($format);
            case 'debt-analysis':
                return $this->exportDebtAnalysis($format);
            case 'purchase-orders':
                return $this->exportPurchaseOrders($format);
            case 'credit-monitoring':
                return $this->exportCreditMonitoring($format);
            case 'payments-analysis':
                return $this->exportPaymentsAnalysis($format);
            case 'supplier-comparison':
                return $this->exportSupplierComparison($format);
            default:
                return response()->json(['error' => 'نوع التقرير غير مدعوم'], 400);
        }
    }

    /**
     * تصدير تقرير مخصص
     */
    public function customExport(Request $request)
    {
        $validated = $request->validate([
            'report_type' => 'required|string',
            'date_from' => 'required|date',
            'date_to' => 'required|date|after_or_equal:date_from',
            'export_format' => 'required|in:excel,pdf,csv'
        ]);

        $reportType = $validated['report_type'];
        $dateFrom = Carbon::parse($validated['date_from']);
        $dateTo = Carbon::parse($validated['date_to']);
        $format = $validated['export_format'];

        switch ($reportType) {
            case 'supplier_performance':
                return $this->generateSupplierPerformanceReport($dateFrom, $dateTo, $format);
            case 'purchase_analysis':
                return $this->generatePurchaseAnalysisReport($dateFrom, $dateTo, $format);
            case 'payment_analysis':
                return $this->generatePaymentAnalysisReport($dateFrom, $dateTo, $format);
            case 'debt_aging':
                return $this->generateDebtAgingReport($dateFrom, $dateTo, $format);
            case 'credit_utilization':
                return $this->generateCreditUtilizationReport($dateFrom, $dateTo, $format);
            case 'delivery_performance':
                return $this->generateDeliveryPerformanceReport($dateFrom, $dateTo, $format);
            default:
                return response()->json(['error' => 'نوع التقرير غير مدعوم'], 400);
        }
    }

    /**
     * تشغيل التقارير المجدولة
     */
    public function runScheduled(Request $request)
    {
        $reportType = $request->input('report_type');
        
        try {
            switch ($reportType) {
                case 'supplier_performance':
                    $this->generateSupplierPerformanceReport(
                        now()->startOfMonth(),
                        now()->endOfMonth(),
                        'excel'
                    );
                    break;
                case 'debt_analysis':
                    $this->generateDebtAnalysisReport(
                        now()->startOfWeek(),
                        now()->endOfWeek(),
                        'pdf'
                    );
                    break;
                case 'credit_monitoring':
                    $this->generateCreditMonitoringReport(
                        now()->startOfDay(),
                        now()->endOfDay(),
                        'excel'
                    );
                    break;
            }
            
            return response()->json(['success' => true]);
        } catch (\Exception $e) {
            return response()->json(['success' => false, 'error' => $e->getMessage()]);
        }
    }

    /**
     * تصدير تقرير أداء الموردين
     */
    private function exportSupplierPerformance($format = 'excel')
    {
        $suppliers = $this->getSupplierPerformanceData();
        
        if ($format === 'pdf') {
            $pdf = PDF::loadView('procurement.reports.pdf.supplier-performance', compact('suppliers'));
            return $pdf->download('تقرير_أداء_الموردين_' . date('Y-m-d') . '.pdf');
        }
        
        return Excel::download(new SupplierPerformanceExport($suppliers), 
            'تقرير_أداء_الموردين_' . date('Y-m-d') . '.xlsx');
    }

    /**
     * تصدير تقرير المديونيات
     */
    private function exportDebtAnalysis($format = 'excel')
    {
        $debtData = $this->getDebtAnalysisData();
        
        if ($format === 'pdf') {
            $pdf = PDF::loadView('procurement.reports.pdf.debt-analysis', compact('debtData'));
            return $pdf->download('تقرير_المديونيات_' . date('Y-m-d') . '.pdf');
        }
        
        return Excel::download(new DebtAnalysisExport($debtData), 
            'تقرير_المديونيات_' . date('Y-m-d') . '.xlsx');
    }

    /**
     * تصدير تقرير أوامر الشراء
     */
    private function exportPurchaseOrders($format = 'excel')
    {
        $orders = PurchaseOrder::with(['supplier', 'user', 'items'])
            ->latest()
            ->get();
            
        if ($format === 'pdf') {
            $pdf = PDF::loadView('procurement.reports.pdf.purchase-orders', compact('orders'));
            return $pdf->download('تقرير_أوامر_الشراء_' . date('Y-m-d') . '.pdf');
        }
        
        return Excel::download(new PurchaseOrdersExport($orders), 
            'تقرير_أوامر_الشراء_' . date('Y-m-d') . '.xlsx');
    }

    /**
     * تصدير تقرير مراقبة الائتمان
     */
    private function exportCreditMonitoring($format = 'excel')
    {
        $creditData = $this->getCreditMonitoringData();
        
        if ($format === 'pdf') {
            $pdf = PDF::loadView('procurement.reports.pdf.credit-monitoring', compact('creditData'));
            return $pdf->download('تقرير_مراقبة_الائتمان_' . date('Y-m-d') . '.pdf');
        }
        
        return Excel::download(new CreditMonitoringExport($creditData), 
            'تقرير_مراقبة_الائتمان_' . date('Y-m-d') . '.xlsx');
    }

    /**
     * تصدير تقرير المدفوعات
     */
    private function exportPaymentsAnalysis($format = 'excel')
    {
        $paymentsData = $this->getPaymentsAnalysisData();
        
        if ($format === 'pdf') {
            $pdf = PDF::loadView('procurement.reports.pdf.payments-analysis', compact('paymentsData'));
            return $pdf->download('تقرير_المدفوعات_' . date('Y-m-d') . '.pdf');
        }
        
        return Excel::download(new PaymentsAnalysisExport($paymentsData), 
            'تقرير_المدفوعات_' . date('Y-m-d') . '.xlsx');
    }

    /**
     * تصدير تقرير مقارنة الموردين
     */
    private function exportSupplierComparison($format = 'excel')
    {
        $comparisonData = $this->getSupplierComparisonData();
        
        if ($format === 'pdf') {
            $pdf = PDF::loadView('procurement.reports.pdf.supplier-comparison', compact('comparisonData'));
            return $pdf->download('تقرير_مقارنة_الموردين_' . date('Y-m-d') . '.pdf');
        }
        
        return Excel::download(new SupplierComparisonExport($comparisonData), 
            'تقرير_مقارنة_الموردين_' . date('Y-m-d') . '.xlsx');
    }

    /**
     * إنتاج تقرير أداء الموردين حسب الفترة
     */
    private function generateSupplierPerformanceReport(Carbon $dateFrom, Carbon $dateTo, $format)
    {
        $suppliers = Supplier::with(['purchaseOrders', 'purchaseInvoices', 'paymentVouchers'])
            ->get()
            ->map(function ($supplier) use ($dateFrom, $dateTo) {
                return [
                    'supplier' => $supplier,
                    'total_orders' => $supplier->purchaseOrders()
                        ->whereBetween('created_at', [$dateFrom, $dateTo])
                        ->count(),
                    'total_value' => $supplier->purchaseInvoices()
                        ->whereBetween('created_at', [$dateFrom, $dateTo])
                        ->sum('total_amount'),
                    'on_time_delivery' => $this->calculateOnTimeDeliveryRate($supplier, $dateFrom, $dateTo),
                    'payment_compliance' => $this->calculatePaymentCompliance($supplier, $dateFrom, $dateTo),
                    'quality_score' => $this->calculateQualityScore($supplier),
                    'risk_level' => $supplier->risk_level,
                    'performance_score' => $supplier->performance_score
                ];
            });

        $fileName = 'تقرير_أداء_الموردين_' . $dateFrom->format('Y-m-d') . '_' . $dateTo->format('Y-m-d');
        
        if ($format === 'pdf') {
            $pdf = PDF::loadView('procurement.reports.pdf.supplier-performance-custom', 
                compact('suppliers', 'dateFrom', 'dateTo'));
            return $pdf->download($fileName . '.pdf');
        }
        
        return Excel::download(new SupplierPerformanceCustomExport($suppliers, $dateFrom, $dateTo), 
            $fileName . '.xlsx');
    }

    /**
     * إنتاج تقرير تحليل المشتريات
     */
    private function generatePurchaseAnalysisReport(Carbon $dateFrom, Carbon $dateTo, $format)
    {
        $analysis = [
            'total_orders' => PurchaseOrder::whereBetween('created_at', [$dateFrom, $dateTo])->count(),
            'total_value' => PurchaseInvoice::whereBetween('created_at', [$dateFrom, $dateTo])->sum('total_amount'),
            'avg_order_value' => PurchaseInvoice::whereBetween('created_at', [$dateFrom, $dateTo])->avg('total_amount'),
            'by_supplier' => $this->getPurchaseAnalysisBySupplier($dateFrom, $dateTo),
            'by_month' => $this->getPurchaseAnalysisByMonth($dateFrom, $dateTo),
            'top_categories' => $this->getTopPurchaseCategories($dateFrom, $dateTo),
            'status_distribution' => $this->getPurchaseStatusDistribution($dateFrom, $dateTo)
        ];

        $fileName = 'تحليل_المشتريات_' . $dateFrom->format('Y-m-d') . '_' . $dateTo->format('Y-m-d');
        
        if ($format === 'pdf') {
            $pdf = PDF::loadView('procurement.reports.pdf.purchase-analysis', 
                compact('analysis', 'dateFrom', 'dateTo'));
            return $pdf->download($fileName . '.pdf');
        }
        
        return Excel::download(new PurchaseAnalysisExport($analysis, $dateFrom, $dateTo), 
            $fileName . '.xlsx');
    }

    /**
     * إنتاج تقرير أعمار الديون
     */
    private function generateDebtAgingReport(Carbon $dateFrom, Carbon $dateTo, $format)
    {
        $debtAging = Supplier::where('current_balance', '>', 0)
            ->with(['purchaseInvoices' => function($query) {
                $query->where('payment_status', '!=', 'paid')
                      ->where('status', '!=', 'cancelled');
            }])
            ->get()
            ->map(function($supplier) {
                $invoices = $supplier->purchaseInvoices;
                
                return [
                    'supplier' => $supplier,
                    'current' => $invoices->where('is_overdue', false)->sum('remaining_amount'),
                    '1_30_days' => $invoices->where('days_overdue', '>=', 1)
                                          ->where('days_overdue', '<=', 30)
                                          ->sum('remaining_amount'),
                    '31_60_days' => $invoices->where('days_overdue', '>=', 31)
                                           ->where('days_overdue', '<=', 60)
                                           ->sum('remaining_amount'),
                    '61_90_days' => $invoices->where('days_overdue', '>=', 61)
                                           ->where('days_overdue', '<=', 90)
                                           ->sum('remaining_amount'),
                    'over_90_days' => $invoices->where('days_overdue', '>', 90)
                                              ->sum('remaining_amount'),
                    'total_debt' => $supplier->current_balance
                ];
            });

        $fileName = 'تقرير_أعمار_الديون_' . $dateFrom->format('Y-m-d') . '_' . $dateTo->format('Y-m-d');
        
        if ($format === 'pdf') {
            $pdf = PDF::loadView('procurement.reports.pdf.debt-aging', 
                compact('debtAging', 'dateFrom', 'dateTo'));
            return $pdf->download($fileName . '.pdf');
        }
        
        return Excel::download(new DebtAgingExport($debtAging, $dateFrom, $dateTo), 
            $fileName . '.xlsx');
    }

    /**
     * الحصول على بيانات أداء الموردين
     */
    private function getSupplierPerformanceData()
    {
        return Supplier::with(['purchaseOrders', 'purchaseInvoices'])
            ->get()
            ->map(function($supplier) {
                return [
                    'supplier' => $supplier,
                    'performance_metrics' => $supplier->getPerformanceMetrics(),
                    'risk_assessment' => $this->calculateRiskAssessment($supplier),
                    'delivery_performance' => $this->calculateDeliveryPerformance($supplier)
                ];
            });
    }

    /**
     * حساب معدل التسليم في الوقت
     */
    private function calculateOnTimeDeliveryRate($supplier, $dateFrom, $dateTo)
    {
        $completedOrders = $supplier->purchaseOrders()
            ->where('status', 'completed')
            ->whereBetween('created_at', [$dateFrom, $dateTo])
            ->count();

        if ($completedOrders == 0) return 0;

        $onTimeOrders = $supplier->purchaseOrders()
            ->where('status', 'completed')
            ->whereBetween('created_at', [$dateFrom, $dateTo])
            ->whereColumn('actual_delivery_date', '<=', 'expected_delivery_date')
            ->count();

        return round(($onTimeOrders / $completedOrders) * 100, 2);
    }

    /**
     * حساب معدل الامتثال للمدفوعات
     */
    private function calculatePaymentCompliance($supplier, $dateFrom, $dateTo)
    {
        $totalInvoices = $supplier->purchaseInvoices()
            ->whereBetween('created_at', [$dateFrom, $dateTo])
            ->where('status', '!=', 'cancelled')
            ->count();

        if ($totalInvoices == 0) return 100;

        $paidOnTime = $supplier->purchaseInvoices()
            ->whereBetween('created_at', [$dateFrom, $dateTo])
            ->where('payment_status', 'paid')
            ->whereColumn('due_date', '>=', 'updated_at')
            ->count();

        return round(($paidOnTime / $totalInvoices) * 100, 2);
    }

    /**
     * حساب نقاط الجودة
     */
    private function calculateQualityScore($supplier)
    {
        // يمكن تحسين هذا بناء على بيانات فعلية للجودة
        return 92; // نقاط افتراضية
    }

    /**
     * تقييم المخاطر
     */
    private function calculateRiskAssessment($supplier)
    {
        $riskFactors = [];
        
        if ($supplier->exceedsCreditLimit()) {
            $riskFactors[] = 'تجاوز حد الائتمان';
        }
        
        if ($supplier->hasOverdueInvoices()) {
            $riskFactors[] = 'وجود فواتير متأخرة';
        }
        
        if ($supplier->performance_score < 70) {
            $riskFactors[] = 'أداء منخفض';
        }

        return $riskFactors;
    }

    /**
     * حساب أداء التسليم
     */
    private function calculateDeliveryPerformance($supplier)
    {
        return [
            'on_time_rate' => $this->calculateOnTimeDeliveryRate($supplier, now()->startOfYear(), now()),
            'average_delay_days' => $this->calculateAverageDelayDays($supplier),
            'delivery_reliability' => $this->calculateDeliveryReliability($supplier)
        ];
    }

    /**
     * حساب متوسط أيام التأخير
     */
    private function calculateAverageDelayDays($supplier)
    {
        $delayedOrders = $supplier->purchaseOrders()
            ->where('status', 'completed')
            ->whereColumn('actual_delivery_date', '>', 'expected_delivery_date')
            ->get();

        if ($delayedOrders->isEmpty()) return 0;

        $totalDelayDays = $delayedOrders->sum(function($order) {
            return $order->expected_delivery_date->diffInDays($order->actual_delivery_date);
        });

        return round($totalDelayDays / $delayedOrders->count(), 1);
    }

    /**
     * حساب موثوقية التسليم
     */
    private function calculateDeliveryReliability($supplier)
    {
        $totalOrders = $supplier->purchaseOrders()->where('status', 'completed')->count();
        
        if ($totalOrders == 0) return 100;

        $reliableDeliveries = $supplier->purchaseOrders()
            ->where('status', 'completed')
            ->where(function($query) {
                $query->whereColumn('actual_delivery_date', '<=', 'expected_delivery_date')
                      ->orWhere(function($q) {
                          $q->whereColumn('actual_delivery_date', '>', 'expected_delivery_date')
                            ->whereRaw('DATEDIFF(actual_delivery_date, expected_delivery_date) <= 2');
                      });
            })
            ->count();

        return round(($reliableDeliveries / $totalOrders) * 100, 2);
    }

    // المزيد من الدوال المساعدة...
    
    private function getDebtAnalysisData()
    {
        // تطبيق منطق تحليل المديونيات
        return [];
    }
    
    private function getCreditMonitoringData()
    {
        // تطبيق منطق مراقبة الائتمان
        return [];
    }
    
    private function getPaymentsAnalysisData()
    {
        // تطبيق منطق تحليل المدفوعات
        return [];
    }
    
    private function getSupplierComparisonData()
    {
        // تطبيق منطق مقارنة الموردين
        return [];
    }
}