<?php

namespace App\Http\Controllers;

use App\Models\PaymentVoucher;
use App\Models\Company;
use App\Models\Customer;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class PaymentVoucherController extends Controller
{
    /**
     * عرض قائمة سندات الصرف
     */
    public function index(Request $request)
    {
        $query = PaymentVoucher::with(['company', 'supplier', 'user'])
            ->latest('payment_date');

        // فلترة حسب الشركة
        if ($request->filled('company_id')) {
            $query->where('company_id', $request->company_id);
        }

        // فلترة حسب المورد
        if ($request->filled('supplier_id')) {
            $query->where('supplier_id', $request->supplier_id);
        }

        // فلترة حسب الحالة
        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        // فلترة حسب طريقة الدفع
        if ($request->filled('payment_method')) {
            $query->where('payment_method', $request->payment_method);
        }

        // فلترة حسب التاريخ
        if ($request->filled('date_from')) {
            $query->whereDate('payment_date', '>=', $request->date_from);
        }

        if ($request->filled('date_to')) {
            $query->whereDate('payment_date', '<=', $request->date_to);
        }

        // البحث
        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('voucher_number', 'like', "%{$search}%")
                  ->orWhere('reference_number', 'like', "%{$search}%")
                  ->orWhere('description', 'like', "%{$search}%")
                  ->orWhere('supplier_name', 'like', "%{$search}%")
                  ->orWhere('beneficiary_name', 'like', "%{$search}%")
                  ->orWhereHas('supplier', function ($supplierQuery) use ($search) {
                      $supplierQuery->where('name', 'like', "%{$search}%");
                  });
            });
        }

        $vouchers = $query->paginate(15);

        // بيانات إضافية للفلاتر
        $companies = Company::all();
        $suppliers = Customer::all(); // استخدام جدول العملاء كموردين أيضاً

        // إحصائيات
        $stats = [
            'total_amount' => PaymentVoucher::active()->sum('amount'),
            'total_count' => PaymentVoucher::active()->count(),
            'pending_count' => PaymentVoucher::where('status', 'pending')->count(),
            'cancelled_count' => PaymentVoucher::where('status', 'cancelled')->count(),
        ];

        return view('payment-vouchers.index', compact('vouchers', 'companies', 'suppliers', 'stats'));
    }

    /**
     * عرض نموذج إنشاء سند صرف جديد
     */
    public function create()
    {
        $companies = Company::all();
        $suppliers = Customer::all(); // استخدام جدول العملاء كموردين

        return view('payment-vouchers.create', compact('companies', 'suppliers'));
    }

    /**
     * حفظ سند صرف جديد
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'company_id' => 'required|exists:companies,id',
            'supplier_id' => 'nullable|exists:customers,id',
            'supplier_name' => 'required_without:supplier_id|string|max:255',
            'payment_date' => 'required|date',
            'amount' => 'required|numeric|min:0.01',
            'payment_method' => 'required|in:cash,bank_transfer,check,credit_card,online_transfer,other',
            'reference_number' => 'nullable|string|max:255',
            'description' => 'required|string|max:500',
            'notes' => 'nullable|string|max:1000',
            'invoice_reference' => 'nullable|string|max:255',
            'account_code' => 'nullable|string|max:50',
            'status' => 'required|in:active,pending',
            'beneficiary_name' => 'nullable|string|max:255',
            'beneficiary_account' => 'nullable|string|max:255'
        ]);

        $validated['user_id'] = Auth::id();

        try {
            $voucher = PaymentVoucher::create($validated);

            return redirect()->route('payment-vouchers.show', $voucher)
                ->with('success', 'تم إنشاء سند الصرف بنجاح');

        } catch (\Exception $e) {
            return back()->withInput()
                ->with('error', 'حدث خطأ أثناء إنشاء سند الصرف: ' . $e->getMessage());
        }
    }

    /**
     * عرض تفاصيل سند الصرف
     */
    public function show(PaymentVoucher $paymentVoucher)
    {
        $paymentVoucher->load(['company', 'supplier', 'user']);
        
        return view('payment-vouchers.show', compact('paymentVoucher'));
    }

    /**
     * عرض نموذج تحرير سند الصرف
     */
    public function edit(PaymentVoucher $paymentVoucher)
    {
        // لا يمكن تحرير السندات الملغية
        if ($paymentVoucher->status === 'cancelled') {
            return redirect()->route('payment-vouchers.show', $paymentVoucher)
                ->with('error', 'لا يمكن تحرير سند صرف ملغي');
        }

        $companies = Company::all();
        $suppliers = Customer::all();

        return view('payment-vouchers.edit', compact('paymentVoucher', 'companies', 'suppliers'));
    }

    /**
     * تحديث سند الصرف
     */
    public function update(Request $request, PaymentVoucher $paymentVoucher)
    {
        // لا يمكن تحديث السندات الملغية
        if ($paymentVoucher->status === 'cancelled') {
            return redirect()->route('payment-vouchers.show', $paymentVoucher)
                ->with('error', 'لا يمكن تحديث سند صرف ملغي');
        }

        $validated = $request->validate([
            'company_id' => 'required|exists:companies,id',
            'supplier_id' => 'nullable|exists:customers,id',
            'supplier_name' => 'required_without:supplier_id|string|max:255',
            'payment_date' => 'required|date',
            'amount' => 'required|numeric|min:0.01',
            'payment_method' => 'required|in:cash,bank_transfer,check,credit_card,online_transfer,other',
            'reference_number' => 'nullable|string|max:255',
            'description' => 'required|string|max:500',
            'notes' => 'nullable|string|max:1000',
            'invoice_reference' => 'nullable|string|max:255',
            'account_code' => 'nullable|string|max:50',
            'status' => 'required|in:active,pending',
            'beneficiary_name' => 'nullable|string|max:255',
            'beneficiary_account' => 'nullable|string|max:255'
        ]);

        try {
            $paymentVoucher->update($validated);

            return redirect()->route('payment-vouchers.show', $paymentVoucher)
                ->with('success', 'تم تحديث سند الصرف بنجاح');

        } catch (\Exception $e) {
            return back()->withInput()
                ->with('error', 'حدث خطأ أثناء تحديث سند الصرف: ' . $e->getMessage());
        }
    }

    /**
     * إلغاء سند الصرف
     */
    public function destroy(PaymentVoucher $paymentVoucher)
    {
        try {
            // تحديث حالة السند إلى ملغي بدلاً من الحذف
            $paymentVoucher->update(['status' => 'cancelled']);

            return redirect()->route('payment-vouchers.index')
                ->with('success', 'تم إلغاء سند الصرف بنجاح');

        } catch (\Exception $e) {
            return back()->with('error', 'حدث خطأ أثناء إلغاء سند الصرف: ' . $e->getMessage());
        }
    }

    /**
     * طباعة سند الصرف
     */
    public function print(PaymentVoucher $paymentVoucher)
    {
        $paymentVoucher->load(['company', 'supplier', 'user']);
        
        return view('payment-vouchers.print', compact('paymentVoucher'));
    }

    /**
     * تصدير تقرير سندات الصرف
     */
    public function export(Request $request)
    {
        $query = PaymentVoucher::with(['company', 'supplier', 'user'])
            ->latest('payment_date');

        // تطبيق الفلاتر
        if ($request->filled('company_id')) {
            $query->where('company_id', $request->company_id);
        }

        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        if ($request->filled('date_from')) {
            $query->whereDate('payment_date', '>=', $request->date_from);
        }

        if ($request->filled('date_to')) {
            $query->whereDate('payment_date', '<=', $request->date_to);
        }

        $vouchers = $query->get();

        $filename = 'payment_vouchers_' . now()->format('Y-m-d_H-i-s') . '.csv';

        $headers = [
            'Content-Type' => 'text/csv',
            'Content-Disposition' => "attachment; filename={$filename}",
        ];

        $callback = function() use ($vouchers) {
            $file = fopen('php://output', 'w');
            
            // العناوين
            fputcsv($file, [
                'رقم السند',
                'الشركة',
                'المورد/المستفيد',
                'التاريخ',
                'المبلغ',
                'طريقة الدفع',
                'الرقم المرجعي',
                'الوصف',
                'الحالة',
                'المستخدم'
            ]);

            // البيانات
            foreach ($vouchers as $voucher) {
                fputcsv($file, [
                    $voucher->voucher_number,
                    $voucher->company->name,
                    $voucher->supplier ? $voucher->supplier->name : $voucher->supplier_name,
                    $voucher->payment_date->format('Y-m-d'),
                    $voucher->amount,
                    $voucher->getPaymentMethodText(),
                    $voucher->reference_number,
                    $voucher->description,
                    $voucher->getStatusText(),
                    $voucher->user->name
                ]);
            }

            fclose($file);
        };

        return response()->stream($callback, 200, $headers);
    }

    /**
     * تقرير ملخص سندات الصرف
     */
    public function summary(Request $request)
    {
        $dateFrom = $request->input('date_from', now()->startOfMonth());
        $dateTo = $request->input('date_to', now()->endOfMonth());

        $query = PaymentVoucher::whereBetween('payment_date', [$dateFrom, $dateTo]);

        if ($request->filled('company_id')) {
            $query->where('company_id', $request->company_id);
        }

        // ملخص حسب طريقة الدفع
        $paymentMethodSummary = $query->clone()
            ->groupBy('payment_method')
            ->selectRaw('payment_method, SUM(amount) as total_amount, COUNT(*) as count')
            ->get();

        // ملخص حسب الحالة
        $statusSummary = $query->clone()
            ->groupBy('status')
            ->selectRaw('status, SUM(amount) as total_amount, COUNT(*) as count')
            ->get();

        // ملخص يومي
        $dailySummary = $query->clone()
            ->groupBy(DB::raw('DATE(payment_date)'))
            ->selectRaw('DATE(payment_date) as date, SUM(amount) as total_amount, COUNT(*) as count')
            ->orderBy('date')
            ->get();

        $companies = Company::all();

        return view('payment-vouchers.summary', compact(
            'paymentMethodSummary',
            'statusSummary',
            'dailySummary',
            'companies',
            'dateFrom',
            'dateTo'
        ));
    }
}