<?php

namespace App\Http\Controllers;

use App\Models\JournalEntry;
use App\Models\JournalEntryLine;
use App\Models\Account;
use App\Models\Customer;
use App\Models\Supplier;
use App\Models\CostCenter;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Auth;

class JournalEntryController extends Controller
{
    public function index(Request $request)
    {
        $query = JournalEntry::with(['creator', 'poster']);

        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        if ($request->filled('type')) {
            $query->where('type', $request->type);
        }

        if ($request->filled('start_date') && $request->filled('end_date')) {
            $query->whereBetween('entry_date', [$request->start_date, $request->end_date]);
        }

        if ($request->filled('search')) {
            $query->where(function ($q) use ($request) {
                $q->where('entry_number', 'like', '%' . $request->search . '%')
                  ->orWhere('description', 'like', '%' . $request->search . '%')
                  ->orWhere('reference', 'like', '%' . $request->search . '%');
            });
        }

        $journalEntries = $query->orderBy('entry_date', 'desc')
            ->orderBy('id', 'desc')
            ->paginate(20);

        return view('accounting.journal-entries.index', compact('journalEntries'));
    }

    public function create()
    {
        $accounts = Account::where('is_active', true)->orderBy('code')->get();
        $customers = Customer::where('is_active', true)->orderBy('name')->get();
        $suppliers = Supplier::where('is_active', true)->orderBy('name')->get();
        $costCenters = CostCenter::where('is_active', true)->orderBy('code')->get();
        $entryNumber = JournalEntry::generateEntryNumber();

        return view('accounting.journal-entries.create', compact('accounts', 'customers', 'suppliers', 'costCenters', 'entryNumber'));
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'entry_date' => 'required|date',
            'reference' => 'nullable|string|max:255',
            'description' => 'required|string',
            'type' => 'required|in:manual,auto,adjustment,closing',
            'lines' => 'required|array|min:2',
            'lines.*.account_id' => 'required|exists:accounts,id',
            'lines.*.description' => 'nullable|string',
            'lines.*.debit_amount' => 'nullable|numeric|min:0',
            'lines.*.credit_amount' => 'nullable|numeric|min:0',
            'lines.*.reference' => 'nullable|string|max:255',
            'lines.*.customer_id' => 'nullable|exists:customers,id',
            'lines.*.supplier_id' => 'nullable|exists:suppliers,id',
            'lines.*.cost_center' => 'nullable|string|max:255',
        ]);

        // Validate that each line has either debit or credit amount
        foreach ($validated['lines'] as $index => $line) {
            $debit = $line['debit_amount'] ?? 0;
            $credit = $line['credit_amount'] ?? 0;
            
            if ($debit == 0 && $credit == 0) {
                return back()->withErrors(["lines.{$index}" => 'يجب إدخال مبلغ مدين أو دائن']);
            }
            
            if ($debit > 0 && $credit > 0) {
                return back()->withErrors(["lines.{$index}" => 'لا يمكن إدخال مبلغ مدين ودائن في نفس السطر']);
            }
        }

        // Calculate totals
        $totalDebit = collect($validated['lines'])->sum('debit_amount');
        $totalCredit = collect($validated['lines'])->sum('credit_amount');

        if (abs($totalDebit - $totalCredit) > 0.01) {
            return back()->withErrors(['lines' => 'إجمالي المدين يجب أن يساوي إجمالي الدائن']);
        }

        DB::beginTransaction();
        try {
            $journalEntry = JournalEntry::create([
                'entry_number' => JournalEntry::generateEntryNumber(),
                'entry_date' => $validated['entry_date'],
                'reference' => $validated['reference'],
                'description' => $validated['description'],
                'type' => $validated['type'],
                'status' => 'draft',
                'total_debit' => $totalDebit,
                'total_credit' => $totalCredit,
                'created_by' => Auth::id()
            ]);

            foreach ($validated['lines'] as $lineData) {
                $journalEntry->lines()->create([
                    'account_id' => $lineData['account_id'],
                    'description' => $lineData['description'],
                    'debit_amount' => $lineData['debit_amount'] ?? 0,
                    'credit_amount' => $lineData['credit_amount'] ?? 0,
                    'reference' => $lineData['reference'],
                    'customer_id' => $lineData['customer_id'],
                    'supplier_id' => $lineData['supplier_id'],
                    'cost_center' => $lineData['cost_center'],
                ]);
            }

            DB::commit();

            return redirect()->route('journal-entries.show', $journalEntry)
                ->with('success', 'تم إنشاء القيد بنجاح');
        } catch (\Exception $e) {
            DB::rollback();
            return back()->withErrors(['error' => 'حدث خطأ أثناء إنشاء القيد']);
        }
    }

    public function show(JournalEntry $journalEntry)
    {
        $journalEntry->load(['lines.account', 'lines.customer', 'lines.supplier', 'creator', 'poster', 'reverser']);
        
        return view('accounting.journal-entries.show', compact('journalEntry'));
    }

    public function edit(JournalEntry $journalEntry)
    {
        if ($journalEntry->status !== 'draft') {
            return back()->withErrors(['error' => 'لا يمكن تعديل قيد محاسبي مرحل']);
        }

        $journalEntry->load('lines');
        $accounts = Account::where('is_active', true)->orderBy('code')->get();
        $customers = Customer::where('is_active', true)->orderBy('name')->get();
        $suppliers = Supplier::where('is_active', true)->orderBy('name')->get();
        $costCenters = CostCenter::where('is_active', true)->orderBy('code')->get();

        return view('accounting.journal-entries.edit', compact('journalEntry', 'accounts', 'customers', 'suppliers', 'costCenters'));
    }

    public function update(Request $request, JournalEntry $journalEntry)
    {
        if ($journalEntry->status !== 'draft') {
            return back()->withErrors(['error' => 'لا يمكن تعديل قيد محاسبي مرحل']);
        }

        $validated = $request->validate([
            'entry_date' => 'required|date',
            'reference' => 'nullable|string|max:255',
            'description' => 'required|string',
            'lines' => 'required|array|min:2',
            'lines.*.account_id' => 'required|exists:accounts,id',
            'lines.*.description' => 'nullable|string',
            'lines.*.debit_amount' => 'nullable|numeric|min:0',
            'lines.*.credit_amount' => 'nullable|numeric|min:0',
            'lines.*.reference' => 'nullable|string|max:255',
            'lines.*.customer_id' => 'nullable|exists:customers,id',
            'lines.*.supplier_id' => 'nullable|exists:suppliers,id',
            'lines.*.cost_center' => 'nullable|string|max:255',
        ]);

        // Calculate totals
        $totalDebit = collect($validated['lines'])->sum('debit_amount');
        $totalCredit = collect($validated['lines'])->sum('credit_amount');

        if (abs($totalDebit - $totalCredit) > 0.01) {
            return back()->withErrors(['lines' => 'إجمالي المدين يجب أن يساوي إجمالي الدائن']);
        }

        DB::beginTransaction();
        try {
            $journalEntry->update([
                'entry_date' => $validated['entry_date'],
                'reference' => $validated['reference'],
                'description' => $validated['description'],
                'total_debit' => $totalDebit,
                'total_credit' => $totalCredit,
            ]);

            // Delete existing lines and create new ones
            $journalEntry->lines()->delete();

            foreach ($validated['lines'] as $lineData) {
                $journalEntry->lines()->create([
                    'account_id' => $lineData['account_id'],
                    'description' => $lineData['description'],
                    'debit_amount' => $lineData['debit_amount'] ?? 0,
                    'credit_amount' => $lineData['credit_amount'] ?? 0,
                    'reference' => $lineData['reference'],
                    'customer_id' => $lineData['customer_id'],
                    'supplier_id' => $lineData['supplier_id'],
                    'cost_center' => $lineData['cost_center'],
                ]);
            }

            DB::commit();

            return redirect()->route('journal-entries.show', $journalEntry)
                ->with('success', 'تم تحديث القيد بنجاح');
        } catch (\Exception $e) {
            DB::rollback();
            return back()->withErrors(['error' => 'حدث خطأ أثناء تحديث القيد']);
        }
    }

    public function post(JournalEntry $journalEntry)
    {
        try {
            $journalEntry->post(Auth::id());
            
            return back()->with('success', 'تم ترحيل القيد بنجاح');
        } catch (\Exception $e) {
            return back()->withErrors(['error' => $e->getMessage()]);
        }
    }

    public function reverse(Request $request, JournalEntry $journalEntry)
    {
        $request->validate([
            'reversal_reason' => 'required|string'
        ]);

        try {
            $journalEntry->reverse(Auth::id(), $request->reversal_reason);
            
            return back()->with('success', 'تم عكس القيد بنجاح');
        } catch (\Exception $e) {
            return back()->withErrors(['error' => $e->getMessage()]);
        }
    }

    public function destroy(JournalEntry $journalEntry)
    {
        if ($journalEntry->status !== 'draft') {
            return back()->withErrors(['error' => 'لا يمكن حذف قيد محاسبي مرحل']);
        }

        $journalEntry->delete();

        return redirect()->route('journal-entries.index')
            ->with('success', 'تم حذف القيد بنجاح');
    }

    public function print(JournalEntry $journalEntry)
    {
        $journalEntry->load(['lines.account', 'creator']);
        
        return view('accounting.journal-entries.print', compact('journalEntry'));
    }
}