<?php

namespace App\Http\Controllers;

use App\Services\ZatcaOperationsService;
use Illuminate\Http\Request;
use App\Models\Invoice;
use App\Models\Customer;
use App\Models\Product;
use App\Models\Company;
use App\Models\Account;
use App\Models\InvoiceItem;
use App\Models\ZatcaSettings;
use App\Models\Category;
use App\Services\ZatcaService;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Mail;
use Barryvdh\DomPDF\Facade\Pdf;
use function Symfony\Component\String\length;

class InvoiceController extends Controller
{
    /**
     * عرض قائمة الفواتير
     */
    public function index(Request $request)
    {
        $query = Invoice::with(['customer', 'company', 'user']);

        // فلترة حسب النوع
        if ($request->filled('type')) {
            $query->where('type', $request->type);
        }

        // فلترة حسب الحالة
        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        // فلترة حسب التاريخ
        if ($request->filled('date_from')) {
            $query->whereDate('issue_date', '>=', $request->date_from);
        }

        if ($request->filled('date_to')) {
            $query->whereDate('issue_date', '<=', $request->date_to);
        }

        // البحث
        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('invoice_number', 'like', "%{$search}%")
                    ->orWhereHas('customer', function ($customerQuery) use ($search) {
                        $customerQuery->where('name', 'like', "%{$search}%");
                    });
            });
        }

        $invoices = $query->orderBy('created_at', 'desc')->paginate(15);

        return view('invoices.index', compact('invoices'));
    }

    /**
     * عرض نموذج إنشاء فاتورة جديدة
     */
    public function create()
    {
        $customers = Customer::active()->get();
        $products = Product::active()->with('category')->get();
        $company = Company::first();
        $accounts = Account::withTax()->orderBy('name')->get();

        // جلب الفئات من جدول categories
        $categories = Category::active()
            ->ordered()
            ->get()
            ->map(function ($category) {
                return [
                    'value' => $category->id,
                    'name' => $category->name,
                    'icon' => $category->icon ?: 'fas fa-folder'
                ];
            });

        if (!$company) {
            return redirect()->route('companies.create')
                ->with('error', 'يجب إنشاء بيانات الشركة أولاً');
        }

        return view('invoices.create', compact('customers', 'products', 'company', 'accounts', 'categories'));
    }

    /**
     * حفظ فاتورة جديدة
     */
    public function store(Request $request)
    {
        // Debug logging
        \Log::info('Invoice store method started', [
            'request_data' => $request->except(['_token']), // إخفاء التوكن من اللوج
            'user_id' => auth()->id(),
            'timestamp' => now(),
            'items_count' => count($request->items ?? [])
        ]);

        // التحقق من وجود عناصر في الطلب
        if (empty($request->items) || count($request->items) == 0) {
            return back()->withInput()
                ->with('error', 'يجب إضافة عنصر واحد على الأقل للفاتورة');
        }

        $request->validate([
            'type' => 'required|in:standard,simplified',
            'customer_id' => 'nullable|exists:customers,id',
            'issue_date' => 'required|date',
            'due_date' => 'nullable|date|after:issue_date',
            'items' => 'required|array|min:1',
            'items.*.product_id' => 'required|exists:products,id',
            'items.*.account_id' => 'nullable|exists:accounts,id',
            'items.*.quantity' => 'required|numeric|min:0.001',
            'items.*.unit_price' => 'required|numeric|min:0',
            'items.*.discount_amount' => 'nullable|numeric|min:0',
            'payment_method' => 'nullable|in:cash,bank_transfer,check,credit_card,installment',
            'paid_amount' => 'nullable|numeric|min:0',
            'payment_date' => 'nullable|date',
            'payment_notes' => 'nullable|string|max:1000',
            // POS specific fields
            'order_type' => 'nullable|in:dine_in,takeaway,delivery',
            'table_number' => 'nullable|integer|min:1',
            'customer_phone' => 'nullable|string|max:20',
            'customer_name' => 'nullable|string|max:255',
            'create_customer' => 'nullable|boolean',
            'new_customer_name' => 'nullable|string|max:255',
            'new_customer_phone' => 'nullable|string|max:20',
            'new_customer_type' => 'nullable|in:individual,business',
        ]);

        \Log::info('Validation passed, starting database transaction');

        DB::beginTransaction();
        try {
            $company = Company::first();
            \Log::info('Company found', ['company_id' => $company?->id]);
            
            // Handle customer creation for POS
            $customerId = $request->customer_id;
            if ($request->create_customer && $request->new_customer_name) {
                $customer = Customer::create([
                    'name' => $request->new_customer_name,
                    'phone' => $request->new_customer_phone,
                    'customer_type' => $request->new_customer_type ?? 'individual',
                    'is_active' => true,
                ]);
                $customerId = $customer->id;
            }

            $invoice = new Invoice();
            $invoice->invoice_number = $this->generateInvoiceNumber($request->type);
            $invoice->type = $request->type;
            $invoice->company_id = $company->id;
            $invoice->customer_id = $customerId;
            $invoice->user_id = auth()->id();
            $invoice->issue_date = $request->issue_date;
            $invoice->due_date = $request->due_date;
            $invoice->supply_date = $request->supply_date ?? $request->issue_date;
            $invoice->currency = 'SAR';
            $invoice->discount_amount = $request->discount_amount ?? 0;
            $invoice->notes = $request->notes;
            $invoice->uuid = Str::uuid();

            $invoice->payment_method = $request->payment_method;
            $invoice->paid_amount = $request->paid_amount ?? 0;
            $invoice->payment_date = $request->payment_date;
            $invoice->payment_notes = $request->payment_notes;

            $invoice->subtotal = 0;
            $invoice->tax_amount = 0;
            $invoice->total_amount = 0;

            $zatcaSettings = ZatcaSettings::where('company_id', $company->id)->first();
            $invoice->invoice_counter = $zatcaSettings ? $zatcaSettings->incrementInvoiceCounter() : 1;

            \Log::info('About to save invoice', [
                'invoice_number' => $invoice->invoice_number,
                'customer_id' => $invoice->customer_id,
                'company_id' => $invoice->company_id
            ]);

            $invoice->save();
            
            \Log::info('Invoice saved successfully', ['invoice_id' => $invoice->id]);

            $subtotal = 0;
            $totalTax = 0;

            foreach ($request->items as $itemData) {
                $product = Product::find($itemData['product_id']);

                $item = new InvoiceItem();
                $item->invoice_id = $invoice->id;
                $item->product_id = $product->id;
                $item->account_id = $itemData['account_id'] ?? $product->account_id ?? null;
                $item->name = $product->name;
                $item->name_en = $product->name_en;
                $item->description = $product->description;
                $item->sku = $product->sku;
                $item->quantity = $itemData['quantity'];
                $item->unit = $product->unit;
                $item->unit_price = $itemData['unit_price'];
                $item->discount_amount = $itemData['discount_amount'] ?? 0;
                $item->tax_rate = $product->tax_rate;
                $item->zatca_category_code = $product->zatca_category_code;
                $item->is_taxable = $product->is_taxable;

                $item->save();

                $subtotal += $item->line_total;
                $totalTax += $item->tax_amount;

                if ($product->type === 'product') {
                    $product->decrement('stock_quantity', $item->quantity);
                }
            }

            $invoice->subtotal = $subtotal;
            $invoice->tax_amount = $totalTax;
            $invoice->total_amount = $subtotal + $totalTax - $invoice->discount_amount;

            $invoice->remaining_amount = $invoice->total_amount - $invoice->paid_amount;
            $invoice->updatePaymentStatus();

            $invoice->save();

            // توليد QR Code
            try {
                $invoice->saveQrCode();
            } catch (\Exception $e) {
                \Log::warning('فشل في توليد QR Code للفاتورة: ' . $invoice->id . ' - ' . $e->getMessage());
            }

            // إرسال إشعارات داخلية أو بريد
            $this->handleNotifications($request, $invoice);

            \Log::info('About to commit transaction');
            DB::commit();
            \Log::info('Transaction committed successfully');

            $message = 'تم إنشاء الفاتورة بنجاح';
            
            // التحقق من نوع العملية المطلوبة
            if ($request->action === 'save_and_send') {
                return redirect()->route('invoices.sendToZatca', $invoice)
                    ->with('success', $message . ' وسيتم إرسالها إلى زاتكا');
            } elseif ($request->action === 'save_and_print' || $request->has('print_invoice')) {
                // إعادة توجيه لصفحة الطباعة مع الطباعة التلقائية
                return redirect()->route('invoices.print', ['invoice' => $invoice, 'auto_print' => 1])
                    ->with('success', $message . ' - جاري تحضير الطباعة');
            }

            // التحقق إذا كان الطلب من نقطة البيع (restaurant POS)
            if ($request->has('order_type') || $request->header('Referer') && strpos($request->header('Referer'), 'invoices/create') !== false) {
                \Log::info('POS request detected, redirecting to create page with success data', [
                    'invoice_id' => $invoice->id,
                    'invoice_number' => $invoice->invoice_number,
                    'total_amount' => $invoice->total_amount,
                    'items_count' => $invoice->items->count(),
                    'customer_created' => $customerId && $request->create_customer ? 'yes' : 'no'
                ]);
                
                // إعداد رسالة نجاح مفصلة
                $successMessage = $message;
                if ($customerId && $request->create_customer) {
                    $successMessage .= ' وتم إنشاء عميل جديد';
                }
                
                // إعادة التوجيه إلى نقطة البيع مع بيانات الفاتورة المحفوظة
                return redirect()->route('invoices.create')
                    ->with('success', $successMessage)
                    ->with('invoice_saved', true)
                    ->with('invoice_data', [
                        'id' => $invoice->id,
                        'invoice_number' => $invoice->invoice_number,
                        'total_amount' => $invoice->total_amount,
                        'paid_amount' => $invoice->paid_amount,
                        'payment_status' => $invoice->payment_status,
                        'items_count' => $invoice->items->count(),
                        'customer_name' => $invoice->customer ? $invoice->customer->name : 'غير محدد'
                    ]);
            }

            return redirect()->route('invoices.show', $invoice)
                ->with('success', $message);
        } catch (\Exception $e) {
            \Log::error('Invoice creation failed', [
                'error' => $e->getMessage(),
                'file' => $e->getFile(),
                'line' => $e->getLine(),
                'trace' => $e->getTraceAsString(),
                'request_data' => $request->except(['_token']),
                'user_id' => auth()->id()
            ]);
            
            DB::rollback();
            
            // رسالة خطأ مفصلة حسب نوع الخطأ
            $errorMessage = 'حدث خطأ أثناء إنشاء الفاتورة';
            
            if (strpos($e->getMessage(), 'Duplicate entry') !== false) {
                $errorMessage = 'خطأ: رقم الفاتورة مكرر. يرجى المحاولة مرة أخرى.';
            } elseif (strpos($e->getMessage(), 'foreign key constraint') !== false) {
                $errorMessage = 'خطأ في البيانات المرتبطة. تأكد من صحة المنتجات والعميل المحدد.';
            } elseif (strpos($e->getMessage(), 'Connection') !== false) {
                $errorMessage = 'خطأ في الاتصال بقاعدة البيانات. يرجى المحاولة لاحقاً.';
            }
            
            // إضافة معلومات إضافية للمطور في بيئة التطوير
            if (config('app.debug')) {
                $errorMessage .= ' (تفاصيل الخطأ: ' . $e->getMessage() . ')';
            }
            
            return back()->withInput()
                ->with('error', $errorMessage)
                ->with('debug_info', config('app.debug') ? [
                    'file' => $e->getFile(),
                    'line' => $e->getLine(),
                    'message' => $e->getMessage()
                ] : null);
        }
    }



    /**
     * عرض تفاصيل الفاتورة
     */
    public function show(Invoice $invoice)
    {
        $invoice->load(['customer', 'company', 'user', 'items.product']);

        return view('invoices.show', compact('invoice'));
    }

    /**
     * عرض نموذج تعديل الفاتورة
     */
    public function edit(Invoice $invoice)
    {
        if ($invoice->status !== 'draft') {
            return redirect()->route('invoices.show', $invoice)
                ->with('error', 'لا يمكن تعديل الفاتورة بعد إرسالها');
        }

        $customers = Customer::active()->get();
        $products = Product::active()->get();
        $accounts = Account::withTax()->orderBy('name')->get();
        $invoice->load(['items.product', 'items.account']);

        return view('invoices.edit', compact('invoice', 'customers', 'products', 'accounts'));
    }

    /**
     * تحديث الفاتورة
     */
    public function update(Request $request, Invoice $invoice)
    {
        if ($invoice->status !== 'draft') {
            return redirect()->route('invoices.show', $invoice)
                ->with('error', 'لا يمكن تعديل الفاتورة بعد إرسالها');
        }

        $request->validate([
            'customer_id' => 'nullable|exists:customers,id',
            'issue_date' => 'required|date',
            'due_date' => 'nullable|date|after:issue_date',
            'items' => 'required|array|min:1',
            'items.*.product_id' => 'required|exists:products,id',
            'items.*.quantity' => 'required|numeric|min:0.001',
            'items.*.unit_price' => 'required|numeric|min:0',
            'items.*.discount_amount' => 'nullable|numeric|min:0',
        ]);

        DB::beginTransaction();
        try {
            // إعادة المخزون للمنتجات القديمة
            foreach ($invoice->items as $oldItem) {
                if ($oldItem->product && $oldItem->product->type === 'product') {
                    $oldItem->product->increment('stock_quantity', $oldItem->quantity);
                }
            }

            // حذف العناصر القديمة
            $invoice->items()->delete();

            // تحديث بيانات الفاتورة
            $invoice->customer_id = $request->customer_id;
            $invoice->issue_date = $request->issue_date;
            $invoice->due_date = $request->due_date;
            $invoice->supply_date = $request->supply_date ?? $request->issue_date;
            $invoice->discount_amount = $request->discount_amount ?? 0;
            $invoice->notes = $request->notes;

            $subtotal = 0;
            $totalTax = 0;

            foreach ($request->items as $itemData) {
                $product = Product::find($itemData['product_id']);

                $item = new InvoiceItem();
                $item->invoice_id = $invoice->id;
                $item->product_id = $product->id;
                $item->account_id = $itemData['account_id'] ?? $product->account_id ?? null;
                $item->name = $product->name;
                $item->name_en = $product->name_en;
                $item->description = $product->description;
                $item->sku = $product->sku;
                $item->quantity = $itemData['quantity'];
                $item->unit = $product->unit;
                $item->unit_price = $itemData['unit_price'];
                $item->discount_amount = $itemData['discount_amount'] ?? 0;
                $item->tax_rate = $product->tax_rate;
                $item->zatca_category_code = $product->zatca_category_code;
                $item->is_taxable = $product->is_taxable;

                $item->save();

                $subtotal += $item->line_total;
                $totalTax += $item->tax_amount;

                if ($product->type === 'product') {
                    $product->decrement('stock_quantity', $item->quantity);
                }
            }

            $invoice->subtotal = $subtotal;
            $invoice->tax_amount = $totalTax;
            $invoice->total_amount = $subtotal + $totalTax - $invoice->discount_amount;

            // تحديث المبلغ المتبقي وحالة الدفع (اختياري حسب منطق المشروع)
            $invoice->remaining_amount = $invoice->total_amount - $invoice->paid_amount;
            $invoice->updatePaymentStatus();

            $invoice->save();
            // إعادة توليد QR Code بعد التحديث
            try {
                $invoice->saveQrCode();
            } catch (\Exception $e) {
                \Log::warning('فشل توليد QR Code عند التحديث: ' . $invoice->id . ' - ' . $e->getMessage());
            }

            DB::commit();

            return redirect()->route('invoices.show', $invoice)
                ->with('success', 'تم تحديث الفاتورة بنجاح');
        } catch (\Exception $e) {
            DB::rollback();
            return back()->withInput()
                ->with('error', 'حدث خطأ أثناء تحديث الفاتورة: ' . $e->getMessage());
        }
    }

    /**
     * حذف الفاتورة
     */
    public function destroy(Invoice $invoice)
    {
        if ($invoice->status !== 'draft') {
            return redirect()->route('invoices.index')
                ->with('error', 'لا يمكن حذف الفاتورة بعد إرسالها');
        }

        DB::beginTransaction();
        try {
            // إرجاع المخزون
            foreach ($invoice->items as $item) {
                if ($item->product && $item->product->type === 'product') {
                    $item->product->increment('stock_quantity', $item->quantity);
                }
            }

            $invoice->delete();
            DB::commit();

            return redirect()->route('invoices.index')
                ->with('success', 'تم حذف الفاتورة بنجاح');
        } catch (\Exception $e) {
            DB::rollback();
            return redirect()->route('invoices.index')
                ->with('error', 'حدث خطأ أثناء حذف الفاتورة');
        }
    }

    /**
     * إرسال الفاتورة إلى ZATCA
     */
    public function sendToZatca(Invoice $invoice)
    {
        // جلب إعدادات ZATCA المرتبطة بالشركة
        $zatcaSettings = ZatcaSettings::where('company_id', $invoice->company_id)
            ->where('is_active', true)
            ->first();

        if (!$zatcaSettings) {
            return back()->with('error', 'لم يتم إعداد اتصال ZATCA لهذه الشركة.');
        }

        try {
            $zatcaService = new ZatcaService($zatcaSettings);

            $items = [];
            foreach ($invoice->items as $item){
                $items[] = [
                    'name' => $item->name,
                    'quantity' => $item->quantity,
                    'price' => $item->unit_price,
                    'vat' => $item->tax_amount
                ];
            }
            $invoiceObj = [
                'invoice_number' => $invoice->invoice_number,
                'total_amount' => $invoice->total_amount,
                'vat_amount' => $invoice->tax_amount,
                'is_pos' => $invoice->customer->vat_number <> 15,
                'is_invoice' => true,
                'buyer_name' => $invoice->customer->name,
                'buyer_address' => $invoice->customer->address,
                'buyer_tax_number' => $invoice->customer->vat_number,
                'buyer_city' => $invoice->customer->city,
                'buyer_state' => $invoice->customer->district,
                'buyer_postal' => $invoice->customer->postal_code,
                'buyer_building_no' => $invoice->customer->unit_number,
                'items' => $items
            ];

            $zatcaOperation = new ZatcaOperationsService();
            $result = $zatcaOperation->prepareInvoiceData($invoiceObj,$invoice->company);

            $invoice->invoice_hash = $result['invoice_hash'];
            $invoice->qr_code = $result['QR'];
            $invoice->xml_file_path = $result['xml_file_path'];
            $invoice->zatca_response = $result['response'];
            $invoice->status = $result['status'];
            $invoice->uuid = $result['UUID'];
            $invoice->sent_to_zatca_at = date('Y-m-d H:i:s');
            $invoice->save();
            // 1. توليد ملف XML الموقع (XAdES-BES)
//            $xmlContent = $zatcaService->generateInvoiceXml($invoice);
//
//            // 2. حساب Hash الفاتورة الحالية
//            $invoiceHash = $zatcaService->calculateInvoiceHash($xmlContent);
//            $invoice->invoice_hash = $invoiceHash;
//
//            // 3. تعيين Hash الفاتورة السابقة (لربط التسلسل)
//            $invoice->previous_invoice_hash = $zatcaSettings->previous_invoice_hash;
//
//            // 4. توليد QR Code للمرحلة الأولى (قبل الإرسال)
//            $invoice->qr_code = $zatcaService->generateQRCode($invoice);
//            $invoice->save();
//
//            // 5. إرسال الفاتورة إلى ZATCA (Clearance أو Reporting)
//            $response = $invoice->isStandard()
//                ? $zatcaService->clearInvoice($invoice, $xmlContent)
//                : $zatcaService->reportInvoice($invoice, $xmlContent);
//
//
//
//            // 6. حفظ XML النهائي
//            $xmlPath = 'invoices/xml/' . $invoice->invoice_number . '.xml';
//            if (!Storage::put($xmlPath, $xmlContent)) {
//                \Log::error("فشل في حفظ ملف XML للفاتورة رقم {$invoice->invoice_number}");
//            }
//
//            $invoice->xml_file_path = $xmlPath;
//
//            // 7. تحديث Hash الفاتورة السابقة في الإعدادات
//            $zatcaSettings->updatePreviousInvoiceHash($invoiceHash);
//
//            // 8. حفظ التحديثات النهائية
//            $invoice->save();
//
//            // 9. إعادة توليد QR Code للمرحلة الثانية (بعد الإرسال)
//            try {
//                $invoice = $invoice->fresh(); // تحميل آخر نسخة من قاعدة البيانات
//                $invoice->updateQrCodeForPhase2();
//
//                \Log::info('تم تحديث QR Code للمرحلة الثانية بنجاح.', [
//                    'invoice_id' => $invoice->id,
//                    'invoice_number' => $invoice->invoice_number,
//                ]);
//            } catch (\Exception $qrEx) {
//                \Log::warning('فشل تحديث QR Code للمرحلة الثانية: ' . $qrEx->getMessage(), [
//                    'invoice_id' => $invoice->id,
//                ]);
//            }

            return back()->with('success', 'تم إرسال الفاتورة إلى ZATCA بنجاح وتحديث QR Code للمرحلة الثانية.');
        } catch (\Exception $e) {
            \Log::error('فشل إرسال الفاتورة إلى ZATCA: ' . $e->getMessage(), [
                'invoice_id' => $invoice->id ?? null,
                'trace' => $e->getTraceAsString(),
            ]);

            return back()->with('error', 'فشل في إرسال الفاتورة إلى ZATCA: ' . $e->getMessage());
        }
    }


    /**
     * تحميل PDF للفاتورة
     */
    public function downloadPdf(Invoice $invoice)
    {
        $invoice->load(['customer', 'company', 'items.product']);

        $pdf = Pdf::loadView('invoices.pdf', compact('invoice'))
            ->setPaper('a4', 'portrait')
            ->setOptions([
                'defaultFont' => 'DejaVu Sans',
                'isHtml5ParserEnabled' => true,
                'isPhpEnabled' => false,
                'isRemoteEnabled' => false,
                'fontSubsetting' => true,
                'dpi' => 150,
                'defaultMediaType' => 'print',
                'isFontSubsettingEnabled' => true,
            ]);

        return $pdf->download($invoice->invoice_number . '.pdf');
    }

    /**
     * تحميل XML للفاتورة
     */
    public function downloadXml(Invoice $invoice)
    {
        if (!$invoice->xml_file_path || !Storage::exists($invoice->xml_file_path)) {
            // توليد XML إذا لم يكن موجود
            $zatcaSettings = ZatcaSettings::where('company_id', $invoice->company_id)->first();
            if ($zatcaSettings) {
                $zatcaService = new ZatcaService($zatcaSettings);
                $xmlContent = $zatcaService->generateInvoiceXml($invoice);

                $xmlPath = 'invoices/xml/' . $invoice->invoice_number . '.xml';
                Storage::put($xmlPath, $xmlContent);
                $invoice->xml_file_path = $xmlPath;
                $invoice->save();
            }
        }

        if ($invoice->xml_file_path && Storage::exists($invoice->xml_file_path)) {
            return Storage::download($invoice->xml_file_path, $invoice->invoice_number . '.xml');
        }

        return back()->with('error', 'ملف XML غير متوفر');
    }

    /**
     * تحميل XML UBL-Invoice-2.1 للفاتورة متضمن امتدادات ZATCA
     */
    public function downloadUblXml(Invoice $invoice)
    {
        try {
            $zatcaSettings = ZatcaSettings::where('company_id', $invoice->company_id)->first();
            if (!$zatcaSettings) {
                return back()->with('error', 'لم يتم إعداد اتصال ZATCA للشركة');
            }

            $zatcaService = new ZatcaService($zatcaSettings);
            $xmlContent = $zatcaService->generateInvoiceXml($invoice);

            $filename = $invoice->invoice_number . '_UBL-2.1.xml';

            return response($xmlContent)
                ->header('Content-Type', 'application/xml; charset=utf-8')
                ->header('Content-Disposition', 'attachment; filename="' . $filename . '"')
                ->header('Cache-Control', 'no-cache, no-store, must-revalidate')
                ->header('Pragma', 'no-cache')
                ->header('Expires', '0');
        } catch (\Exception $e) {
            return back()->with('error', 'حدث خطأ أثناء توليد ملف XML UBL: ' . $e->getMessage());
        }
    }

    /**
     * البحث عن منتج (AJAX)
     */
    public function getProduct(Product $product)
    {
        return response()->json($product);
    }

    /**
     * البحث عن العملاء (AJAX)
     */
    public function searchCustomers(Request $request)
    {
        $search = $request->get('search');

        $customers = Customer::active()
            ->where('name', 'like', "%{$search}%")
            ->limit(10)
            ->get(['id', 'name', 'customer_type', 'vat_number']);

        return response()->json($customers);
    }

    /**
     * توليد رقم الفاتورة
     */
    private function generateInvoiceNumber($type)
    {
        $prefix = $type === 'standard' ? 'INV' : 'SIM';
        $year = now()->year;
        $month = now()->format('m');

        $count = Invoice::where('type', $type)
            ->whereYear('created_at', $year)
            ->whereMonth('created_at', $month)
            ->count() + 1;

        return sprintf('%s-%d%s-%06d', $prefix, $year, $month, $count);
    }

    /**
     * إرسال الفاتورة عبر الواتساب
     */
    public function sendToWhatsApp(Invoice $invoice)
    {
        try {
            $invoice->load(['customer', 'company']);

            if (!$invoice->customer || !$invoice->customer->phone) {
                return back()->with('error', 'لا يوجد رقم هاتف للعميل');
            }

            $phone = $this->formatPhoneNumber($invoice->customer->phone);
            $message = $this->generateWhatsAppMessage($invoice);

            // إنشاء رابط الواتساب
            $whatsappUrl = "https://wa.me/{$phone}?text=" . urlencode($message);

            return redirect($whatsappUrl);
        } catch (\Exception $e) {
            return back()->with('error', 'حدث خطأ أثناء إرسال الرسالة: ' . $e->getMessage());
        }
    }

    /**
     * إرسال الفاتورة عبر البريد الإلكتروني
     */
    public function sendEmail(Invoice $invoice)
    {
        try {
            $invoice->load(['customer', 'company', 'items.product']);

            if (!$invoice->customer || !$invoice->customer->email) {
                return back()->with('error', 'لا يوجد بريد إلكتروني للعميل');
            }

            // إنشاء PDF
            $pdf = Pdf::loadView('invoices.pdf', compact('invoice'))
                ->setPaper('a4', 'portrait')
                ->setOptions([
                    'defaultFont' => 'DejaVu Sans',
                    'isHtml5ParserEnabled' => true,
                    'isPhpEnabled' => false,
                    'isRemoteEnabled' => false,
                    'fontSubsetting' => true,
                    'dpi' => 150,
                    'defaultMediaType' => 'print',
                    'isFontSubsettingEnabled' => true,
                ]);

            $pdfContent = $pdf->output();

            // إرسال البريد
            Mail::send('emails.invoice', compact('invoice'), function ($message) use ($invoice, $pdfContent) {
                $message->to($invoice->customer->email, $invoice->customer->name)
                    ->subject('فاتورة رقم ' . $invoice->invoice_number)
                    ->attachData($pdfContent, $invoice->invoice_number . '.pdf', [
                        'mime' => 'application/pdf',
                    ]);
            });

            return back()->with('success', 'تم إرسال الفاتورة بنجاح إلى ' . $invoice->customer->email);
        } catch (\Exception $e) {
            return back()->with('error', 'حدث خطأ أثناء إرسال البريد: ' . $e->getMessage());
        }
    }

    /**
     * صفحة الطباعة
     */
    public function printPage(Invoice $invoice)
    {
        $invoice->load(['customer', 'company', 'items.product']);

        // توليد QR Code إذا لم يكن موجود
        if (!$invoice->qr_code) {
            try {
                $invoice->saveQrCode();
            } catch (\Exception $e) {
                \Log::warning('فشل في توليد QR Code للطباعة: ' . $invoice->id . ' - ' . $e->getMessage());
            }
        }

        return view('invoices.print', compact('invoice'));
    }

    /**
     * عرض صفحة الطباعة الحرارية للفاتورة
     * Thermal printing view for invoice - optimized for 58mm/80mm thermal printers
     */
    public function thermalPrintPage(Invoice $invoice)
    {
        $invoice->load(['customer', 'company', 'items.product']);

        // توليد QR Code إذا لم يكن موجود
        if (!$invoice->qr_code) {
            try {
                $invoice->saveQrCode();
            } catch (\Exception $e) {
                \Log::warning('فشل في توليد QR Code للطباعة الحرارية: ' . $invoice->id . ' - ' . $e->getMessage());
            }
        }

        // تحديث تاريخ آخر طباعة
        $invoice->last_printed_at = now();
        $invoice->save();

        return view('invoices.thermal-print', compact('invoice'));
    }

    /**
     * تنسيق رقم الهاتف للواتساب
     */
    private function formatPhoneNumber($phone)
    {
        // إزالة جميع الرموز غير الرقمية
        $phone = preg_replace('/[^0-9]/', '', $phone);

        // إضافة كود السعودية إذا لم يكن موجود
        if (!str_starts_with($phone, '966')) {
            if (str_starts_with($phone, '0')) {
                $phone = '966' . substr($phone, 1);
            } else {
                $phone = '966' . $phone;
            }
        }

        return $phone;
    }

    /**
     * إنشاء رسالة الواتساب
     */
    private function generateWhatsAppMessage(Invoice $invoice)
    {
        $message = "السلام عليكم ورحمة الله وبركاته\n\n";
        $message .= "تحية طيبة من شركة " . $invoice->company->name . "\n\n";
        $message .= "نرفق لكم فاتورة رقم: " . $invoice->invoice_number . "\n";
        $message .= "تاريخ الإصدار: " . $invoice->issue_date->format('Y-m-d') . "\n";
        $message .= "المبلغ الإجمالي: " . number_format($invoice->total_amount, 2) . " ر.س\n\n";

        if ($invoice->due_date) {
            $message .= "تاريخ الاستحقاق: " . $invoice->due_date->format('Y-m-d') . "\n\n";
        }

        $message .= "لتحميل الفاتورة بصيغة PDF، يرجى الضغط على الرابط التالي:\n";
        $message .= route('invoices.pdf', $invoice) . "\n\n";
        $message .= "شكراً لتعاملكم معنا";

        return $message;
    }

    /**
     * توليد QR Code للفاتورة
     */
    public function generateQr(Invoice $invoice)
    {
        try {
            // توليد QR Code وحفظه
            $invoice->saveQrCode();

            return back()->with('success', 'تم توليد رمز QR بنجاح');
        } catch (\Exception $e) {
            return back()->with('error', 'حدث خطأ أثناء توليد رمز QR: ' . $e->getMessage());
        }
    }

    /**
     * التحقق من الفاتورة عبر QR Code (مسار عام)
     */
    public function verify($uuid)
    {
        try {
            $invoice = Invoice::where('uuid', $uuid)
                ->with(['customer', 'company', 'items'])
                ->first();

            if (!$invoice) {
                abort(404, 'الفاتورة غير موجودة');
            }

            return view('invoices.verify', compact('invoice'));
        } catch (\Exception $e) {
            abort(404, 'الفاتورة غير موجودة');
        }
    }

    /**
     * عرض نافذة مزامنة الفواتير مع ZATCA
     */
    public function syncWindow(Request $request)
    {
        // الحصول على الفواتير المرسلة لـ ZATCA
        $sentInvoices = Invoice::whereNotNull('sent_to_zatca_at')
            ->with(['customer', 'company'])
            ->orderBy('sent_to_zatca_at', 'desc')
            ->paginate(20);

        // الحصول على إحصائيات سريعة
        $stats = [
            'total_sent' => Invoice::whereNotNull('sent_to_zatca_at')->count(),
            'cleared' => Invoice::where('status', 'cleared')->count(),
            'reported' => Invoice::where('status', 'reported')->count(),
            'rejected' => Invoice::where('status', 'rejected')->count(),
            'pending_sync' => Invoice::whereNotNull('sent_to_zatca_at')
                ->where('updated_at', '>', now()->subHours(24))
                ->count()
        ];

        // إذا كان طلب AJAX، أرجع البيانات فقط
        if ($request->ajax() || $request->get('ajax') == '1') {
            return response()->json([
                'stats' => [
                    'total_sent' => $stats['total_sent'],
                    'cleared_reported' => $stats['cleared'] + $stats['reported'],
                    'rejected' => $stats['rejected'],
                    'pending_sync' => $stats['pending_sync']
                ]
            ]);
        }

        return view('invoices.sync', compact('sentInvoices', 'stats'));
    }

    /**
     * مزامنة فاتورة واحدة مع ZATCA
     */
    public function syncSingle(Invoice $invoice)
    {
        $zatcaSettings = ZatcaSettings::where('company_id', $invoice->company_id)
            ->where('is_active', true)
            ->first();

        if (!$zatcaSettings) {
            return back()->with('error', 'لم يتم إعداد اتصال ZATCA للشركة');
        }

        try {
            $zatcaService = new ZatcaService($zatcaSettings);
            $result = $zatcaService->checkInvoiceStatus($invoice);

            if ($result['success']) {
                $message = 'تم تحديث حالة الفاتورة من ZATCA بنجاح';
                if (isset($result['data']['status'])) {
                    $message .= ' - الحالة: ' . $this->getStatusInArabic($result['data']['status']);
                }

                return back()->with('success', $message)
                    ->with('sync_result', $result['data'] ?? []);
            } else {
                // معالجة أنواع الأخطاء المختلفة
                switch ($result['error']) {
                    case 'NOT_SENT_TO_ZATCA':
                        return back()->with('warning', $result['message'])
                            ->with('sync_action', 'يمكنك إرسال الفاتورة إلى ZATCA أولاً');

                    case 'INVOICE_NOT_FOUND':
                        $message = $result['message'];
                        if (isset($result['details'])) {
                            $message .= ' - ' . $result['details'];
                        }
                        return back()->with('error', $message)
                            ->with('sync_action', 'قد تحتاج لإعادة إرسال الفاتورة إلى ZATCA');

                    case 'UUID_MISSING':
                        return back()->with('error', $result['message'])
                            ->with('sync_action', 'يرجى تحديث UUID الفاتورة من الإعدادات');

                    case 'UNAUTHORIZED':
                        $settingsUrl = route('zatca-settings.edit', $zatcaSettings->id);
                        return back()->with('error', $result['message'])
                            ->with('sync_action', "تحقق من بيانات الاعتماد في <a href='{$settingsUrl}' target='_blank'>إعدادات ZATCA</a>");

                    case 'RATE_LIMITED':
                        return back()->with('warning', $result['message'])
                            ->with('sync_action', 'يرجى المحاولة بعد بضع دقائق');

                    case 'SERVER_ERROR':
                        return back()->with('error', $result['message'])
                            ->with('sync_action', 'مشكلة مؤقتة في خوادم ZATCA، يرجى المحاولة لاحقاً');

                    default:
                        return back()->with('error', $result['message'])
                            ->with('sync_details', $result);
                }
            }
        } catch (\Exception $e) {
            \Log::error('Unexpected error in syncSingle', [
                'invoice_id' => $invoice->id,
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);

            return back()->with('error', 'حدث خطأ غير متوقع: ' . $e->getMessage());
        }
    }

    /**
     * مزامنة جميع الفواتير مع ZATCA
     */
    public function syncAll(Request $request)
    {
        try {
            $companies = Company::whereHas('zatcaSettings', function ($query) {
                $query->where('is_active', true);
            })->get();

            $results = [
                'success' => 0,
                'failed' => 0,
                'not_sent' => 0,
                'not_found' => 0,
                'details' => []
            ];

            foreach ($companies as $company) {
                $zatcaSettings = $company->zatcaSettings()->where('is_active', true)->first();
                if (!$zatcaSettings) continue;

                $invoices = Invoice::where('company_id', $company->id)
                    ->whereNotNull('sent_to_zatca_at')
                    ->where('updated_at', '>', now()->subDays(7)) // آخر أسبوع فقط
                    ->get();

                try {
                    $zatcaService = new ZatcaService($zatcaSettings);
                    $companyResults = $zatcaService->syncMultipleInvoices($invoices);

                    $results['success'] += $companyResults['success'];
                    $results['failed'] += $companyResults['failed'];
                    $results['not_sent'] += $companyResults['not_sent'] ?? 0;
                    $results['not_found'] += $companyResults['not_found'] ?? 0;

                    $results['details'][$company->name] = [
                        'success' => $companyResults['success'],
                        'failed' => $companyResults['failed'],
                        'not_sent' => $companyResults['not_sent'] ?? 0,
                        'not_found' => $companyResults['not_found'] ?? 0,
                        'total_processed' => $invoices->count(),
                        'details' => $companyResults['details'] ?? []
                    ];
                } catch (\Exception $e) {
                    $results['failed'] += $invoices->count();
                    $results['details'][$company->name] = [
                        'success' => 0,
                        'failed' => $invoices->count(),
                        'not_sent' => 0,
                        'not_found' => 0,
                        'error' => $e->getMessage(),
                        'total_processed' => $invoices->count()
                    ];
                }
            }

            // إنشاء رسالة مفصلة
            $message = "تمت المزامنة - ";
            $messageParts = [];

            if ($results['success'] > 0) {
                $messageParts[] = "نجح: {$results['success']}";
            }
            if ($results['failed'] > 0) {
                $messageParts[] = "فشل: {$results['failed']}";
            }
            if ($results['not_found'] > 0) {
                $messageParts[] = "غير موجود في ZATCA: {$results['not_found']}";
            }
            if ($results['not_sent'] > 0) {
                $messageParts[] = "لم يتم إرساله: {$results['not_sent']}";
            }

            $message .= implode(', ', $messageParts);

            // إذا كان طلب AJAX، أرجع JSON
            if ($request->expectsJson() || $request->ajax()) {
                return response()->json([
                    'success' => true,
                    'message' => $message,
                    'results' => $results
                ]);
            }

            $alertType = $results['failed'] > 0 || $results['not_found'] > 0 ? 'warning' : 'success';

            return back()->with($alertType, $message)
                ->with('sync_results', $results);
        } catch (\Exception $e) {
            if ($request->expectsJson() || $request->ajax()) {
                return response()->json([
                    'success' => false,
                    'message' => 'حدث خطأ أثناء المزامنة: ' . $e->getMessage()
                ], 500);
            }

            return back()->with('error', 'حدث خطأ أثناء المزامنة: ' . $e->getMessage());
        }
    }

    /**
     * تصدير تقرير مزامنة ZATCA
     */
    public function exportSyncReport()
    {
        $invoices = Invoice::whereNotNull('sent_to_zatca_at')
            ->with(['customer', 'company'])
            ->orderBy('sent_to_zatca_at', 'desc')
            ->get();

        $csvData = [];
        $csvData[] = [
            'رقم الفاتورة',
            'نوع الفاتورة',
            'الشركة',
            'العميل',
            'المبلغ الإجمالي',
            'تاريخ الإرسال',
            'الحالة',
            'رقم الموافقة (IRN)',
            'استجابة ZATCA'
        ];

        foreach ($invoices as $invoice) {
            $csvData[] = [
                $invoice->invoice_number,
                $invoice->type === 'standard' ? 'معيارية' : 'مبسطة',
                $invoice->company->name ?? '',
                $invoice->customer->name ?? 'غير محدد',
                number_format($invoice->total_amount, 2),
                $invoice->sent_to_zatca_at ? $invoice->sent_to_zatca_at->format('Y-m-d H:i:s') : '',
                $this->getStatusInArabic($invoice->status),
                $invoice->irn ?? '',
                $invoice->zatca_response ? json_encode($invoice->zatca_response, JSON_UNESCAPED_UNICODE) : ''
            ];
        }

        $filename = 'zatca_sync_report_' . now()->format('Y-m-d_H-i-s') . '.csv';

        $callback = function () use ($csvData) {
            $file = fopen('php://output', 'w');

            // إضافة BOM للـ UTF-8
            fprintf($file, chr(0xEF) . chr(0xBB) . chr(0xBF));

            foreach ($csvData as $row) {
                fputcsv($file, $row);
            }
            fclose($file);
        };

        return response()->stream($callback, 200, [
            'Content-Type' => 'text/csv; charset=UTF-8',
            'Content-Disposition' => 'attachment; filename="' . $filename . '"',
        ]);
    }

    /**
     * ترجمة حالة الفاتورة للعربية
     */
    private function getStatusInArabic($status)
    {
        $statuses = [
            'draft' => 'مسودة',
            'sent' => 'مرسلة',
            'cleared' => 'مؤكدة',
            'reported' => 'مبلغة',
            'rejected' => 'مرفوضة',
            'not_found_in_zatca' => 'غير موجودة في ZATCA'
        ];

        return $statuses[$status] ?? $status;
    }

    /**
     * اختبار XML المحسن مع عرض التفاصيل والأخطاء
     */
    public function testImprovedXml(Invoice $invoice)
    {
        try {
            $zatcaSettings = ZatcaSettings::where('company_id', $invoice->company_id)->first();
            if (!$zatcaSettings) {
                return back()->with('error', 'لم يتم إعداد اتصال ZATCA للشركة');
            }

            $zatcaService = new ZatcaService($zatcaSettings);

            // توليد XML المحسن
            $xmlContent = $zatcaService->generateInvoiceXml($invoice);

            // التحقق من صحة XML
            $validationResults = $this->validateZatcaXml($xmlContent);

            // عرض تفاصيل شاملة
            $testResults = [
                'invoice' => $invoice,
                'xml_content' => $xmlContent,
                'xml_size' => strlen($xmlContent),
                'validation' => $validationResults,
                'zatca_compliance' => $this->checkZatcaCompliance($invoice),
                'qr_code_info' => $invoice->getQrCodeInfo(),
                'company_address_complete' => $this->checkCompanyAddressCompliance($invoice->company),
                'xml_structure' => $this->analyzeXmlStructure($xmlContent)
            ];

            // إذا كان طلب AJAX، أرجع JSON
            if (request()->expectsJson() || request()->ajax()) {
                return response()->json($testResults);
            }

            // عرض صفحة تفصيلية للاختبار
            return view('invoices.test-xml', $testResults);
        } catch (\Exception $e) {
            $error = [
                'error' => $e->getMessage(),
                'file' => $e->getFile(),
                'line' => $e->getLine(),
                'trace' => $e->getTraceAsString()
            ];

            if (request()->expectsJson() || request()->ajax()) {
                return response()->json($error, 500);
            }

            return back()->with('error', 'حدث خطأ أثناء اختبار XML: ' . $e->getMessage())
                ->with('debug_info', $error);
        }
    }

    /**
     * التحقق من صحة XML حسب معايير ZATCA
     */
    private function validateZatcaXml($xmlContent)
    {
        $results = [
            'is_valid' => true,
            'errors' => [],
            'warnings' => [],
            'checks' => []
        ];

        try {
            // تحميل XML
            $dom = new \DOMDocument('1.0', 'UTF-8');
            $dom->formatOutput = true;
            $dom->loadXML($xmlContent);

            // التحقق من البنية الأساسية
            $results['checks']['basic_structure'] = $this->checkBasicXmlStructure($dom);

            // التحقق من العناصر المطلوبة
            $results['checks']['required_elements'] = $this->checkRequiredElements($dom);

            // التحقق من التوزيع التفصيلي للضريبة (BG-23)
            $results['checks']['vat_breakdown'] = $this->checkVatBreakdown($dom);

            // التحقق من QR Code
            $results['checks']['qr_code'] = $this->checkQrCodeCompliance($dom);

            // التحقق من رمز الفاتورة (KSA-2)
            $results['checks']['invoice_code'] = $this->checkInvoiceCode($dom);

            // التحقق من عداد الفاتورة (KSA-16)
            $results['checks']['invoice_counter'] = $this->checkInvoiceCounter($dom);

            // جمع الأخطاء
            foreach ($results['checks'] as $check) {
                if (isset($check['errors'])) {
                    $results['errors'] = array_merge($results['errors'], $check['errors']);
                }
                if (isset($check['warnings'])) {
                    $results['warnings'] = array_merge($results['warnings'], $check['warnings']);
                }
            }

            $results['is_valid'] = empty($results['errors']);
        } catch (\Exception $e) {
            $results['is_valid'] = false;
            $results['errors'][] = 'خطأ في تحليل XML: ' . $e->getMessage();
        }

        return $results;
    }

    /**
     * التحقق من البنية الأساسية للـ XML
     */
    private function checkBasicXmlStructure($dom)
    {
        $errors = [];
        $warnings = [];

        // التحقق من العنصر الجذر
        if ($dom->documentElement->tagName !== 'Invoice') {
            $errors[] = 'العنصر الجذر يجب أن يكون Invoice';
        }

        // التحقق من Namespaces المطلوبة
        $requiredNamespaces = [
            '' => 'urn:oasis:names:specification:ubl:schema:xsd:Invoice-2',
            'cac' => 'urn:oasis:names:specification:ubl:schema:xsd:CommonAggregateComponents-2',
            'cbc' => 'urn:oasis:names:specification:ubl:schema:xsd:CommonBasicComponents-2',
            'ext' => 'urn:oasis:names:specification:ubl:schema:xsd:CommonExtensionComponents-2'
        ];

        foreach ($requiredNamespaces as $prefix => $uri) {
            $attrName = $prefix ? "xmlns:$prefix" : 'xmlns';
            if (
                !$dom->documentElement->hasAttribute($attrName) ||
                $dom->documentElement->getAttribute($attrName) !== $uri
            ) {
                $errors[] = "Namespace مفقود أو خاطئ: $attrName";
            }
        }

        return [
            'passed' => empty($errors),
            'errors' => $errors,
            'warnings' => $warnings
        ];
    }

    /**
     * التحقق من العناصر المطلوبة
     */
    private function checkRequiredElements($dom)
    {
        $errors = [];
        $warnings = [];

        $requiredElements = [
            'cbc:ID' => 'رقم الفاتورة مطلوب',
            'cbc:UUID' => 'UUID مطلوب',
            'cbc:IssueDate' => 'تاريخ الإصدار مطلوب',
            'cbc:InvoiceTypeCode' => 'نوع الفاتورة مطلوب',
            'cbc:BusinessProcessTypeCode' => 'نوع العملية التجارية مطلوب (BR-KSA-EN16931-01)',
            'cbc:DocumentCurrencyCode' => 'العملة مطلوبة',
            'cac:AccountingSupplierParty' => 'معلومات البائع مطلوبة',
            'cac:LegalMonetaryTotal' => 'الإجماليات النقدية مطلوبة'
        ];

        foreach ($requiredElements as $element => $message) {
            $nodes = $dom->getElementsByTagName(str_replace('cbc:', '', str_replace('cac:', '', $element)));
            if ($nodes->length === 0) {
                $errors[] = $message;
            }
        }

        // التحقق من InstructionID (عداد الفاتورة)
        $instructionId = $dom->getElementsByTagName('InstructionID');
        if ($instructionId->length === 0) {
            $errors[] = 'عداد الفاتورة (InstructionID) مطلوب (BR-KSA-33)';
        } elseif (!is_numeric($instructionId->item(0)->nodeValue)) {
            $errors[] = 'عداد الفاتورة يجب أن يحتوي على أرقام فقط (BR-KSA-34)';
        }

        return [
            'passed' => empty($errors),
            'errors' => $errors,
            'warnings' => $warnings
        ];
    }

    /**
     * التحقق من التوزيع التفصيلي للضريبة (BG-23)
     */
    private function checkVatBreakdown($dom)
    {
        $errors = [];
        $warnings = [];

        $taxTotal = $dom->getElementsByTagName('TaxTotal');
        if ($taxTotal->length === 0) {
            $errors[] = 'إجمالي الضريبة مطلوب (BR-CO-18)';
            return [
                'passed' => false,
                'errors' => $errors,
                'warnings' => $warnings
            ];
        }

        $taxSubtotal = $dom->getElementsByTagName('TaxSubtotal');
        if ($taxSubtotal->length === 0) {
            $errors[] = 'مجموعة التوزيع التفصيلي لضريبة القيمة المضافة مطلوبة (BG-23)';
        }

        return [
            'passed' => empty($errors),
            'errors' => $errors,
            'warnings' => $warnings
        ];
    }

    /**
     * التحقق من QR Code
     */
    private function checkQrCodeCompliance($dom)
    {
        $errors = [];
        $warnings = [];

        $qrElements = $dom->getElementsByTagName('EmbeddedDocumentBinaryObject');
        $qrFound = false;

        foreach ($qrElements as $element) {
            if ($element->getAttribute('filename') === 'QR_CODE') {
                $qrFound = true;
                $qrData = $element->nodeValue;

                if (strlen($qrData) > 1000) {
                    $errors[] = 'طول QR Code يتجاوز الحد المسموح 1000 حرف (BR-CL-KSA-14)';
                }
                break;
            }
        }

        if (!$qrFound) {
            $warnings[] = 'QR Code غير موجود';
        }

        return [
            'passed' => empty($errors),
            'errors' => $errors,
            'warnings' => $warnings
        ];
    }

    /**
     * التحقق من رمز الفاتورة (KSA-2)
     */
    private function checkInvoiceCode($dom)
    {
        $errors = [];
        $warnings = [];

        $notes = $dom->getElementsByTagName('Note');
        $codeFound = false;

        foreach ($notes as $note) {
            $code = $note->nodeValue;
            if (preg_match('/^\d{7}$/', $code)) {
                $codeFound = true;
                // التحقق من البنية NNPNESB
                if (!preg_match('/^(01|02)[01]{5}$/', $code)) {
                    $errors[] = 'رمز الفاتورة لا يتبع الهيكل المطلوب NNPNESB (BR-KSA-06)';
                }
                break;
            }
        }

        if (!$codeFound) {
            $errors[] = 'رمز الفاتورة (KSA-2) مطلوب (BR-KSA-06)';
        }

        return [
            'passed' => empty($errors),
            'errors' => $errors,
            'warnings' => $warnings
        ];
    }

    /**
     * التحقق من عداد الفاتورة
     */
    private function checkInvoiceCounter($dom)
    {
        $errors = [];
        $warnings = [];

        $instructionId = $dom->getElementsByTagName('InstructionID');
        if ($instructionId->length === 0) {
            $errors[] = 'عداد الفاتورة (InstructionID) مطلوب (BR-KSA-33)';
        } else {
            $counter = $instructionId->item(0)->nodeValue;
            if (!is_numeric($counter) || intval($counter) <= 0) {
                $errors[] = 'عداد الفاتورة يجب أن يكون رقم صحيح موجب (BR-KSA-34)';
            }
        }

        return [
            'passed' => empty($errors),
            'errors' => $errors,
            'warnings' => $warnings
        ];
    }

    /**
     * فحص امتثال ZATCA
     */
    private function checkZatcaCompliance($invoice)
    {
        $checks = [];

        // فحص البيانات الأساسية
        $checks['basic_data'] = [
            'invoice_number' => !empty($invoice->invoice_number),
            'uuid' => !empty($invoice->uuid),
            'issue_date' => !empty($invoice->issue_date),
            'currency' => !empty($invoice->currency)
        ];

        // فحص الشركة
        $checks['company'] = [
            'name' => !empty($invoice->company->name),
            'vat_number' => !empty($invoice->company->vat_number),
            'address_complete' => $this->isAddressComplete($invoice->company)
        ];

        // فحص العناصر
        $checks['items'] = [
            'has_items' => $invoice->items->count() > 0,
            'all_have_vat_classification' => $invoice->items->every(function ($item) {
                return isset($item->is_taxable);
            })
        ];

        return $checks;
    }

    /**
     * فحص اكتمال العنوان
     */
    private function isAddressComplete($company)
    {
        return !empty($company->city) &&
            !empty($company->district) &&
            !empty($company->street) &&
            !empty($company->building_number) &&
            !empty($company->postal_code);
    }

    /**
     * فحص امتثال عنوان الشركة
     */
    private function checkCompanyAddressCompliance($company)
    {
        $requirements = [
            'street' => !empty($company->street),
            'building_number' => !empty($company->building_number),
            'city' => !empty($company->city),
            'postal_code' => !empty($company->postal_code),
            'district' => !empty($company->district),
            'building_number_format' => !empty($company->building_number) &&
                strlen($company->building_number) === 4 &&
                is_numeric($company->building_number)
        ];

        return [
            'compliant' => !in_array(false, $requirements),
            'requirements' => $requirements,
            'missing' => array_keys(array_filter($requirements, function ($val) {
                return !$val;
            }))
        ];
    }

    /**
     * تحليل هيكل XML
     */
    private function analyzeXmlStructure($xmlContent)
    {
        $structure = [
            'size_bytes' => strlen($xmlContent),
            'size_kb' => round(strlen($xmlContent) / 1024, 2),
            'lines' => substr_count($xmlContent, "\n"),
            'elements_count' => substr_count($xmlContent, '<'),
            'has_signature' => strpos($xmlContent, 'ds:Signature') !== false,
            'has_extensions' => strpos($xmlContent, 'ext:UBLExtensions') !== false,
            'encoding' => 'UTF-8'
        ];

        return $structure;
    }

    /**
     * معالجة إرسال الإشعارات
     */
    private function handleNotifications(Request $request, Invoice $invoice)
    {
        try {
            // إرسال إشعار للدائن
            if ($request->has('send_creditor_notification')) {
                $invoice->sendCreditorNotification();
            }

            // إرسال إشعار للمدين (العميل)
            if ($request->has('send_debtor_notification') && $invoice->customer) {
                $invoice->sendDebtorNotification();

                // إرسال إيميل للعميل إذا كان متوفراً
                if ($invoice->customer->email) {
                    try {
                        $this->sendEmailNotification($invoice);
                    } catch (\Exception $e) {
                        \Log::warning('فشل في إرسال إيميل للعميل: ' . $e->getMessage());
                    }
                }
            }
        } catch (\Exception $e) {
            \Log::warning('فشل في إرسال الإشعارات: ' . $e->getMessage());
        }
    }

    /**
     * إرسال إشعار بالإيميل للعميل
     */
    private function sendEmailNotification(Invoice $invoice)
    {
        $invoice->load(['customer', 'company', 'items.product']);

        // إنشاء PDF
        $pdf = Pdf::loadView('invoices.pdf', compact('invoice'))
            ->setPaper('a4', 'portrait')
            ->setOptions([
                'defaultFont' => 'DejaVu Sans',
                'isHtml5ParserEnabled' => true,
                'isPhpEnabled' => false,
                'isRemoteEnabled' => false,
            ]);

        $pdfContent = $pdf->output();

        // إرسال البريد
        Mail::send('emails.invoice', compact('invoice'), function ($message) use ($invoice, $pdfContent) {
            $message->to($invoice->customer->email, $invoice->customer->name)
                ->subject('فاتورة جديدة رقم ' . $invoice->invoice_number)
                ->attachData($pdfContent, $invoice->invoice_number . '.pdf', [
                    'mime' => 'application/pdf',
                ]);
        });
    }

    /**
     * إضافة دفعة جديدة للفاتورة
     */
    public function addPayment(Request $request, Invoice $invoice)
    {
        $request->validate([
            'amount' => 'required|numeric|min:0.01',
            'method' => 'nullable|in:cash,bank_transfer,check,credit_card,installment',
            'notes' => 'nullable|string|max:1000',
            'payment_date' => 'nullable|date'
        ]);

        try {
            $amount = $request->amount;

            // التحقق من عدم تجاوز المبلغ الإجمالي
            if ($invoice->paid_amount + $amount > $invoice->total_amount) {
                if ($request->expectsJson() || $request->ajax()) {
                    return response()->json([
                        'success' => false,
                        'message' => 'المبلغ المدفوع لا يمكن أن يتجاوز إجمالي الفاتورة'
                    ], 400);
                }
                return back()->with('error', 'المبلغ المدفوع لا يمكن أن يتجاوز إجمالي الفاتورة');
            }

            $invoice->addPayment(
                $amount,
                $request->method,
                $request->notes
            );

            if ($request->payment_date) {
                $invoice->payment_date = $request->payment_date;
                $invoice->save();
            }

            $message = 'تم إضافة الدفعة بنجاح. ';
            if ($invoice->isPaid()) {
                $message .= 'الفاتورة مدفوعة بالكامل الآن.';
            } else {
                $remaining = $invoice->total_amount - $invoice->paid_amount;
                $message .= 'المبلغ المتبقي: ' . number_format($remaining, 2) . ' ر.س';
            }

            // إذا كان الطلب AJAX، إرجاع JSON
            if ($request->expectsJson() || $request->ajax()) {
                return response()->json([
                    'success' => true,
                    'message' => $message,
                    'data' => [
                        'paid_amount' => number_format($invoice->paid_amount, 2),
                        'remaining_amount' => number_format($invoice->remaining_amount, 2),
                        'payment_status' => $invoice->payment_status,
                        'is_paid' => $invoice->isPaid()
                    ]
                ]);
            }

            return back()->with('success', $message);
        } catch (\Exception $e) {
            if ($request->expectsJson() || $request->ajax()) {
                return response()->json([
                    'success' => false,
                    'message' => 'حدث خطأ أثناء إضافة الدفعة: ' . $e->getMessage()
                ], 500);
            }
            return back()->with('error', 'حدث خطأ أثناء إضافة الدفعة: ' . $e->getMessage());
        }
    }

    /**
     * عرض تقرير المدفوعات
     */
    public function paymentsReport(Request $request)
    {
        $query = Invoice::with(['customer', 'company']);

        // فلترة حسب حالة السداد
        if ($request->filled('payment_status')) {
            $query->where('payment_status', $request->payment_status);
        }

        // فلترة حسب التاريخ
        if ($request->filled('date_from')) {
            $query->whereDate('issue_date', '>=', $request->date_from);
        }

        if ($request->filled('date_to')) {
            $query->whereDate('issue_date', '<=', $request->date_to);
        }

        $invoices = $query->orderBy('issue_date', 'desc')->paginate(20);

        // حساب الإحصائيات
        $stats = [
            'total_invoices' => Invoice::count(),
            'paid_invoices' => Invoice::paid()->count(),
            'unpaid_invoices' => Invoice::unpaid()->count(),
            'partially_paid' => Invoice::partiallyPaid()->count(),
            'overdue_invoices' => Invoice::overdue()->count(),
            'total_amount' => Invoice::sum('total_amount'),
            'paid_amount' => Invoice::sum('paid_amount'),
            'remaining_amount' => Invoice::sum('remaining_amount'),
        ];

        return view('invoices.payments-report', compact('invoices', 'stats'));
    }

    /**
     * عرض تحليلات المدفوعات المتقدمة
     */
    public function paymentAnalytics(Request $request)
    {
        // حساب الإحصائيات المتقدمة
        $totalRevenue = Invoice::sum('total_amount');
        $collectedAmount = Invoice::sum('paid_amount');
        $pendingAmount = Invoice::sum('remaining_amount');
        $overdueAmount = Invoice::overdue()->sum('remaining_amount');

        // الإيرادات الشهرية لآخر 6 أشهر
        $monthlyData = [];
        for ($i = 5; $i >= 0; $i--) {
            $date = now()->subMonths($i);
            $monthlyData[] = [
                'month' => $date->format('Y-m'),
                'month_name' => $date->translatedFormat('F'),
                'revenue' => Invoice::whereYear('issue_date', $date->year)
                    ->whereMonth('issue_date', $date->month)
                    ->sum('total_amount'),
                'collected' => Invoice::whereYear('issue_date', $date->year)
                    ->whereMonth('issue_date', $date->month)
                    ->sum('paid_amount')
            ];
        }

        // توزيع طرق السداد
        $paymentMethods = [
            'cash' => Invoice::where('payment_method', 'cash')->count(),
            'bank_transfer' => Invoice::where('payment_method', 'bank_transfer')->count(),
            'credit_card' => Invoice::where('payment_method', 'credit_card')->count(),
            'check' => Invoice::where('payment_method', 'check')->count(),
            'installment' => Invoice::where('payment_method', 'installment')->count(),
        ];

        // أفضل العملاء
        $topCustomers = Customer::withCount('invoices')
            ->withSum('invoices', 'total_amount')
            ->withSum('invoices', 'paid_amount')
            ->having('invoices_count', '>', 0)
            ->orderBy('invoices_sum_total_amount', 'desc')
            ->limit(10)
            ->get()
            ->map(function ($customer) {
                $paymentRate = $customer->invoices_sum_total_amount > 0
                    ? ($customer->invoices_sum_paid_amount / $customer->invoices_sum_total_amount) * 100
                    : 0;

                return [
                    'name' => $customer->name,
                    'invoices_count' => $customer->invoices_count,
                    'total_amount' => $customer->invoices_sum_total_amount,
                    'payment_rate' => round($paymentRate, 2)
                ];
            });

        // تحليل الديون حسب العمر
        $debtAnalysis = [
            '0-30' => Invoice::whereBetween('created_at', [now()->subDays(30), now()])
                ->where('remaining_amount', '>', 0)
                ->sum('remaining_amount'),
            '31-60' => Invoice::whereBetween('created_at', [now()->subDays(60), now()->subDays(31)])
                ->where('remaining_amount', '>', 0)
                ->sum('remaining_amount'),
            '61-90' => Invoice::whereBetween('created_at', [now()->subDays(90), now()->subDays(61)])
                ->where('remaining_amount', '>', 0)
                ->sum('remaining_amount'),
            '90+' => Invoice::where('created_at', '<', now()->subDays(90))
                ->where('remaining_amount', '>', 0)
                ->sum('remaining_amount'),
        ];

        // معدل التحصيل
        $collectionRate = $totalRevenue > 0 ? ($collectedAmount / $totalRevenue) * 100 : 0;

        // متوسط وقت التحصيل
        $avgCollectionTime = Invoice::whereNotNull('payment_date')
            ->selectRaw('AVG(DATEDIFF(payment_date, issue_date)) as avg_days')
            ->first()->avg_days ?? 0;

        // نمو الإيرادات (مقارنة بالشهر الماضي)
        $thisMonth = Invoice::whereYear('issue_date', now()->year)
            ->whereMonth('issue_date', now()->month)
            ->sum('total_amount');

        $lastMonth = Invoice::whereYear('issue_date', now()->subMonth()->year)
            ->whereMonth('issue_date', now()->subMonth()->month)
            ->sum('total_amount');

        $revenueGrowth = $lastMonth > 0 ? (($thisMonth - $lastMonth) / $lastMonth) * 100 : 0;

        $analytics = [
            'totals' => [
                'revenue' => $totalRevenue,
                'collected' => $collectedAmount,
                'pending' => $pendingAmount,
                'overdue' => $overdueAmount,
                'collection_rate' => round($collectionRate, 2),
                'avg_collection_time' => round($avgCollectionTime, 1),
                'revenue_growth' => round($revenueGrowth, 2)
            ],
            'monthly_data' => $monthlyData,
            'payment_methods' => $paymentMethods,
            'top_customers' => $topCustomers,
            'debt_analysis' => $debtAnalysis
        ];

        if ($request->expectsJson()) {
            return response()->json($analytics);
        }

        return view('invoices.payment-analytics', compact('analytics'));
    }

    /**
     * تصدير تحليلات المدفوعات
     */
    public function exportPaymentAnalytics()
    {
        $analytics = $this->paymentAnalytics(request());

        if (request()->get('format') === 'json') {
            return response()->json($analytics->getData());
        }

        // تصدير CSV
        $csvData = [
            ['تحليلات المدفوعات - ' . now()->format('Y-m-d H:i:s')],
            [''],
            ['الإحصائيات العامة'],
            ['إجمالي الإيرادات', number_format($analytics->getData()['totals']['revenue'], 2)],
            ['المحصل فعلياً', number_format($analytics->getData()['totals']['collected'], 2)],
            ['قيد التحصيل', number_format($analytics->getData()['totals']['pending'], 2)],
            ['متأخر السداد', number_format($analytics->getData()['totals']['overdue'], 2)],
            ['معدل التحصيل', $analytics->getData()['totals']['collection_rate'] . '%'],
            ['متوسط وقت التحصيل', $analytics->getData()['totals']['avg_collection_time'] . ' يوم'],
            ['نمو الإيرادات', $analytics->getData()['totals']['revenue_growth'] . '%'],
            [''],
            ['أفضل العملاء'],
            ['اسم العميل', 'عدد الفواتير', 'إجمالي المبلغ', 'معدل السداد']
        ];

        foreach ($analytics->getData()['top_customers'] as $customer) {
            $csvData[] = [
                $customer['name'],
                $customer['invoices_count'],
                number_format($customer['total_amount'], 2),
                $customer['payment_rate'] . '%'
            ];
        }

        $filename = 'payment_analytics_' . now()->format('Y-m-d_H-i-s') . '.csv';

        $callback = function () use ($csvData) {
            $file = fopen('php://output', 'w');
            fprintf($file, chr(0xEF) . chr(0xBB) . chr(0xBF)); // UTF-8 BOM

            foreach ($csvData as $row) {
                fputcsv($file, $row);
            }
            fclose($file);
        };

        return response()->stream($callback, 200, [
            'Content-Type' => 'text/csv; charset=UTF-8',
            'Content-Disposition' => 'attachment; filename="' . $filename . '"',
        ]);
    }

    /**
     * API للحصول على بيانات الرسوم البيانية
     */
    public function getChartData(Request $request)
    {
        $type = $request->get('type');

        switch ($type) {
            case 'revenue':
                return $this->getRevenueChartData();
            case 'payment_methods':
                return $this->getPaymentMethodsChartData();
            case 'debt_analysis':
                return $this->getDebtAnalysisChartData();
            case 'trends':
                return $this->getTrendsChartData();
            default:
                return response()->json(['error' => 'نوع الرسم البياني غير صحيح'], 400);
        }
    }

    private function getRevenueChartData()
    {
        $data = [];
        for ($i = 11; $i >= 0; $i--) {
            $date = now()->subMonths($i);
            $data[] = [
                'month' => $date->translatedFormat('F Y'),
                'revenue' => Invoice::whereYear('issue_date', $date->year)
                    ->whereMonth('issue_date', $date->month)
                    ->sum('total_amount'),
                'collected' => Invoice::whereYear('issue_date', $date->year)
                    ->whereMonth('issue_date', $date->month)
                    ->sum('paid_amount')
            ];
        }

        return response()->json($data);
    }

    private function getPaymentMethodsChartData()
    {
        $methods = [
            'cash' => 'نقداً',
            'bank_transfer' => 'تحويل بنكي',
            'credit_card' => 'بطاقة ائتمان',
            'check' => 'شيك',
            'installment' => 'تقسيط'
        ];

        $data = [];
        foreach ($methods as $key => $name) {
            $count = Invoice::where('payment_method', $key)->count();
            if ($count > 0) {
                $data[] = [
                    'method' => $name,
                    'count' => $count,
                    'amount' => Invoice::where('payment_method', $key)->sum('paid_amount')
                ];
            }
        }

        return response()->json($data);
    }

    private function getDebtAnalysisChartData()
    {
        $ranges = [
            '0-30 يوم' => [now()->subDays(30), now()],
            '31-60 يوم' => [now()->subDays(60), now()->subDays(31)],
            '61-90 يوم' => [now()->subDays(90), now()->subDays(61)],
            '+90 يوم' => [null, now()->subDays(90)]
        ];

        $data = [];
        foreach ($ranges as $label => $dateRange) {
            $query = Invoice::where('remaining_amount', '>', 0);

            if ($dateRange[0] && $dateRange[1]) {
                $query->whereBetween('issue_date', $dateRange);
            } elseif ($dateRange[1]) {
                $query->where('issue_date', '<', $dateRange[1]);
            }

            $data[] = [
                'range' => $label,
                'amount' => $query->sum('remaining_amount'),
                'count' => $query->count()
            ];
        }

        return response()->json($data);
    }

    private function getTrendsChartData()
    {
        $data = [];
        for ($i = 11; $i >= 0; $i--) {
            $date = now()->subMonths($i);
            $totalInvoices = Invoice::whereYear('issue_date', $date->year)
                ->whereMonth('issue_date', $date->month)
                ->count();

            $paidInvoices = Invoice::whereYear('issue_date', $date->year)
                ->whereMonth('issue_date', $date->month)
                ->where('payment_status', 'paid')
                ->count();

            $collectionRate = $totalInvoices > 0 ? ($paidInvoices / $totalInvoices) * 100 : 0;

            $avgDays = Invoice::whereYear('issue_date', $date->year)
                ->whereMonth('issue_date', $date->month)
                ->whereNotNull('payment_date')
                ->selectRaw('AVG(DATEDIFF(payment_date, issue_date)) as avg_days')
                ->first()->avg_days ?? 0;

            $data[] = [
                'month' => $date->translatedFormat('F'),
                'collection_rate' => round($collectionRate, 1),
                'avg_collection_days' => round($avgDays, 1)
            ];
        }

        return response()->json($data);
    }

    /**
     * عرض صفحة إعدادات الإشعارات
     */
    public function notificationSettings()
    {
        // الحصول على الإعدادات المحفوظة
        $settings = [
            'auto_notifications' => config('invoices.auto_notifications', false),
            'payment_reminders' => config('invoices.payment_reminders', true),
            'default_creditor_email' => config('invoices.default_creditor_email', ''),
            'sender_name' => config('invoices.sender_name', config('app.name')),
            'new_invoice_subject' => config('invoices.new_invoice_subject', 'فاتورة جديدة رقم {invoice_number}'),
            'reminder_subject' => config('invoices.reminder_subject', 'تذكير بسداد فاتورة رقم {invoice_number}'),
            'whatsapp_enabled' => config('invoices.whatsapp_enabled', true),
            'whatsapp_auto' => config('invoices.whatsapp_auto', false),
            'whatsapp_template' => config('invoices.whatsapp_template', 'مرحباً {customer_name}، نرسل لكم فاتورة رقم {invoice_number} بمبلغ {total_amount} ر.س. يرجى المراجعة والسداد في الموعد المحدد. شكراً لتعاملكم معنا.'),
            'first_reminder_days' => config('invoices.first_reminder_days', 3),
            'second_reminder_days' => config('invoices.second_reminder_days', 7),
            'final_reminder_days' => config('invoices.final_reminder_days', 15),
            'reminder_email' => config('invoices.reminder_email', true),
            'reminder_whatsapp' => config('invoices.reminder_whatsapp', false),
            'notify_new_invoice' => config('invoices.notify_new_invoice', true),
            'notify_payment_received' => config('invoices.notify_payment_received', true),
            'notify_overdue' => config('invoices.notify_overdue', true),
            'notify_zatca_status' => config('invoices.notify_zatca_status', true),
        ];

        return view('invoices.notification-settings', compact('settings'));
    }

    /**
     * حفظ إعدادات الإشعارات
     */
    public function saveNotificationSettings(Request $request)
    {
        $request->validate([
            'default_creditor_email' => 'nullable|email',
            'sender_name' => 'required|string|max:255',
            'new_invoice_subject' => 'required|string|max:255',
            'reminder_subject' => 'required|string|max:255',
            'whatsapp_template' => 'required|string|max:1000',
            'first_reminder_days' => 'required|integer|min:1|max:30',
            'second_reminder_days' => 'required|integer|min:1|max:30',
            'final_reminder_days' => 'required|integer|min:1|max:60',
        ]);

        try {
            // حفظ الإعدادات في ملف التكوين
            $configPath = config_path('invoices.php');

            $settings = [
                'auto_notifications' => $request->has('auto_notifications'),
                'payment_reminders' => $request->has('payment_reminders'),
                'default_creditor_email' => $request->default_creditor_email,
                'sender_name' => $request->sender_name,
                'new_invoice_subject' => $request->new_invoice_subject,
                'reminder_subject' => $request->reminder_subject,
                'whatsapp_enabled' => $request->has('whatsapp_enabled'),
                'whatsapp_auto' => $request->has('whatsapp_auto'),
                'whatsapp_template' => $request->whatsapp_template,
                'first_reminder_days' => $request->first_reminder_days,
                'second_reminder_days' => $request->second_reminder_days,
                'final_reminder_days' => $request->final_reminder_days,
                'reminder_email' => $request->has('reminder_email'),
                'reminder_whatsapp' => $request->has('reminder_whatsapp'),
                'notify_new_invoice' => $request->has('notify_new_invoice'),
                'notify_payment_received' => $request->has('notify_payment_received'),
                'notify_overdue' => $request->has('notify_overdue'),
                'notify_zatca_status' => $request->has('notify_zatca_status'),
            ];

            // إنشاء محتوى ملف التكوين
            $configContent = "<?php\n\nreturn " . var_export($settings, true) . ";\n";

            // كتابة الملف
            file_put_contents($configPath, $configContent);

            return back()->with('success', 'تم حفظ إعدادات الإشعارات بنجاح');
        } catch (\Exception $e) {
            return back()->with('error', 'حدث خطأ أثناء حفظ الإعدادات: ' . $e->getMessage());
        }
    }

    /**
     * اختبار الإشعارات
     */
    public function testNotifications(Request $request)
    {
        $request->validate([
            'test_email' => 'required|email'
        ]);

        try {
            // إنشاء فاتورة وهمية للاختبار
            $testInvoice = (object) [
                'invoice_number' => 'TEST-' . now()->format('Ymd-His'),
                'total_amount' => 1500.00,
                'issue_date' => now(),
                'customer' => (object) [
                    'name' => 'عميل تجريبي',
                    'email' => $request->test_email
                ],
                'company' => Company::first() ?? (object) ['name' => 'شركة تجريبية']
            ];

            // إرسال إيميل تجريبي
            Mail::send('emails.test-notification', ['invoice' => $testInvoice], function ($message) use ($testInvoice) {
                $message->to($testInvoice->customer->email)
                    ->subject('اختبار إشعارات النظام - فاتورة رقم ' . $testInvoice->invoice_number);
            });

            return response()->json([
                'success' => true,
                'message' => 'تم إرسال الإشعار التجريبي بنجاح إلى ' . $request->test_email
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'فشل في إرسال الإشعار: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * إرسال تذكيرات السداد التلقائية
     */
    public function sendPaymentReminders()
    {
        try {
            $results = [
                'sent' => 0,
                'failed' => 0,
                'details' => []
            ];

            // الحصول على الفواتير المستحقة للتذكير
            $overdueInvoices = Invoice::with(['customer', 'company'])
                ->where('payment_status', '!=', 'paid')
                ->whereNotNull('due_date')
                ->where('due_date', '<', now())
                ->get();

            foreach ($overdueInvoices as $invoice) {
                try {
                    // حساب عدد الأيام منذ الاستحقاق
                    $daysSinceDue = now()->diffInDays($invoice->due_date);

                    // تحديد نوع التذكير
                    $reminderType = $this->getReminderType($daysSinceDue);

                    if ($reminderType && $this->shouldSendReminder($invoice, $reminderType)) {
                        $this->sendReminderNotification($invoice, $reminderType);
                        $results['sent']++;
                        $results['details'][] = "تم إرسال تذكير {$reminderType} للفاتورة {$invoice->invoice_number}";
                    }
                } catch (\Exception $e) {
                    $results['failed']++;
                    $results['details'][] = "فشل في إرسال تذكير للفاتورة {$invoice->invoice_number}: " . $e->getMessage();
                }
            }

            return response()->json([
                'success' => true,
                'message' => "تم إرسال {$results['sent']} تذكير، فشل في {$results['failed']}",
                'results' => $results
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'حدث خطأ في إرسال التذكيرات: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * تحديد نوع التذكير حسب عدد الأيام
     */
    private function getReminderType($daysSinceDue)
    {
        $firstReminderDays = config('invoices.first_reminder_days', 3);
        $secondReminderDays = config('invoices.second_reminder_days', 7);
        $finalReminderDays = config('invoices.final_reminder_days', 15);

        if ($daysSinceDue <= $firstReminderDays) {
            return 'first';
        } elseif ($daysSinceDue <= $secondReminderDays) {
            return 'second';
        } elseif ($daysSinceDue <= $finalReminderDays) {
            return 'final';
        }

        return null; // لا نرسل تذكيرات بعد الحد الأقصى
    }

    /**
     * التحقق من ضرورة إرسال التذكير
     */
    private function shouldSendReminder($invoice, $reminderType)
    {
        // فحص إذا تم إرسال هذا النوع من التذكير مسبقاً
        // يمكن إضافة جدول منفصل لتتبع التذكيرات المرسلة
        return true; // للتبسيط، نفترض أنه يجب إرسال التذكير
    }

    /**
     * إرسال تذكير محدد
     */
    private function sendReminderNotification($invoice, $reminderType)
    {
        if (!$invoice->customer || !$invoice->customer->email) {
            throw new \Exception('لا يوجد بريد إلكتروني للعميل');
        }

        $subject = config('invoices.reminder_subject', 'تذكير بسداد فاتورة رقم {invoice_number}');
        $subject = str_replace('{invoice_number}', $invoice->invoice_number, $subject);

        // إرسال الإيميل
        Mail::send('emails.payment-reminder', [
            'invoice' => $invoice,
            'reminderType' => $reminderType
        ], function ($message) use ($invoice, $subject) {
            $message->to($invoice->customer->email, $invoice->customer->name)
                ->subject($subject);
        });

        // إرسال واتساب إذا كان مفعل
        if (config('invoices.reminder_whatsapp', false) && $invoice->customer->phone) {
            $this->sendWhatsAppReminder($invoice, $reminderType);
        }
    }

    /**
     * إرسال تذكير عبر الواتساب
     */
    private function sendWhatsAppReminder($invoice, $reminderType)
    {
        $template = config('invoices.whatsapp_template', 'تذكير بسداد فاتورة رقم {invoice_number}');

        $message = str_replace([
            '{customer_name}',
            '{invoice_number}',
            '{total_amount}',
            '{due_date}',
            '{company_name}'
        ], [
            $invoice->customer->name,
            $invoice->invoice_number,
            number_format($invoice->remaining_amount, 2),
            $invoice->due_date->format('Y-m-d'),
            $invoice->company->name
        ], $template);

        // هنا يمكن دمج مع API واتساب حقيقي
        // للآن سنسجل في اللوج فقط
        \Log::info('WhatsApp reminder sent', [
            'invoice' => $invoice->invoice_number,
            'customer' => $invoice->customer->name,
            'phone' => $invoice->customer->phone,
            'message' => $message
        ]);
    }

    /**
     * عرض صفحة إدارة الديون
     */
    public function debtManagement(Request $request)
    {
        $query = Invoice::with(['customer', 'company'])
            ->where('payment_status', '!=', 'paid')
            ->whereNotNull('due_date')
            ->where('due_date', '<', now());

        // فلترة حسب عمر الدين
        if ($request->filled('debt_age')) {
            switch ($request->debt_age) {
                case '0-30':
                    $query->whereBetween('due_date', [now()->subDays(30), now()]);
                    break;
                case '31-60':
                    $query->whereBetween('due_date', [now()->subDays(60), now()->subDays(31)]);
                    break;
                case '61-90':
                    $query->whereBetween('due_date', [now()->subDays(90), now()->subDays(61)]);
                    break;
                case '90+':
                    $query->where('due_date', '<', now()->subDays(90));
                    break;
            }
        }

        // فلترة حسب المبلغ
        if ($request->filled('amount_from')) {
            $query->where('remaining_amount', '>=', $request->amount_from);
        }
        if ($request->filled('amount_to')) {
            $query->where('remaining_amount', '<=', $request->amount_to);
        }

        // فلترة حسب العميل
        if ($request->filled('customer')) {
            $query->whereHas('customer', function ($q) use ($request) {
                $q->where('name', 'like', '%' . $request->customer . '%');
            });
        }

        // الترتيب
        $sortBy = $request->get('sort_by', 'overdue_days');
        $sortDirection = $request->get('sort_direction', 'desc');

        switch ($sortBy) {
            case 'amount':
                $query->orderBy('remaining_amount', $sortDirection);
                break;
            case 'customer':
                $query->join('customers', 'invoices.customer_id', '=', 'customers.id')
                    ->orderBy('customers.name', $sortDirection);
                break;
            case 'due_date':
                $query->orderBy('due_date', $sortDirection);
                break;
            default: // overdue_days
                $query->orderBy('due_date', 'asc');
                break;
        }

        $invoices = $query->paginate(20)->appends($request->all());

        // حساب الإحصائيات
        $stats = $this->calculateDebtStats();

        return view('invoices.debt-management', compact('invoices', 'stats'));
    }

    /**
     * حساب إحصائيات الديون
     */
    private function calculateDebtStats()
    {
        $baseQuery = Invoice::where('payment_status', '!=', 'paid')
            ->whereNotNull('due_date')
            ->where('due_date', '<', now());

        $stats = [
            'total_debt' => $baseQuery->sum('remaining_amount'),
            'total_invoices' => $baseQuery->count(),

            // 0-30 يوم
            'debt_0_30' => $baseQuery->clone()->whereBetween('due_date', [now()->subDays(30), now()])->sum('remaining_amount'),
            'count_0_30' => $baseQuery->clone()->whereBetween('due_date', [now()->subDays(30), now()])->count(),

            // 30-90 يوم
            'debt_30_90' => $baseQuery->clone()->whereBetween('due_date', [now()->subDays(90), now()->subDays(30)])->sum('remaining_amount'),
            'count_30_90' => $baseQuery->clone()->whereBetween('due_date', [now()->subDays(90), now()->subDays(30)])->count(),

            // أكثر من 90 يوم
            'debt_90_plus' => $baseQuery->clone()->where('due_date', '<', now()->subDays(90))->sum('remaining_amount'),
            'count_90_plus' => $baseQuery->clone()->where('due_date', '<', now()->subDays(90))->count(),
        ];

        return $stats;
    }

    /**
     * إرسال تذكير لفاتورة واحدة
     */
    public function sendSingleReminder(Invoice $invoice)
    {
        try {
            if ($invoice->payment_status === 'paid') {
                return response()->json([
                    'success' => false,
                    'message' => 'الفاتورة مدفوعة بالكامل'
                ]);
            }

            // تحديد نوع التذكير حسب أيام التأخير
            $daysSinceDue = now()->diffInDays($invoice->due_date);
            $reminderType = $this->getReminderType($daysSinceDue);

            if (!$reminderType) {
                return response()->json([
                    'success' => false,
                    'message' => 'تجاوزت الفاتورة الحد الأقصى لإرسال التذكيرات'
                ]);
            }

            $this->sendReminderNotification($invoice, $reminderType);

            // تحديث تاريخ آخر تذكير
            $invoice->last_reminder_sent_at = now();
            $invoice->save();

            return response()->json([
                'success' => true,
                'message' => 'تم إرسال التذكير بنجاح'
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'حدث خطأ أثناء إرسال التذكير: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * إرسال تذكيرات جماعية
     */
    public function sendBulkReminders(Request $request)
    {
        $request->validate([
            'invoice_ids' => 'required|array',
            'invoice_ids.*' => 'exists:invoices,id'
        ]);

        try {
            $results = [
                'sent' => 0,
                'failed' => 0,
                'details' => []
            ];

            $invoices = Invoice::whereIn('id', $request->invoice_ids)
                ->where('payment_status', '!=', 'paid')
                ->get();

            foreach ($invoices as $invoice) {
                try {
                    $daysSinceDue = now()->diffInDays($invoice->due_date);
                    $reminderType = $this->getReminderType($daysSinceDue);

                    if ($reminderType) {
                        $this->sendReminderNotification($invoice, $reminderType);
                        $invoice->last_reminder_sent_at = now();
                        $invoice->save();

                        $results['sent']++;
                        $results['details'][] = "تم إرسال تذكير للفاتورة {$invoice->invoice_number}";
                    } else {
                        $results['failed']++;
                        $results['details'][] = "الفاتورة {$invoice->invoice_number} تجاوزت الحد الأقصى";
                    }
                } catch (\Exception $e) {
                    $results['failed']++;
                    $results['details'][] = "فشل في إرسال تذكير للفاتورة {$invoice->invoice_number}: " . $e->getMessage();
                }
            }

            return response()->json([
                'success' => true,
                'message' => "تم إرسال {$results['sent']} تذكير، فشل في {$results['failed']}",
                'results' => $results
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'حدث خطأ أثناء إرسال التذكيرات: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * تصنيف فاتورة كعالية المخاطر
     */
    public function markAsHighRisk(Invoice $invoice)
    {
        try {
            $invoice->is_high_risk = true;
            $invoice->risk_marked_at = now();
            $invoice->save();

            // إرسال إشعار للإدارة
            $this->sendHighRiskNotification($invoice);

            return response()->json([
                'success' => true,
                'message' => 'تم تصنيف الفاتورة كعالية المخاطر'
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'حدث خطأ أثناء التصنيف: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * تصدير تقرير الديون
     */
    public function exportDebtReport()
    {
        try {
            $invoices = Invoice::with(['customer', 'company'])
                ->where('payment_status', '!=', 'paid')
                ->whereNotNull('due_date')
                ->where('due_date', '<', now())
                ->orderBy('due_date', 'asc')
                ->get();

            $csvData = [
                ['تقرير الديون المتأخرة - ' . now()->format('Y-m-d H:i:s')],
                [''],
                [
                    'رقم الفاتورة',
                    'العميل',
                    'البريد الإلكتروني',
                    'الهاتف',
                    'تاريخ الإصدار',
                    'تاريخ الاستحقاق',
                    'أيام التأخير',
                    'المبلغ الإجمالي',
                    'المبلغ المتبقي',
                    'نسبة السداد',
                    'مستوى المخاطر',
                    'آخر تذكير'
                ]
            ];

            foreach ($invoices as $invoice) {
                $overdueDays = now()->diffInDays($invoice->due_date);
                $riskLevel = $overdueDays <= 30 ? 'منخفض' : ($overdueDays <= 90 ? 'متوسط' : 'عالي');
                $paymentPercentage = $invoice->total_amount > 0 ? (($invoice->paid_amount / $invoice->total_amount) * 100) : 0;

                $csvData[] = [
                    $invoice->invoice_number,
                    $invoice->customer->name ?? 'غير محدد',
                    $invoice->customer->email ?? '',
                    $invoice->customer->phone ?? '',
                    $invoice->issue_date->format('Y-m-d'),
                    $invoice->due_date->format('Y-m-d'),
                    $overdueDays,
                    number_format($invoice->total_amount, 2),
                    number_format($invoice->remaining_amount, 2),
                    round($paymentPercentage, 2) . '%',
                    $riskLevel,
                    $invoice->last_reminder_sent_at ? $invoice->last_reminder_sent_at->format('Y-m-d') : 'لم يتم الإرسال'
                ];
            }

            $filename = 'debt_report_' . now()->format('Y-m-d_H-i-s') . '.csv';

            $callback = function () use ($csvData) {
                $file = fopen('php://output', 'w');
                fprintf($file, chr(0xEF) . chr(0xBB) . chr(0xBF)); // UTF-8 BOM

                foreach ($csvData as $row) {
                    fputcsv($file, $row);
                }
                fclose($file);
            };

            return response()->stream($callback, 200, [
                'Content-Type' => 'text/csv; charset=UTF-8',
                'Content-Disposition' => 'attachment; filename="' . $filename . '"',
            ]);
        } catch (\Exception $e) {
            return back()->with('error', 'حدث خطأ أثناء تصدير التقرير: ' . $e->getMessage());
        }
    }

    /**
     * إرسال إشعار المخاطر العالية
     */
    private function sendHighRiskNotification($invoice)
    {
        try {
            $adminEmail = config('invoices.default_creditor_email');
            if (!$adminEmail) return;

            Mail::send('emails.high-risk-notification', [
                'invoice' => $invoice
            ], function ($message) use ($adminEmail, $invoice) {
                $message->to($adminEmail)
                    ->subject('تنبيه: فاتورة عالية المخاطر - ' . $invoice->invoice_number);
            });
        } catch (\Exception $e) {
            \Log::warning('فشل في إرسال إشعار المخاطر العالية: ' . $e->getMessage());
        }
    }

    /**
     * عرض تقرير أداء التحصيل
     */
    public function collectionPerformance(Request $request)
    {
        $period = $request->get('period', 12);

        $performanceData = [];
        for ($i = intval($period) - 1; $i >= 0; $i--) {
            $date = now()->subMonths($i);

            $monthlyInvoices = Invoice::whereYear('issue_date', $date->year)
                ->whereMonth('issue_date', $date->month);

            $totalInvoices = $monthlyInvoices->count();
            $totalAmount = $monthlyInvoices->sum('total_amount');
            $collectedAmount = $monthlyInvoices->sum('paid_amount');
            $overdueInvoices = $monthlyInvoices->where('due_date', '<', now())
                ->where('payment_status', '!=', 'paid')->count();

            $collectionRate = $totalAmount > 0 ? ($collectedAmount / $totalAmount) * 100 : 0;

            $avgCollectionTime = $monthlyInvoices->whereNotNull('payment_date')
                ->selectRaw('AVG(DATEDIFF(payment_date, issue_date)) as avg_days')
                ->first()->avg_days ?? 0;

            $performanceData[] = [
                'month' => $date->translatedFormat('F Y'),
                'total_invoices' => $totalInvoices,
                'total_amount' => $totalAmount,
                'collected_amount' => $collectedAmount,
                'collection_rate' => round($collectionRate, 2),
                'overdue_invoices' => $overdueInvoices,
                'avg_collection_time' => round($avgCollectionTime, 1)
            ];
        }

        return view('invoices.collection-performance', compact('performanceData', 'period'));
    }

    /**
     * تصدير تقرير أداء التحصيل
     */
    public function exportCollectionPerformance(Request $request)
    {
        try {
            $period = $request->get('period', '12');

            $performanceData = [];
            for ($i = intval($period) - 1; $i >= 0; $i--) {
                $date = now()->subMonths($i);

                $monthlyInvoices = Invoice::whereYear('issue_date', $date->year)
                    ->whereMonth('issue_date', $date->month);

                $totalInvoices = $monthlyInvoices->count();
                $totalAmount = $monthlyInvoices->sum('total_amount');
                $collectedAmount = $monthlyInvoices->sum('paid_amount');
                $overdueInvoices = $monthlyInvoices->where('due_date', '<', now())
                    ->where('payment_status', '!=', 'paid')->count();

                $collectionRate = $totalAmount > 0 ? ($collectedAmount / $totalAmount) * 100 : 0;

                $avgCollectionTime = $monthlyInvoices->whereNotNull('payment_date')
                    ->selectRaw('AVG(DATEDIFF(payment_date, issue_date)) as avg_days')
                    ->first()->avg_days ?? 0;

                $performanceData[] = [
                    'month' => $date->translatedFormat('F Y'),
                    'total_invoices' => $totalInvoices,
                    'total_amount' => $totalAmount,
                    'collected_amount' => $collectedAmount,
                    'collection_rate' => round($collectionRate, 2),
                    'overdue_invoices' => $overdueInvoices,
                    'avg_collection_time' => round($avgCollectionTime, 1)
                ];
            }

            $csvData = [
                ['تقرير أداء التحصيل - ' . now()->format('Y-m-d H:i:s')],
                ['فترة التقرير: آخر ' . $period . ' شهر'],
                [''],
                [
                    'الشهر',
                    'عدد الفواتير',
                    'إجمالي المبلغ (ر.س)',
                    'المحصل فعلياً (ر.س)',
                    'معدل التحصيل (%)',
                    'الفواتير المتأخرة',
                    'متوسط وقت التحصيل (يوم)',
                    'تقييم الأداء'
                ]
            ];

            foreach ($performanceData as $data) {
                $performance = 'ممتاز';
                if ($data['collection_rate'] < 70) {
                    $performance = 'ضعيف';
                } elseif ($data['collection_rate'] < 85) {
                    $performance = 'متوسط';
                } elseif ($data['collection_rate'] < 95) {
                    $performance = 'جيد';
                }

                $csvData[] = [
                    $data['month'],
                    $data['total_invoices'],
                    number_format($data['total_amount'], 2),
                    number_format($data['collected_amount'], 2),
                    $data['collection_rate'],
                    $data['overdue_invoices'],
                    $data['avg_collection_time'],
                    $performance
                ];
            }

            // إضافة الملخص العام
            $csvData[] = [''];
            $csvData[] = ['الملخص العام'];
            $csvData[] = ['متوسط معدل التحصيل', number_format(collect($performanceData)->avg('collection_rate'), 2) . '%'];
            $csvData[] = ['متوسط وقت التحصيل', number_format(collect($performanceData)->avg('avg_collection_time'), 1) . ' يوم'];
            $csvData[] = ['إجمالي الفواتير', number_format(collect($performanceData)->sum('total_invoices'))];
            $csvData[] = ['إجمالي المبلغ', number_format(collect($performanceData)->sum('total_amount'), 2) . ' ر.س'];
            $csvData[] = ['إجمالي المحصل', number_format(collect($performanceData)->sum('collected_amount'), 2) . ' ر.س'];

            $filename = 'collection_performance_' . now()->format('Y-m-d_H-i-s') . '.csv';

            $callback = function () use ($csvData) {
                $file = fopen('php://output', 'w');
                fprintf($file, chr(0xEF) . chr(0xBB) . chr(0xBF)); // UTF-8 BOM

                foreach ($csvData as $row) {
                    fputcsv($file, $row);
                }
                fclose($file);
            };

            return response()->stream($callback, 200, [
                'Content-Type' => 'text/csv; charset=UTF-8',
                'Content-Disposition' => 'attachment; filename="' . $filename . '"',
            ]);
        } catch (\Exception $e) {
            return back()->with('error', 'حدث خطأ أثناء تصدير التقرير: ' . $e->getMessage());
        }
    }

    /**
     * عرض كشف حساب العملاء
     */
    public function customerStatement(Request $request)
    {
        $query = Customer::with(['invoices' => function ($q) {
            $q->orderBy('issue_date', 'asc');
        }]);

        // فلترة حسب العميل
        if ($request->filled('customer_id')) {
            $query->where('id', $request->customer_id);
        }

        // فلترة حسب التاريخ
        if ($request->filled('date_from') || $request->filled('date_to')) {
            $query->whereHas('invoices', function ($q) use ($request) {
                if ($request->filled('date_from')) {
                    $q->whereDate('issue_date', '>=', $request->date_from);
                }
                if ($request->filled('date_to')) {
                    $q->whereDate('issue_date', '<=', $request->date_to);
                }
            });
        }

        // البحث في اسم العميل
        if ($request->filled('search')) {
            $query->where('name', 'like', '%' . $request->search . '%');
        }

        $customers = $query->get();

        // حساب البيانات لكل عميل
        $customerStatements = $customers->map(function ($customer) use ($request) {
            $invoicesQuery = $customer->invoices();

            // تطبيق فلاتر التاريخ على الفواتير
            if ($request->filled('date_from')) {
                $invoicesQuery->whereDate('issue_date', '>=', $request->date_from);
            }
            if ($request->filled('date_to')) {
                $invoicesQuery->whereDate('issue_date', '<=', $request->date_to);
            }

            $invoices = $invoicesQuery->orderBy('issue_date', 'asc')->get();

            $totalDebit = 0;  // إجمالي المدين (الفواتير)
            $totalCredit = 0; // إجمالي الدائن (المدفوعات)
            $runningBalance = 0; // الرصيد الجاري

            $transactions = collect();

            foreach ($invoices as $invoice) {
                // إضافة الفاتورة كمعاملة مدينة
                $totalDebit += $invoice->total_amount;
                $runningBalance += $invoice->total_amount;

                $transactions->push([
                    'date' => $invoice->issue_date,
                    'type' => 'invoice',
                    'reference' => $invoice->invoice_number,
                    'description' => 'فاتورة رقم ' . $invoice->invoice_number,
                    'debit' => $invoice->total_amount,
                    'credit' => 0,
                    'balance' => $runningBalance,
                    'invoice' => $invoice
                ]);

                // إضافة المدفوعات كمعاملات دائنة
                if ($invoice->paid_amount > 0) {
                    $totalCredit += $invoice->paid_amount;
                    $runningBalance -= $invoice->paid_amount;

                    $transactions->push([
                        'date' => $invoice->payment_date ?? $invoice->issue_date,
                        'type' => 'payment',
                        'reference' => $invoice->invoice_number,
                        'description' => 'دفعة للفاتورة ' . $invoice->invoice_number .
                            ($invoice->payment_method ? ' (' . $this->getPaymentMethodName($invoice->payment_method) . ')' : ''),
                        'debit' => 0,
                        'credit' => $invoice->paid_amount,
                        'balance' => $runningBalance,
                        'invoice' => $invoice
                    ]);
                }
            }

            // ترتيب المعاملات حسب التاريخ
            $transactions = $transactions->sortBy('date')->values();

            return [
                'customer' => $customer,
                'total_debit' => $totalDebit,
                'total_credit' => $totalCredit,
                'balance' => $totalDebit - $totalCredit,
                'transactions' => $transactions,
                'invoices_count' => $invoices->count(),
                'paid_invoices_count' => $invoices->where('payment_status', 'paid')->count(),
                'pending_invoices_count' => $invoices->where('payment_status', '!=', 'paid')->count()
            ];
        })->filter(function ($statement) {
            // إظهار العملاء الذين لديهم معاملات فقط
            return $statement['transactions']->count() > 0;
        });

        // حساب الإحصائيات العامة
        $summary = [
            'total_customers' => $customerStatements->count(),
            'total_debit' => $customerStatements->sum('total_debit'),
            'total_credit' => $customerStatements->sum('total_credit'),
            'total_balance' => $customerStatements->sum('balance'),
            'customers_with_balance' => $customerStatements->where('balance', '>', 0)->count(),
            'customers_with_credit' => $customerStatements->where('balance', '<', 0)->count(),
        ];

        return view('invoices.customer-statement', compact('customerStatements', 'summary'));
    }

    /**
     * تصدير كشف حساب العملاء
     */
    public function exportCustomerStatement(Request $request)
    {
        try {
            $query = Customer::with(['invoices' => function ($q) {
                $q->orderBy('issue_date', 'asc');
            }]);

            // تطبيق نفس الفلاتر
            if ($request->filled('customer_id')) {
                $query->where('id', $request->customer_id);
            }

            if ($request->filled('date_from') || $request->filled('date_to')) {
                $query->whereHas('invoices', function ($q) use ($request) {
                    if ($request->filled('date_from')) {
                        $q->whereDate('issue_date', '>=', $request->date_from);
                    }
                    if ($request->filled('date_to')) {
                        $q->whereDate('issue_date', '<=', $request->date_to);
                    }
                });
            }

            if ($request->filled('search')) {
                $query->where('name', 'like', '%' . $request->search . '%');
            }

            $customers = $query->get();

            $csvData = [
                ['كشف حساب العملاء - ' . now()->format('Y-m-d H:i:s')],
                ['فترة التقرير: من ' . ($request->date_from ?? 'البداية') . ' إلى ' . ($request->date_to ?? 'النهاية')],
                [''],
            ];

            foreach ($customers as $customer) {
                $invoicesQuery = $customer->invoices();

                if ($request->filled('date_from')) {
                    $invoicesQuery->whereDate('issue_date', '>=', $request->date_from);
                }
                if ($request->filled('date_to')) {
                    $invoicesQuery->whereDate('issue_date', '<=', $request->date_to);
                }

                $invoices = $invoicesQuery->orderBy('issue_date', 'asc')->get();

                if ($invoices->count() === 0) continue;

                $totalDebit = 0;
                $totalCredit = 0;
                $runningBalance = 0;

                // إضافة رأس العميل
                $csvData[] = ['العميل: ' . $customer->name];
                $csvData[] = ['البريد الإلكتروني: ' . $customer->email];
                $csvData[] = ['الهاتف: ' . $customer->phone];
                $csvData[] = [''];
                $csvData[] = [
                    'التاريخ',
                    'نوع المعاملة',
                    'المرجع',
                    'الوصف',
                    'مدين (ر.س)',
                    'دائن (ر.س)',
                    'الرصيد (ر.س)'
                ];

                foreach ($invoices as $invoice) {
                    // معاملة الفاتورة
                    $totalDebit += $invoice->total_amount;
                    $runningBalance += $invoice->total_amount;

                    $csvData[] = [
                        $invoice->issue_date->format('Y-m-d'),
                        'فاتورة',
                        $invoice->invoice_number,
                        'فاتورة رقم ' . $invoice->invoice_number,
                        number_format($invoice->total_amount, 2),
                        '0.00',
                        number_format($runningBalance, 2)
                    ];

                    // معاملة الدفع إن وجدت
                    if ($invoice->paid_amount > 0) {
                        $totalCredit += $invoice->paid_amount;
                        $runningBalance -= $invoice->paid_amount;

                        $csvData[] = [
                            ($invoice->payment_date ?? $invoice->issue_date)->format('Y-m-d'),
                            'دفعة',
                            $invoice->invoice_number,
                            'دفعة للفاتورة ' . $invoice->invoice_number .
                                ($invoice->payment_method ? ' (' . $this->getPaymentMethodName($invoice->payment_method) . ')' : ''),
                            '0.00',
                            number_format($invoice->paid_amount, 2),
                            number_format($runningBalance, 2)
                        ];
                    }
                }

                // إضافة ملخص العميل
                $csvData[] = [''];
                $csvData[] = ['ملخص العميل'];
                $csvData[] = ['إجمالي المدين', '', '', '', number_format($totalDebit, 2), '', ''];
                $csvData[] = ['إجمالي الدائن', '', '', '', '', number_format($totalCredit, 2), ''];
                $csvData[] = ['الرصيد المتبقي', '', '', '', '', '', number_format($totalDebit - $totalCredit, 2)];
                $csvData[] = ['', '', '', '', '', '', ''];
                $csvData[] = ['', '', '', '', '', '', ''];
            }

            $filename = 'customer_statement_' . now()->format('Y-m-d_H-i-s') . '.csv';

            $callback = function () use ($csvData) {
                $file = fopen('php://output', 'w');
                fprintf($file, chr(0xEF) . chr(0xBB) . chr(0xBF)); // UTF-8 BOM

                foreach ($csvData as $row) {
                    fputcsv($file, $row);
                }
                fclose($file);
            };

            return response()->stream($callback, 200, [
                'Content-Type' => 'text/csv; charset=UTF-8',
                'Content-Disposition' => 'attachment; filename="' . $filename . '"',
            ]);
        } catch (\Exception $e) {
            return back()->with('error', 'حدث خطأ أثناء تصدير التقرير: ' . $e->getMessage());
        }
    }

    /**
     * ترجمة طريقة الدفع للعربية
     */
    private function getPaymentMethodName($method)
    {
        $methods = [
            'cash' => 'نقداً',
            'bank_transfer' => 'تحويل بنكي',
            'credit_card' => 'بطاقة ائتمان',
            'check' => 'شيك',
            'installment' => 'تقسيط'
        ];

        return $methods[$method] ?? $method;
    }

    /**
     * عرض تقرير حركة الصندوق للفواتير
     */
    public function cashFlowReport(Request $request)
    {
        $startDate = $request->get('start_date', now()->startOfMonth()->format('Y-m-d'));
        $endDate = $request->get('end_date', now()->endOfMonth()->format('Y-m-d'));
        $paymentMethod = $request->get('payment_method');

        // بناء الاستعلام الأساسي للفواتير
        $invoicesQuery = Invoice::with(['customer', 'company'])
            ->whereBetween('issue_date', [$startDate, $endDate]);

        if ($paymentMethod) {
            $invoicesQuery->where('payment_method', $paymentMethod);
        }

        $invoices = $invoicesQuery->get();

        // حساب الحركات النقدية
        $cashFlowData = [
            'total_invoices' => $invoices->count(),
            'total_amount' => $invoices->sum('total_amount'),
            'cash_received' => $invoices->where('payment_method', 'cash')->sum('paid_amount'),
            'bank_transfers' => $invoices->where('payment_method', 'bank_transfer')->sum('paid_amount'),
            'card_payments' => $invoices->where('payment_method', 'credit_card')->sum('paid_amount'),
            'check_payments' => $invoices->where('payment_method', 'check')->sum('paid_amount'),
            'installment_payments' => $invoices->where('payment_method', 'installment')->sum('paid_amount'),
            'total_received' => $invoices->sum('paid_amount'),
            'pending_amount' => $invoices->sum('remaining_amount')
        ];

        // حركة الصندوق اليومية
        $dailyCashFlow = $invoices->groupBy(function($invoice) {
            return $invoice->issue_date->format('Y-m-d');
        })->map(function($dailyInvoices) {
            return [
                'date' => $dailyInvoices->first()->issue_date->format('Y-m-d'),
                'invoices_count' => $dailyInvoices->count(),
                'total_amount' => $dailyInvoices->sum('total_amount'),
                'cash_received' => $dailyInvoices->where('payment_method', 'cash')->sum('paid_amount'),
                'other_received' => $dailyInvoices->whereNotIn('payment_method', ['cash', null])->sum('paid_amount'),
                'pending' => $dailyInvoices->sum('remaining_amount')
            ];
        })->sortBy('date');

        // أفضل العملاء نقدياً
        $topCashCustomers = $invoices->where('payment_method', 'cash')
            ->groupBy('customer_id')
            ->map(function($customerInvoices) {
                $customer = $customerInvoices->first()->customer;
                return [
                    'customer' => $customer,
                    'invoices_count' => $customerInvoices->count(),
                    'total_paid' => $customerInvoices->sum('paid_amount'),
                    'avg_payment' => $customerInvoices->avg('paid_amount')
                ];
            })
            ->sortByDesc('total_paid')
            ->take(10);

        return view('invoices.cash-flow-report', compact(
            'cashFlowData',
            'dailyCashFlow',
            'topCashCustomers',
            'startDate',
            'endDate',
            'paymentMethod'
        ));
    }

    /**
     * البحث المتقدم في الفواتير
     */
    public function advancedSearch(Request $request)
    {
        $query = Invoice::with(['customer', 'company', 'user']);

        // البحث في رقم الفاتورة
        if ($request->filled('invoice_number')) {
            $query->where('invoice_number', 'like', '%' . $request->invoice_number . '%');
        }

        // البحث في اسم العميل أو رقم الهاتف أو البريد
        if ($request->filled('customer_search')) {
            $customerSearch = $request->customer_search;
            $query->whereHas('customer', function ($q) use ($customerSearch) {
                $q->where('name', 'like', "%{$customerSearch}%")
                  ->orWhere('email', 'like', "%{$customerSearch}%")
                  ->orWhere('phone', 'like', "%{$customerSearch}%")
                  ->orWhere('vat_number', 'like', "%{$customerSearch}%");
            });
        }

        // البحث حسب المنتجات
        if ($request->filled('product_search')) {
            $productSearch = $request->product_search;
            $query->whereHas('items', function ($q) use ($productSearch) {
                $q->where('name', 'like', "%{$productSearch}%")
                  ->orWhere('sku', 'like', "%{$productSearch}%");
            });
        }

        // فلترة حسب النوع
        if ($request->filled('type')) {
            $query->where('type', $request->type);
        }

        // فلترة حسب الحالة
        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        // فلترة حسب حالة الدفع
        if ($request->filled('payment_status')) {
            $query->where('payment_status', $request->payment_status);
        }

        // فلترة حسب طريقة الدفع
        if ($request->filled('payment_method')) {
            $query->where('payment_method', $request->payment_method);
        }

        // فلترة حسب التاريخ
        if ($request->filled('date_from')) {
            $query->whereDate('issue_date', '>=', $request->date_from);
        }

        if ($request->filled('date_to')) {
            $query->whereDate('issue_date', '<=', $request->date_to);
        }

        // فلترة حسب المبلغ
        if ($request->filled('amount_from')) {
            $query->where('total_amount', '>=', $request->amount_from);
        }

        if ($request->filled('amount_to')) {
            $query->where('total_amount', '<=', $request->amount_to);
        }

        // فلترة العملاء التجاريين فقط
        if ($request->filled('business_customers_only')) {
            $query->whereHas('customer', function ($q) {
                $q->where('customer_type', 'business');
            });
        }

        // فلترة الفواتير المرسلة لزاتكا
        if ($request->filled('zatca_sent_only')) {
            $query->whereNotNull('sent_to_zatca_at');
        }

        // فلترة حسب المستخدم المنشئ
        if ($request->filled('created_by')) {
            $query->where('user_id', $request->created_by);
        }

        // الترتيب
        $sortBy = $request->get('sort_by', 'created_at');
        $sortDirection = $request->get('sort_direction', 'desc');
        $query->orderBy($sortBy, $sortDirection);

        $invoices = $query->paginate(20)->appends($request->all());

        // حساب الإحصائيات للنتائج المفلترة
        $stats = [
            'total_invoices' => $query->count(),
            'total_amount' => $query->sum('total_amount'),
            'total_paid' => $query->sum('paid_amount'),
            'total_pending' => $query->sum('remaining_amount'),
            'standard_invoices' => $query->where('type', 'standard')->count(),
            'simplified_invoices' => $query->where('type', 'simplified')->count(),
        ];

        return view('invoices.advanced-search', compact('invoices', 'stats'));
    }

    /**
     * ربط دفعة الفاتورة بحركة نقدية
     */
    public function linkCashTransaction(Request $request, Invoice $invoice)
    {
        $request->validate([
            'amount' => 'required|numeric|min:0.01',
            'cash_account_id' => 'required|exists:accounts,id',
            'payment_method' => 'required|in:cash,bank_transfer,check,credit_card',
            'notes' => 'nullable|string|max:1000',
            'reference' => 'nullable|string|max:255'
        ]);

        DB::beginTransaction();
        try {
            // إضافة الدفعة للفاتورة
            $amount = $request->amount;

            // التحقق من عدم تجاوز المبلغ
            if ($invoice->paid_amount + $amount > $invoice->total_amount) {
                return back()->with('error', 'المبلغ يتجاوز المبلغ المستحق للفاتورة');
            }

            // إنشاء حركة نقدية
            $cashTransaction = \App\Models\CashTransaction::create([
                'transaction_number' => \App\Models\CashTransaction::generateTransactionNumber('receipt'),
                'transaction_date' => now(),
                'cash_account_id' => $request->cash_account_id,
                'type' => 'receipt',
                'amount' => $amount,
                'reference' => $request->reference ?: $invoice->invoice_number,
                'description' => 'استلام دفعة للفاتورة ' . $invoice->invoice_number,
                'contra_account_id' => $this->getCustomerReceivableAccount($invoice->customer_id),
                'notes' => $request->notes,
                'created_by' => auth()->id(),
                'invoice_id' => $invoice->id, // ربط بالفاتورة
                'received_from' => $invoice->customer ? $invoice->customer->name : null,
                'payment_method' => $request->payment_method
            ]);

            // إنشاء القيد المحاسبي
            $cashTransaction->createJournalEntry();

            // تحديث الفاتورة
            $invoice->increment('paid_amount', $amount);
            $invoice->decrement('remaining_amount', $amount);
            $invoice->payment_method = $request->payment_method;
            $invoice->payment_date = now();
            $invoice->payment_notes = $request->notes;
            $invoice->updatePaymentStatus();
            $invoice->save();

            DB::commit();

            return back()->with('success', 'تم ربط الدفعة بحركة الصندوق بنجاح');

        } catch (\Exception $e) {
            DB::rollback();
            return back()->with('error', 'حدث خطأ أثناء ربط الدفعة: ' . $e->getMessage());
        }
    }

    /**
     * الحصول على حساب المدينين للعميل
     */
    private function getCustomerReceivableAccount($customerId)
    {
        // البحث عن حساب مدينين أو إنشاء حساب افتراضي
        $receivableAccount = \App\Models\Account::where('account_type', 'receivable')
            ->where('name', 'like', '%مدينين%')
            ->first();

        if (!$receivableAccount) {
            // إنشاء حساب مدينين افتراضي إذا لم يوجد
            $receivableAccount = \App\Models\Account::create([
                'code' => '1210',
                'name' => 'حسابات مدينة - عملاء',
                'account_type' => 'receivable',
                'parent_id' => null,
                'is_active' => true
            ]);
        }

        return $receivableAccount->id;
    }

    /**
     * تصدير تقرير حركة الصندوق
     */
    public function exportCashFlowReport(Request $request)
    {
        $startDate = $request->get('start_date', now()->startOfMonth()->format('Y-m-d'));
        $endDate = $request->get('end_date', now()->endOfMonth()->format('Y-m-d'));
        $paymentMethod = $request->get('payment_method');

        $invoicesQuery = Invoice::with(['customer'])
            ->whereBetween('issue_date', [$startDate, $endDate]);

        if ($paymentMethod) {
            $invoicesQuery->where('payment_method', $paymentMethod);
        }

        $invoices = $invoicesQuery->get();

        $csvData = [
            ['تقرير حركة الصندوق للفواتير - ' . now()->format('Y-m-d H:i:s')],
            ['فترة التقرير: من ' . $startDate . ' إلى ' . $endDate],
            [''],
            [
                'رقم الفاتورة',
                'العميل',
                'تاريخ الإصدار',
                'طريقة الدفع',
                'المبلغ الإجمالي',
                'المبلغ المدفوع',
                'المبلغ المتبقي',
                'حالة الدفع',
                'ملاحظات الدفع'
            ]
        ];

        foreach ($invoices as $invoice) {
            $csvData[] = [
                $invoice->invoice_number,
                $invoice->customer ? $invoice->customer->name : 'غير محدد',
                $invoice->issue_date->format('Y-m-d'),
                $this->getPaymentMethodName($invoice->payment_method),
                number_format($invoice->total_amount, 2),
                number_format($invoice->paid_amount, 2),
                number_format($invoice->remaining_amount, 2),
                $this->getPaymentStatusName($invoice->payment_status),
                $invoice->payment_notes ?: ''
            ];
        }

        // إضافة الملخص
        $csvData[] = [''];
        $csvData[] = ['ملخص التقرير'];
        $csvData[] = ['إجمالي الفواتير', $invoices->count()];
        $csvData[] = ['إجمالي المبلغ', number_format($invoices->sum('total_amount'), 2) . ' ر.س'];
        $csvData[] = ['إجمالي المحصل', number_format($invoices->sum('paid_amount'), 2) . ' ر.س'];
        $csvData[] = ['إجمالي المتبقي', number_format($invoices->sum('remaining_amount'), 2) . ' ر.س'];

        $filename = 'invoice_cash_flow_report_' . now()->format('Y-m-d_H-i-s') . '.csv';

        $callback = function () use ($csvData) {
            $file = fopen('php://output', 'w');
            fprintf($file, chr(0xEF) . chr(0xBB) . chr(0xBF)); // UTF-8 BOM

            foreach ($csvData as $row) {
                fputcsv($file, $row);
            }
            fclose($file);
        };

        return response()->stream($callback, 200, [
            'Content-Type' => 'text/csv; charset=UTF-8',
            'Content-Disposition' => 'attachment; filename="' . $filename . '"',
        ]);
    }

    /**
     * ترجمة حالة الدفع للعربية
     */
    private function getPaymentStatusName($status)
    {
        $statuses = [
            'paid' => 'مدفوع',
            'partial' => 'مدفوع جزئياً',
            'unpaid' => 'غير مدفوع',
            'overdue' => 'متأخر',
        ];

        return $statuses[$status] ?? $status;
    }

    /**
     * الحصول على آخر فاتورة لطباعتها
     */
    public function getLastInvoice()
    {
        try {
            // البحث عن آخر فاتورة تم إنشاؤها
            $lastInvoice = Invoice::with(['customer', 'company', 'items.product'])
                ->orderBy('created_at', 'desc')
                ->first();

            if (!$lastInvoice) {
                return response()->json([
                    'success' => false,
                    'message' => 'لا توجد فواتير في النظام'
                ], 404);
            }

            // إعداد بيانات الفاتورة للإرجاع
            $invoiceData = [
                'id' => $lastInvoice->id,
                'invoice_number' => $lastInvoice->invoice_number,
                'type' => $lastInvoice->type,
                'issue_date' => $lastInvoice->issue_date->format('Y-m-d'),
                'due_date' => $lastInvoice->due_date ? $lastInvoice->due_date->format('Y-m-d') : null,
                'total_amount' => $lastInvoice->total_amount,
                'paid_amount' => $lastInvoice->paid_amount,
                'remaining_amount' => $lastInvoice->remaining_amount,
                'payment_status' => $lastInvoice->payment_status,
                'payment_method' => $lastInvoice->payment_method,
                'currency' => $lastInvoice->currency,
                'notes' => $lastInvoice->notes,
                'customer' => $lastInvoice->customer ? [
                    'id' => $lastInvoice->customer->id,
                    'name' => $lastInvoice->customer->name,
                    'email' => $lastInvoice->customer->email,
                    'phone' => $lastInvoice->customer->phone,
                    'customer_type' => $lastInvoice->customer->customer_type
                ] : null,
                'company' => [
                    'name' => $lastInvoice->company->name,
                    'vat_number' => $lastInvoice->company->vat_number,
                    'address' => $lastInvoice->company->address
                ],
                'items' => $lastInvoice->items->map(function ($item) {
                    return [
                        'name' => $item->name,
                        'quantity' => $item->quantity,
                        'unit_price' => $item->unit_price,
                        'line_total' => $item->line_total,
                        'tax_amount' => $item->tax_amount
                    ];
                }),
                'print_url' => route('invoices.print', $lastInvoice->id),
                'thermal_print_url' => route('invoices.thermal-print', $lastInvoice->id),
                'pdf_url' => route('invoices.pdf', $lastInvoice->id)
            ];

            return response()->json([
                'success' => true,
                'message' => 'تم العثور على آخر فاتورة بنجاح',
                'data' => $invoiceData
            ]);

        } catch (\Exception $e) {
            \Log::error('خطأ في الحصول على آخر فاتورة: ' . $e->getMessage(), [
                'error' => $e->getMessage(),
                'file' => $e->getFile(),
                'line' => $e->getLine()
            ]);

            return response()->json([
                'success' => false,
                'message' => 'حدث خطأ أثناء الحصول على آخر فاتورة'
            ], 500);
        }
    }

}
