<?php

namespace App\Http\Controllers;

use App\Models\Product;
use App\Models\InventoryMovement;
use App\Models\InventoryAdjustment;
use App\Models\InventoryAdjustmentItem;
use App\Exports\InventoryReportExport;
use Illuminate\Http\Request;
use Illuminate\Http\RedirectResponse;
use Illuminate\View\View;
use Illuminate\Support\Facades\DB;
use Maatwebsite\Excel\Facades\Excel;

class InventoryController extends Controller
{
    /**
     * عرض تقرير المخزون
     */
    public function index(Request $request): View
    {
        $query = Product::with(['inventoryMovements' => function($q) {
            $q->latest()->take(5);
        }]);

        // فلترة المنتجات منخفضة المخزون
        if ($request->filled('low_stock')) {
            $query->lowStock();
        }

        // فلترة المنتجات النافدة
        if ($request->filled('out_of_stock')) {
            $query->where('stock_quantity', '<=', 0);
        }

        $products = $query->paginate(15);

        return view('inventory.index', compact('products'));
    }

    /**
     * عرض حركات المخزون
     */
    public function movements(Request $request): View
    {
        $query = InventoryMovement::with(['product', 'user'])
            ->latest('movement_date');

        // فلترة حسب المنتج
        if ($request->filled('product_id')) {
            $query->where('product_id', $request->product_id);
        }

        // فلترة حسب النوع
        if ($request->filled('type')) {
            $query->where('type', $request->type);
        }

        // فلترة حسب التاريخ
        if ($request->filled('date_from')) {
            $query->whereDate('movement_date', '>=', $request->date_from);
        }

        if ($request->filled('date_to')) {
            $query->whereDate('movement_date', '<=', $request->date_to);
        }

        $movements = $query->paginate(20);
        $products = Product::active()->get();

        return view('inventory.movements', compact('movements', 'products'));
    }

    /**
     * عرض صفحة إنشاء تسوية مخزون
     */
    public function createAdjustment(): View
    {
        $products = Product::active()->get();
        return view('inventory.adjustments.create', compact('products'));
    }

    /**
     * حفظ تسوية المخزون
     */
    public function storeAdjustment(Request $request): RedirectResponse
    {
        $validated = $request->validate([
            'adjustment_date' => 'required|date',
            'type' => 'required|in:increase,decrease,recount',
            'reason' => 'required|string|max:255',
            'notes' => 'nullable|string',
            'items' => 'required|array|min:1',
            'items.*.product_id' => 'required|exists:products,id',
            'items.*.current_quantity' => 'required|integer|min:0',
            'items.*.adjusted_quantity' => 'required|integer|min:0',
            'items.*.unit_cost' => 'required|numeric|min:0',
            'items.*.reason' => 'nullable|string',
        ]);

        DB::transaction(function () use ($validated) {
            // إنشاء رقم التسوية
            $adjustmentNumber = 'ADJ-' . date('Y') . '-' . str_pad(
                InventoryAdjustment::whereYear('created_at', date('Y'))->count() + 1,
                6, '0', STR_PAD_LEFT
            );

            // إنشاء تسوية المخزون
            $adjustment = InventoryAdjustment::create([
                'user_id' => auth()->id(),
                'adjustment_number' => $adjustmentNumber,
                'adjustment_date' => $validated['adjustment_date'],
                'type' => $validated['type'],
                'reason' => $validated['reason'],
                'notes' => $validated['notes'],
                'status' => 'pending',
            ]);

            // إنشاء عناصر التسوية
            foreach ($validated['items'] as $item) {
                $quantityDifference = $item['adjusted_quantity'] - $item['current_quantity'];

                if ($quantityDifference != 0) {
                    InventoryAdjustmentItem::create([
                        'inventory_adjustment_id' => $adjustment->id,
                        'product_id' => $item['product_id'],
                        'current_quantity' => $item['current_quantity'],
                        'adjusted_quantity' => $item['adjusted_quantity'],
                        'quantity_difference' => $quantityDifference,
                        'unit_cost' => $item['unit_cost'],
                        'reason' => $item['reason'],
                    ]);
                }
            }
        });

        return redirect()->route('inventory.adjustments.index')
            ->with('success', 'تم إنشاء تسوية المخزون بنجاح');
    }

    /**
     * عرض قائمة تسويات المخزون
     */
    public function adjustments(Request $request): View
    {
        $query = InventoryAdjustment::with(['user'])
            ->latest();

        // فلترة حسب الحالة
        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        // فلترة حسب النوع
        if ($request->filled('type')) {
            $query->where('type', $request->type);
        }

        $adjustments = $query->paginate(15);

        return view('inventory.adjustments.index', compact('adjustments'));
    }

    /**
     * عرض تفاصيل تسوية المخزون
     */
    public function showAdjustment(InventoryAdjustment $adjustment): View
    {
        $adjustment->load(['user', 'items.product']);

        return view('inventory.adjustments.show', compact('adjustment'));
    }

    /**
     * اعتماد تسوية المخزون
     */
    public function approveAdjustment(InventoryAdjustment $adjustment): RedirectResponse
    {
        if ($adjustment->status !== 'pending') {
            return redirect()->back()
                ->with('error', 'لا يمكن اعتماد تسوية المخزون في هذه الحالة');
        }

        DB::transaction(function () use ($adjustment) {
            $adjustment->update(['status' => 'approved']);

            // تطبيق التسويات على المخزون
            foreach ($adjustment->items as $item) {
                $product = $item->product;
                
                // تحديث كمية المخزون
                $product->update(['stock_quantity' => $item->adjusted_quantity]);

                // إنشاء حركة مخزون
                InventoryMovement::create([
                    'product_id' => $product->id,
                    'user_id' => auth()->id(),
                    'type' => $item->quantity_difference > 0 ? 'adjustment_in' : 'adjustment_out',
                    'quantity' => abs($item->quantity_difference),
                    'unit_cost' => $item->unit_cost,
                    'total_cost' => abs($item->quantity_difference) * $item->unit_cost,
                    'reference_type' => InventoryAdjustment::class,
                    'reference_id' => $adjustment->id,
                    'notes' => 'تسوية مخزون: ' . $adjustment->reason,
                    'movement_date' => $adjustment->adjustment_date,
                ]);
            }
        });

        return redirect()->back()
            ->with('success', 'تم اعتماد تسوية المخزون وتحديث المخزون بنجاح');
    }

    /**
     * تقرير قيمة المخزون
     */
    public function valuation(): View
    {
        $products = Product::where('stock_quantity', '>', 0)
            ->get()
            ->map(function ($product) {
                $averageCost = $product->getAverageCost();
                $inventoryValue = $product->getCurrentInventoryValue();
                
                return [
                    'product' => $product,
                    'average_cost' => $averageCost,
                    'inventory_value' => $inventoryValue,
                ];
            });

        $totalInventoryValue = $products->sum('inventory_value');

        return view('inventory.valuation', compact('products', 'totalInventoryValue'));
    }

    /**
     * تقارير المخزون الشاملة
     */
    public function reports(Request $request): View
    {
        $dateFrom = $request->input('date_from', now()->subMonth()->format('Y-m-d'));
        $dateTo = $request->input('date_to', now()->format('Y-m-d'));
        $category = $request->input('category');

        // حركات المخزون
        $movementsQuery = InventoryMovement::with(['product', 'user'])
            ->whereDate('movement_date', '>=', $dateFrom)
            ->whereDate('movement_date', '<=', $dateTo)
            ->latest('movement_date');

        if ($category) {
            $movementsQuery->whereHas('product', function($q) use ($category) {
                $q->where('category', $category);
            });
        }

        $movements = $movementsQuery->get();

        // تقرير حالة المخزون
        $stockQuery = Product::query();
        if ($category) {
            $stockQuery->where('category', $category);
        }

        $stockReport = $stockQuery->get()->map(function($product) {
            return [
                'product' => $product,
                'average_cost' => $product->getAverageCost(),
                'inventory_value' => $product->getCurrentInventoryValue(),
            ];
        });

        // بيانات الملخص
        $summaryData = [
            'total_products' => $stockReport->count(),
            'total_movements' => $movements->count(),
            'total_quantity_in' => $movements->where('type', 'LIKE', '%_in')->sum('quantity'),
            'total_value' => $stockReport->sum('inventory_value'),
        ];

        // جميع التصنيفات
        $categories = Product::distinct()->pluck('category')->filter()->sort();

        return view('inventory.reports', compact(
            'movements',
            'stockReport',
            'summaryData',
            'categories'
        ));
    }

    /**
     * تصدير تقارير المخزون إلى Excel
     */
    public function exportReport(Request $request)
    {
        $type = $request->input('type', 'movements'); // movements أو stock
        $dateFrom = $request->input('date_from');
        $dateTo = $request->input('date_to');
        $category = $request->input('category');

        $filename = 'inventory-' . $type . '-report-' . date('Y-m-d') . '.xlsx';

        return Excel::download(
            new InventoryReportExport($type, $dateFrom, $dateTo, $category),
            $filename
        );
    }
}