<?php

namespace App\Http\Controllers;

use App\Models\GoodsReceipt;
use App\Models\GoodsReceiptItem;
use App\Models\PurchaseOrder;
use App\Models\InventoryMovement;
use Illuminate\Http\Request;
use Illuminate\Http\RedirectResponse;
use Illuminate\View\View;
use Illuminate\Support\Facades\DB;

class GoodsReceiptController extends Controller
{
    /**
     * عرض قائمة استلام البضائع
     */
    public function index(Request $request): View
    {
        $query = GoodsReceipt::with(['purchaseOrder.supplier', 'user'])
            ->latest();

        // فلترة حسب الحالة
        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        $receipts = $query->paginate(15);

        return view('goods-receipts.index', compact('receipts'));
    }

    /**
     * عرض نموذج إنشاء استلام بضائع جديد
     */
    public function create(Request $request): View
    {
        $purchaseOrderId = $request->get('purchase_order_id');
        
        if ($purchaseOrderId) {
            $purchaseOrder = PurchaseOrder::with(['supplier', 'items.product'])
                ->findOrFail($purchaseOrderId);
                
            return view('goods-receipts.create', compact('purchaseOrder'));
        }

        $purchaseOrders = PurchaseOrder::whereIn('status', ['approved', 'sent', 'partially_received'])
            ->with('supplier')
            ->get();

        return view('goods-receipts.create', compact('purchaseOrders'));
    }

    /**
     * حفظ استلام البضائع
     */
    public function store(Request $request): RedirectResponse
    {
        $validated = $request->validate([
            'purchase_order_id' => 'required|exists:purchase_orders,id',
            'receipt_date' => 'required|date',
            'notes' => 'nullable|string',
            'items' => 'required|array|min:1',
            'items.*.purchase_order_item_id' => 'required|exists:purchase_order_items,id',
            'items.*.received_quantity' => 'required|integer|min:0',
            'items.*.unit_cost' => 'required|numeric|min:0',
        ]);

        DB::transaction(function () use ($validated) {
            $purchaseOrder = PurchaseOrder::findOrFail($validated['purchase_order_id']);

            // إنشاء رقم الاستلام
            $receiptNumber = 'GR-' . date('Y') . '-' . str_pad(
                GoodsReceipt::whereYear('created_at', date('Y'))->count() + 1,
                6, '0', STR_PAD_LEFT
            );

            // إنشاء استلام البضائع
            $receipt = GoodsReceipt::create([
                'purchase_order_id' => $validated['purchase_order_id'],
                'user_id' => auth()->id(),
                'receipt_number' => $receiptNumber,
                'receipt_date' => $validated['receipt_date'],
                'notes' => $validated['notes'],
                'status' => 'pending',
            ]);

            // إنشاء عناصر الاستلام
            foreach ($validated['items'] as $item) {
                if ($item['received_quantity'] > 0) {
                    $purchaseOrderItem = $purchaseOrder->items()
                        ->findOrFail($item['purchase_order_item_id']);

                    GoodsReceiptItem::create([
                        'goods_receipt_id' => $receipt->id,
                        'purchase_order_item_id' => $item['purchase_order_item_id'],
                        'product_id' => $purchaseOrderItem->product_id,
                        'ordered_quantity' => $purchaseOrderItem->quantity,
                        'received_quantity' => $item['received_quantity'],
                        'unit_cost' => $item['unit_cost'],
                        'total_cost' => $item['received_quantity'] * $item['unit_cost'],
                    ]);
                }
            }
        });

        return redirect()->route('goods-receipts.index')
            ->with('success', 'تم إنشاء استلام البضائع بنجاح');
    }

    /**
     * عرض تفاصيل استلام البضائع
     */
    public function show(GoodsReceipt $goodsReceipt): View
    {
        $goodsReceipt->load([
            'purchaseOrder.supplier',
            'user',
            'items.product',
            'items.purchaseOrderItem'
        ]);

        return view('goods-receipts.show', compact('goodsReceipt'));
    }

    /**
     * تأكيد استلام البضائع
     */
    public function confirm(GoodsReceipt $goodsReceipt): RedirectResponse
    {
        if ($goodsReceipt->status !== 'pending') {
            return redirect()->back()
                ->with('error', 'لا يمكن تأكيد استلام البضائع في هذه الحالة');
        }

        DB::transaction(function () use ($goodsReceipt) {
            $goodsReceipt->update(['status' => 'confirmed']);

            // تحديث المخزون وإنشاء حركات المخزون
            foreach ($goodsReceipt->items as $item) {
                if ($item->received_quantity > 0) {
                    $product = $item->product;
                    
                    // تحديث كمية المخزون
                    $product->increment('stock_quantity', $item->received_quantity);

                    // إنشاء حركة مخزون
                    InventoryMovement::create([
                        'product_id' => $product->id,
                        'user_id' => auth()->id(),
                        'type' => 'purchase',
                        'quantity' => $item->received_quantity,
                        'unit_cost' => $item->unit_cost,
                        'total_cost' => $item->total_cost,
                        'reference_type' => GoodsReceipt::class,
                        'reference_id' => $goodsReceipt->id,
                        'notes' => 'استلام بضائع - ' . $goodsReceipt->receipt_number,
                        'movement_date' => $goodsReceipt->receipt_date,
                    ]);
                }
            }

            // تحديث حالة أمر الشراء
            $this->updatePurchaseOrderStatus($goodsReceipt->purchaseOrder);
        });

        return redirect()->back()
            ->with('success', 'تم تأكيد استلام البضائع وتحديث المخزون بنجاح');
    }

    /**
     * إلغاء استلام البضائع
     */
    public function cancel(GoodsReceipt $goodsReceipt): RedirectResponse
    {
        if ($goodsReceipt->status === 'confirmed') {
            return redirect()->back()
                ->with('error', 'لا يمكن إلغاء استلام البضائع المؤكد');
        }

        $goodsReceipt->update(['status' => 'cancelled']);

        return redirect()->back()
            ->with('success', 'تم إلغاء استلام البضائع');
    }

    /**
     * تحديث حالة أمر الشراء بناءً على الاستلام
     */
    private function updatePurchaseOrderStatus(PurchaseOrder $purchaseOrder): void
    {
        $totalOrdered = $purchaseOrder->items->sum('quantity');
        $totalReceived = $purchaseOrder->receipts()
            ->where('status', 'confirmed')
            ->get()
            ->sum(function ($receipt) {
                return $receipt->items->sum('received_quantity');
            });

        if ($totalReceived >= $totalOrdered) {
            $purchaseOrder->update(['status' => 'received']);
        } elseif ($totalReceived > 0) {
            $purchaseOrder->update(['status' => 'partially_received']);
        }
    }

    /**
     * حذف استلام البضائع
     */
    public function destroy(GoodsReceipt $goodsReceipt): RedirectResponse
    {
        if ($goodsReceipt->status === 'confirmed') {
            return redirect()->route('goods-receipts.index')
                ->with('error', 'لا يمكن حذف استلام البضائع المؤكد');
        }

        $goodsReceipt->delete();

        return redirect()->route('goods-receipts.index')
            ->with('success', 'تم حذف استلام البضائع بنجاح');
    }
}