<?php

namespace App\Http\Controllers;

use App\Models\FiscalYear;
use App\Models\Account;
use App\Models\AccountBalance;
use App\Services\AccountingService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;

class FiscalYearController extends Controller
{
    protected $accountingService;

    public function __construct(AccountingService $accountingService)
    {
        $this->middleware('auth');
        $this->middleware('accounting.permissions:view_fiscal_years')->only(['index', 'show']);
        $this->middleware('accounting.permissions:create_fiscal_years')->only(['create', 'store']);
        $this->middleware('accounting.permissions:edit_fiscal_years')->only(['edit', 'update']);
        $this->middleware('accounting.permissions:delete_fiscal_years')->only(['destroy']);
        $this->middleware('accounting.permissions:close_fiscal_years')->only(['close', 'reopen']);
        
        $this->accountingService = $accountingService;
    }

    /**
     * Display a listing of fiscal years
     */
    public function index(Request $request)
    {
        $query = FiscalYear::query();

        // البحث
        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('name', 'like', "%{$search}%")
                  ->orWhere('start_date', 'like', "%{$search}%")
                  ->orWhere('end_date', 'like', "%{$search}%");
            });
        }

        // فلترة حسب الحالة
        if ($request->filled('status')) {
            switch ($request->status) {
                case 'current':
                    $query->current();
                    break;
                case 'open':
                    $query->open();
                    break;
                case 'closed':
                    $query->closed();
                    break;
            }
        }

        $fiscalYears = $query->orderBy('start_date', 'desc')->paginate(15);

        // إحصائيات
        $stats = [
            'total' => FiscalYear::count(),
            'current' => FiscalYear::current()->count(),
            'open' => FiscalYear::open()->count(),
            'closed' => FiscalYear::closed()->count(),
        ];

        return view('fiscal-years.index', compact('fiscalYears', 'stats'));
    }

    /**
     * Show the form for creating a new fiscal year
     */
    public function create()
    {
        // الحصول على آخر سنة مالية لتحديد التواريخ المقترحة
        $lastFiscalYear = FiscalYear::orderBy('end_date', 'desc')->first();
        
        $suggestedStartDate = $lastFiscalYear 
            ? $lastFiscalYear->end_date->addDay()
            : now()->startOfYear();
            
        $suggestedEndDate = $suggestedStartDate->copy()->addYear()->subDay();

        return view('fiscal-years.create', compact('suggestedStartDate', 'suggestedEndDate'));
    }

    /**
     * Store a newly created fiscal year
     */
    public function store(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:100|unique:fiscal_years',
            'start_date' => 'required|date',
            'end_date' => 'required|date|after:start_date',
            'is_current' => 'boolean',
            'opening_balance_total' => 'nullable|numeric',
            'notes' => 'nullable|string|max:1000',
        ]);

        DB::beginTransaction();
        
        try {
            // التحقق من عدم التداخل مع سنوات مالية أخرى
            $overlapping = FiscalYear::where(function ($query) use ($request) {
                $query->whereBetween('start_date', [$request->start_date, $request->end_date])
                      ->orWhereBetween('end_date', [$request->start_date, $request->end_date])
                      ->orWhere(function ($q) use ($request) {
                          $q->where('start_date', '<=', $request->start_date)
                            ->where('end_date', '>=', $request->end_date);
                      });
            })->exists();

            if ($overlapping) {
                return back()->withErrors(['date_overlap' => 'تتداخل السنة المالية مع سنة مالية أخرى موجودة'])
                            ->withInput();
            }

            $fiscalYear = FiscalYear::create([
                'name' => $request->name,
                'start_date' => $request->start_date,
                'end_date' => $request->end_date,
                'is_current' => $request->boolean('is_current', false),
                'opening_balance_total' => $request->opening_balance_total ?? 0,
                'notes' => $request->notes,
            ]);

            // إنشاء الأرصدة الافتتاحية إذا كانت محددة
            if ($request->filled('create_opening_balances')) {
                $this->createOpeningBalances($fiscalYear);
            }

            DB::commit();

            return redirect()->route('fiscal-years.index')
                           ->with('success', 'تم إنشاء السنة المالية بنجاح');

        } catch (\Exception $e) {
            DB::rollback();
            return back()->withErrors(['error' => 'حدث خطأ: ' . $e->getMessage()])
                        ->withInput();
        }
    }

    /**
     * Display the specified fiscal year
     */
    public function show(FiscalYear $fiscalYear)
    {
        $fiscalYear->load(['closedBy']);

        // إحصائيات السنة المالية
        $stats = [
            'total_accounts' => Account::count(),
            'journal_entries' => $fiscalYear->journalEntries()->count(),
            'approved_entries' => $fiscalYear->journalEntries()->where('is_approved', true)->count(),
            'budgets' => $fiscalYear->budgets()->count(),
            'account_balances' => $fiscalYear->accountBalances()->count(),
        ];

        // الأرصدة الحالية
        $balances = AccountBalance::where('fiscal_year_id', $fiscalYear->id)
            ->with('account')
            ->orderBy('balance_date', 'desc')
            ->limit(10)
            ->get();

        // الميزانيات النشطة
        $budgets = $fiscalYear->budgets()
            ->where('is_active', true)
            ->with(['costCenter', 'account'])
            ->orderBy('variance_percentage', 'desc')
            ->limit(5)
            ->get();

        return view('fiscal-years.show', compact('fiscalYear', 'stats', 'balances', 'budgets'));
    }

    /**
     * Show the form for editing the fiscal year
     */
    public function edit(FiscalYear $fiscalYear)
    {
        // التحقق من إمكانية التعديل
        if ($fiscalYear->is_closed) {
            return redirect()->route('fiscal-years.index')
                           ->with('error', 'لا يمكن تعديل سنة مالية مغلقة');
        }

        return view('fiscal-years.edit', compact('fiscalYear'));
    }

    /**
     * Update the specified fiscal year
     */
    public function update(Request $request, FiscalYear $fiscalYear)
    {
        // التحقق من إمكانية التعديل
        if ($fiscalYear->is_closed) {
            return redirect()->route('fiscal-years.index')
                           ->with('error', 'لا يمكن تعديل سنة مالية مغلقة');
        }

        $request->validate([
            'name' => 'required|string|max:100|unique:fiscal_years,name,' . $fiscalYear->id,
            'start_date' => 'required|date',
            'end_date' => 'required|date|after:start_date',
            'is_current' => 'boolean',
            'opening_balance_total' => 'nullable|numeric',
            'notes' => 'nullable|string|max:1000',
        ]);

        DB::beginTransaction();
        
        try {
            // التحقق من عدم التداخل (باستثناء السنة الحالية)
            $overlapping = FiscalYear::where('id', '!=', $fiscalYear->id)
                ->where(function ($query) use ($request) {
                    $query->whereBetween('start_date', [$request->start_date, $request->end_date])
                          ->orWhereBetween('end_date', [$request->start_date, $request->end_date])
                          ->orWhere(function ($q) use ($request) {
                              $q->where('start_date', '<=', $request->start_date)
                                ->where('end_date', '>=', $request->end_date);
                          });
                })->exists();

            if ($overlapping) {
                return back()->withErrors(['date_overlap' => 'تتداخل السنة المالية مع سنة مالية أخرى موجودة'])
                            ->withInput();
            }

            $fiscalYear->update([
                'name' => $request->name,
                'start_date' => $request->start_date,
                'end_date' => $request->end_date,
                'is_current' => $request->boolean('is_current', false),
                'opening_balance_total' => $request->opening_balance_total ?? 0,
                'notes' => $request->notes,
            ]);

            DB::commit();

            return redirect()->route('fiscal-years.show', $fiscalYear)
                           ->with('success', 'تم تحديث السنة المالية بنجاح');

        } catch (\Exception $e) {
            DB::rollback();
            return back()->withErrors(['error' => 'حدث خطأ: ' . $e->getMessage()])
                        ->withInput();
        }
    }

    /**
     * Close the fiscal year
     */
    public function close(FiscalYear $fiscalYear)
    {
        if ($fiscalYear->is_closed) {
            return redirect()->route('fiscal-years.index')
                           ->with('error', 'السنة المالية مغلقة مسبقاً');
        }

        try {
            // التحقق من إمكانية الإغلاق
            if (!$fiscalYear->canBeClosed()) {
                return redirect()->route('fiscal-years.show', $fiscalYear)
                               ->with('error', 'لا يمكن إغلاق السنة المالية. تأكد من اعتماد جميع القيود');
            }

            DB::beginTransaction();

            // حساب الأرصدة الختامية
            AccountBalance::calculateAllBalances($fiscalYear->id, 'closing');

            // إغلاق السنة المالية
            $fiscalYear->close(auth()->id());

            // إنشاء السنة المالية التالية إذا لم تكن موجودة
            $nextYear = FiscalYear::where('start_date', '>', $fiscalYear->end_date)
                ->orderBy('start_date')
                ->first();

            if (!$nextYear) {
                $nextYear = $fiscalYear->createNext();
            }

            DB::commit();

            return redirect()->route('fiscal-years.index')
                           ->with('success', 'تم إغلاق السنة المالية بنجاح');

        } catch (\Exception $e) {
            DB::rollback();
            return redirect()->route('fiscal-years.show', $fiscalYear)
                           ->with('error', 'حدث خطأ أثناء إغلاق السنة المالية: ' . $e->getMessage());
        }
    }

    /**
     * Reopen the fiscal year
     */
    public function reopen(FiscalYear $fiscalYear)
    {
        if (!$fiscalYear->is_closed) {
            return redirect()->route('fiscal-years.index')
                           ->with('error', 'السنة المالية غير مغلقة');
        }

        try {
            DB::beginTransaction();

            $fiscalYear->reopen();

            DB::commit();

            return redirect()->route('fiscal-years.show', $fiscalYear)
                           ->with('success', 'تم إعادة فتح السنة المالية بنجاح');

        } catch (\Exception $e) {
            DB::rollback();
            return redirect()->route('fiscal-years.show', $fiscalYear)
                           ->with('error', 'حدث خطأ أثناء إعادة فتح السنة المالية: ' . $e->getMessage());
        }
    }

    /**
     * Set as current fiscal year
     */
    public function setCurrent(FiscalYear $fiscalYear)
    {
        try {
            $fiscalYear->setCurrent();

            return redirect()->route('fiscal-years.index')
                           ->with('success', 'تم تعيين السنة المالية كحالية بنجاح');

        } catch (\Exception $e) {
            return redirect()->route('fiscal-years.index')
                           ->with('error', 'حدث خطأ: ' . $e->getMessage());
        }
    }

    /**
     * Remove the specified fiscal year
     */
    public function destroy(FiscalYear $fiscalYear)
    {
        // التحقق من إمكانية الحذف
        if ($fiscalYear->is_closed) {
            return redirect()->route('fiscal-years.index')
                           ->with('error', 'لا يمكن حذف سنة مالية مغلقة');
        }

        if ($fiscalYear->is_current) {
            return redirect()->route('fiscal-years.index')
                           ->with('error', 'لا يمكن حذف السنة المالية الحالية');
        }

        // التحقق من وجود قيود أو أرصدة
        if ($fiscalYear->journalEntries()->exists() || $fiscalYear->accountBalances()->exists()) {
            return redirect()->route('fiscal-years.index')
                           ->with('error', 'لا يمكن حذف سنة مالية تحتوي على قيود أو أرصدة');
        }

        try {
            DB::beginTransaction();

            $fiscalYear->delete();

            DB::commit();

            return redirect()->route('fiscal-years.index')
                           ->with('success', 'تم حذف السنة المالية بنجاح');

        } catch (\Exception $e) {
            DB::rollback();
            return redirect()->route('fiscal-years.index')
                           ->with('error', 'حدث خطأ أثناء حذف السنة المالية: ' . $e->getMessage());
        }
    }

    /**
     * Show closing report
     */
    public function closingReport(FiscalYear $fiscalYear)
    {
        if (!$fiscalYear->is_closed) {
            return redirect()->route('fiscal-years.show', $fiscalYear)
                           ->with('error', 'لا يمكن عرض تقرير الإغلاق لسنة مالية غير مغلقة');
        }

        // الأرصدة الختامية
        $closingBalances = AccountBalance::where('fiscal_year_id', $fiscalYear->id)
            ->where('balance_type', 'closing')
            ->with('account')
            ->orderBy('account.code')
            ->get();

        // إجماليات حسب نوع الحساب
        $totals = $closingBalances->groupBy('account.type')
            ->map(function ($balances) {
                return [
                    'count' => $balances->count(),
                    'total_debit' => $balances->sum('debit_balance'),
                    'total_credit' => $balances->sum('credit_balance'),
                    'net_balance' => $balances->sum('net_balance'),
                ];
            });

        return view('fiscal-years.closing-report', compact('fiscalYear', 'closingBalances', 'totals'));
    }

    /**
     * Create opening balances for the fiscal year
     */
    private function createOpeningBalances(FiscalYear $fiscalYear)
    {
        $accounts = Account::where('is_active', true)->get();

        foreach ($accounts as $account) {
            AccountBalance::createOpeningBalance(
                $account->id,
                $fiscalYear->id,
                $account->opening_balance ?? 0
            );
        }
    }
}