<?php

namespace App\Http\Controllers;

use App\Models\Account;
use App\Models\JournalEntry;
use App\Models\JournalEntryLine;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Maatwebsite\Excel\Facades\Excel;
use App\Exports\TrialBalanceExport;
use App\Exports\BalanceSheetExport;
use App\Exports\IncomeStatementExport;

class FinancialReportController extends Controller
{
    public function trialBalance(Request $request)
    {
        $endDate = $request->end_date ?? now()->format('Y-m-d');
        $startDate = $request->start_date ?? now()->startOfYear()->format('Y-m-d');

        $accounts = Account::with(['journalEntryLines' => function ($query) use ($startDate, $endDate) {
            $query->whereHas('journalEntry', function ($q) use ($startDate, $endDate) {
                $q->whereBetween('entry_date', [$startDate, $endDate])
                  ->where('status', 'posted');
            });
        }])->where('is_active', true)->orderBy('code')->get();

        $trialBalanceData = [];
        $totalDebits = 0;
        $totalCredits = 0;

        foreach ($accounts as $account) {
            $debitTotal = $account->journalEntryLines->sum('debit_amount');
            $creditTotal = $account->journalEntryLines->sum('credit_amount');
            
            if ($debitTotal > 0 || $creditTotal > 0 || $account->opening_balance != 0) {
                $balance = $account->opening_balance;
                
                if ($account->debit_credit === 'debit') {
                    $balance += $debitTotal - $creditTotal;
                    $debitBalance = $balance > 0 ? $balance : 0;
                    $creditBalance = $balance < 0 ? abs($balance) : 0;
                } else {
                    $balance += $creditTotal - $debitTotal;
                    $creditBalance = $balance > 0 ? $balance : 0;
                    $debitBalance = $balance < 0 ? abs($balance) : 0;
                }

                $trialBalanceData[] = [
                    'account' => $account,
                    'debit_balance' => $debitBalance,
                    'credit_balance' => $creditBalance,
                    'debit_movements' => $debitTotal,
                    'credit_movements' => $creditTotal
                ];

                $totalDebits += $debitBalance;
                $totalCredits += $creditBalance;
            }
        }

        return view('accounting.reports.trial-balance', compact(
            'trialBalanceData', 'totalDebits', 'totalCredits', 'startDate', 'endDate'
        ));
    }

    public function balanceSheet(Request $request)
    {
        $asOfDate = $request->as_of_date ?? now()->format('Y-m-d');

        // Assets
        $assets = $this->getAccountBalances('asset', $asOfDate);
        $currentAssets = $assets->where('account.subtype', 'current_asset');
        $fixedAssets = $assets->where('account.subtype', 'fixed_asset');

        // Liabilities
        $liabilities = $this->getAccountBalances('liability', $asOfDate);
        $currentLiabilities = $liabilities->where('account.subtype', 'current_liability');
        $longTermLiabilities = $liabilities->where('account.subtype', 'long_term_liability');

        // Equity
        $equityAccounts = $this->getAccountBalances('equity', $asOfDate);

        // Calculate retained earnings
        $retainedEarnings = $this->calculateNetIncome($asOfDate);

        // Calculate totals
        $totalCurrentAssets = $currentAssets->sum('balance');
        $totalFixedAssets = $fixedAssets->sum('balance');
        $totalAssets = $totalCurrentAssets + $totalFixedAssets;

        $totalCurrentLiabilities = $currentLiabilities->sum('balance');
        $totalLongTermLiabilities = $longTermLiabilities->sum('balance');
        $totalLiabilities = $totalCurrentLiabilities + $totalLongTermLiabilities;

        $totalEquity = $equityAccounts->sum('balance') + $retainedEarnings;
        $totalLiabilitiesAndEquity = $totalLiabilities + $totalEquity;

        return view('accounting.reports.balance-sheet', compact(
            'currentAssets', 'fixedAssets', 'currentLiabilities',
            'longTermLiabilities', 'equityAccounts', 'retainedEarnings', 'asOfDate',
            'totalCurrentAssets', 'totalFixedAssets', 'totalAssets',
            'totalCurrentLiabilities', 'totalLongTermLiabilities', 'totalLiabilities',
            'totalEquity', 'totalLiabilitiesAndEquity'
        ));
    }

    public function incomeStatement(Request $request)
    {
        $startDate = $request->start_date ?? now()->startOfYear()->format('Y-m-d');
        $endDate = $request->end_date ?? now()->format('Y-m-d');

        // Revenue Accounts
        $revenueAccounts = $this->getAccountBalancesForPeriod('revenue', $startDate, $endDate)
            ->map(function($item) {
                return [
                    'id' => $item['id'],
                    'name' => $item['name'],
                    'amount' => $item['balance']
                ];
            });

        // COGS Accounts
        $cogsAccounts = $this->getAccountBalancesForPeriod('expense', $startDate, $endDate)
            ->filter(function($item) {
                return isset($item['account']) && $item['account']->subtype === 'cost_of_goods_sold';
            })
            ->map(function($item) {
                return [
                    'id' => $item['id'],
                    'name' => $item['name'],
                    'amount' => $item['balance']
                ];
            });

        // Expense Accounts (excluding COGS)
        $expenseAccounts = $this->getAccountBalancesForPeriod('expense', $startDate, $endDate)
            ->filter(function($item) {
                return !isset($item['account']) || $item['account']->subtype !== 'cost_of_goods_sold';
            })
            ->map(function($item) {
                return [
                    'id' => $item['id'],
                    'name' => $item['name'],
                    'amount' => $item['balance']
                ];
            });

        // Calculate totals
        $totalRevenue = $revenueAccounts->sum('amount');
        $totalCOGS = $cogsAccounts->sum('amount');
        $totalExpenses = $expenseAccounts->sum('amount');
        $grossProfit = $totalRevenue - $totalCOGS;
        $netIncome = $grossProfit - $totalExpenses;

        // Calculate profit margin
        $profitMargin = $totalRevenue > 0 ? ($netIncome / $totalRevenue) * 100 : 0;

        return view('accounting.reports.income-statement', compact(
            'revenueAccounts', 'cogsAccounts', 'expenseAccounts',
            'totalRevenue', 'totalCOGS', 'totalExpenses', 'grossProfit',
            'netIncome', 'profitMargin', 'startDate', 'endDate'
        ));
    }

    public function generalLedger(Request $request)
    {
        $startDate = $request->start_date ?? now()->startOfMonth()->format('Y-m-d');
        $endDate = $request->end_date ?? now()->endOfMonth()->format('Y-m-d');
        $accountId = $request->account_id;

        $accounts = Account::where('is_active', true)->orderBy('code')->get();
        $ledgerData = [];
        $totalDebits = 0;
        $totalCredits = 0;
        $totalEntries = 0;

        // Build query for accounts to include
        $accountsToProcess = $accountId ? Account::where('id', $accountId)->get() :
                           Account::where('is_active', true)->orderBy('code')->get();

        foreach ($accountsToProcess as $account) {
            $entries = $account->journalEntryLines()
                ->with(['journalEntry'])
                ->whereHas('journalEntry', function ($query) use ($startDate, $endDate) {
                    $query->whereBetween('entry_date', [$startDate, $endDate])
                          ->where('status', 'posted');
                })
                ->get()
                ->map(function($line) {
                    return (object)[
                        'date' => $line->journalEntry->entry_date,
                        'journal_entry_id' => $line->journalEntry->id,
                        'reference' => $line->journalEntry->reference_number,
                        'description' => $line->journalEntry->description,
                        'debit_amount' => $line->debit_amount,
                        'credit_amount' => $line->credit_amount,
                    ];
                })
                ->sortBy('date');

            // Skip accounts with no entries unless they have opening balance
            if ($entries->isEmpty() && $account->opening_balance == 0) {
                continue;
            }

            $accountDebits = $entries->sum('debit_amount');
            $accountCredits = $entries->sum('credit_amount');
            
            // Calculate closing balance
            $openingBalance = $account->opening_balance;
            $closingBalance = $openingBalance;
            
            if ($account->debit_credit === 'debit') {
                $closingBalance += $accountDebits - $accountCredits;
            } else {
                $closingBalance += $accountCredits - $accountDebits;
            }

            $ledgerData[] = [
                'account' => $account,
                'entries' => $entries,
                'opening_balance' => $openingBalance,
                'closing_balance' => $closingBalance,
                'total_debits' => $accountDebits,
                'total_credits' => $accountCredits,
            ];

            $totalDebits += $accountDebits;
            $totalCredits += $accountCredits;
            $totalEntries += $entries->count();
        }

        return view('accounting.reports.general-ledger', compact(
            'accounts', 'ledgerData', 'startDate', 'endDate', 'accountId',
            'totalDebits', 'totalCredits', 'totalEntries'
        ));
    }

    public function cashFlow(Request $request)
    {
        $startDate = $request->start_date ?? now()->startOfYear()->format('Y-m-d');
        $endDate = $request->end_date ?? now()->format('Y-m-d');

        // Operating Activities
        $operatingActivities = [
            ['description' => 'الإيرادات النقدية من المبيعات', 'amount' => 0],
            ['description' => 'المدفوعات النقدية للموردين', 'amount' => 0],
            ['description' => 'المدفوعات النقدية للموظفين', 'amount' => 0],
        ];
        $operatingCashFlow = collect($operatingActivities)->sum('amount');
        
        // Investing Activities
        $investingActivities = [
            ['description' => 'شراء الأصول الثابتة', 'amount' => 0],
            ['description' => 'بيع الأصول الثابتة', 'amount' => 0],
        ];
        $investingCashFlow = collect($investingActivities)->sum('amount');
        
        // Financing Activities
        $financingActivities = [
            ['description' => 'القروض المحصلة', 'amount' => 0],
            ['description' => 'سداد القروض', 'amount' => 0],
            ['description' => 'رأس المال المدفوع', 'amount' => 0],
        ];
        $financingCashFlow = collect($financingActivities)->sum('amount');

        $netCashFlow = $operatingCashFlow + $investingCashFlow + $financingCashFlow;
        
        // Calculate cash balances
        $beginningCashBalance = 0; // This should be calculated from actual cash accounts
        $endingCashBalance = $beginningCashBalance + $netCashFlow;

        return view('accounting.reports.cash-flow', compact(
            'operatingActivities', 'investingActivities', 'financingActivities',
            'operatingCashFlow', 'investingCashFlow', 'financingCashFlow',
            'netCashFlow', 'beginningCashBalance', 'endingCashBalance',
            'startDate', 'endDate'
        ));
    }

    public function exportTrialBalance(Request $request)
    {
        $endDate = $request->end_date ?? now()->format('Y-m-d');
        return Excel::download(new TrialBalanceExport($endDate), 'trial-balance.xlsx');
    }

    public function exportBalanceSheet(Request $request)
    {
        $asOfDate = $request->as_of_date ?? now()->format('Y-m-d');
        return Excel::download(new BalanceSheetExport($asOfDate), 'balance-sheet.xlsx');
    }

    public function exportIncomeStatement(Request $request)
    {
        $startDate = $request->start_date ?? now()->startOfYear()->format('Y-m-d');
        $endDate = $request->end_date ?? now()->format('Y-m-d');
        return Excel::download(new IncomeStatementExport($startDate, $endDate), 'income-statement.xlsx');
    }

    public function exportCashFlow(Request $request)
    {
        $startDate = $request->start_date ?? now()->startOfYear()->format('Y-m-d');
        $endDate = $request->end_date ?? now()->format('Y-m-d');
        
        // For now, return a simple CSV file. A proper Excel export class can be created later
        $filename = 'cash-flow-' . $startDate . '-to-' . $endDate . '.csv';
        
        $headers = [
            'Content-Type' => 'text/csv',
            'Content-Disposition' => 'attachment; filename="' . $filename . '"',
        ];
        
        $callback = function() use ($startDate, $endDate) {
            $file = fopen('php://output', 'w');
            
            // Add headers
            fputcsv($file, ['تقرير التدفق النقدي']);
            fputcsv($file, ['من تاريخ: ' . $startDate . ' إلى تاريخ: ' . $endDate]);
            fputcsv($file, []);
            fputcsv($file, ['البند', 'المبلغ']);
            
            // Calculate cash flow data
            $operatingCashFlow = $this->calculateOperatingCashFlow($startDate, $endDate);
            $investingCashFlow = $this->calculateInvestingCashFlow($startDate, $endDate);
            $financingCashFlow = $this->calculateFinancingCashFlow($startDate, $endDate);
            $netCashFlow = $operatingCashFlow + $investingCashFlow + $financingCashFlow;
            
            // Add data rows
            fputcsv($file, ['الأنشطة التشغيلية', number_format($operatingCashFlow, 2)]);
            fputcsv($file, ['الأنشطة الاستثمارية', number_format($investingCashFlow, 2)]);
            fputcsv($file, ['الأنشطة التمويلية', number_format($financingCashFlow, 2)]);
            fputcsv($file, []);
            fputcsv($file, ['صافي التدفق النقدي', number_format($netCashFlow, 2)]);
            
            fclose($file);
        };
        
        return response()->stream($callback, 200, $headers);
    }

    public function exportGeneralLedger(Request $request)
    {
        $startDate = $request->start_date ?? now()->startOfMonth()->format('Y-m-d');
        $endDate = $request->end_date ?? now()->endOfMonth()->format('Y-m-d');
        $accountId = $request->account_id;
        
        $filename = 'general-ledger-' . $startDate . '-to-' . $endDate . '.csv';
        
        $headers = [
            'Content-Type' => 'text/csv',
            'Content-Disposition' => 'attachment; filename="' . $filename . '"',
        ];
        
        $callback = function() use ($startDate, $endDate, $accountId) {
            $file = fopen('php://output', 'w');
            
            // Add headers
            fputcsv($file, ['تقرير دفتر الأستاذ العام']);
            fputcsv($file, ['من تاريخ: ' . $startDate . ' إلى تاريخ: ' . $endDate]);
            
            if ($accountId) {
                $account = Account::findOrFail($accountId);
                fputcsv($file, ['الحساب: ' . $account->name . ' (' . $account->code . ')']);
                fputcsv($file, []);
                fputcsv($file, ['التاريخ', 'البيان', 'المرجع', 'مدين', 'دائن', 'الرصيد']);
                
                $transactions = $account->journalEntryLines()
                    ->with(['journalEntry'])
                    ->whereHas('journalEntry', function ($query) use ($startDate, $endDate) {
                        $query->whereBetween('entry_date', [$startDate, $endDate])
                              ->where('status', 'posted');
                    })
                    ->orderBy('created_at')
                    ->get();

                $runningBalance = $account->opening_balance;
                
                foreach ($transactions as $transaction) {
                    if ($account->debit_credit === 'debit') {
                        $runningBalance += $transaction->debit_amount - $transaction->credit_amount;
                    } else {
                        $runningBalance += $transaction->credit_amount - $transaction->debit_amount;
                    }

                    fputcsv($file, [
                        $transaction->journalEntry->entry_date,
                        $transaction->journalEntry->description,
                        $transaction->journalEntry->reference_number,
                        number_format($transaction->debit_amount, 2),
                        number_format($transaction->credit_amount, 2),
                        number_format($runningBalance, 2)
                    ]);
                }
            } else {
                fputcsv($file, ['يرجى تحديد حساب لتصدير دفتر الأستاذ']);
            }
            
            fclose($file);
        };
        
        return response()->stream($callback, 200, $headers);
    }

    private function getAccountBalances($type, $asOfDate)
    {
        return Account::where('type', $type)
            ->where('is_active', true)
            ->get()
            ->map(function ($account) use ($asOfDate) {
                // Calculate balance using journal entry lines
                $debitTotal = $account->journalEntryLines()
                    ->whereHas('journalEntry', function ($query) use ($asOfDate) {
                        $query->where('entry_date', '<=', $asOfDate)
                              ->where('status', 'posted');
                    })
                    ->sum('debit_amount');
                    
                $creditTotal = $account->journalEntryLines()
                    ->whereHas('journalEntry', function ($query) use ($asOfDate) {
                        $query->where('entry_date', '<=', $asOfDate)
                              ->where('status', 'posted');
                    })
                    ->sum('credit_amount');

                $balance = $account->opening_balance;
                if ($account->debit_credit === 'debit') {
                    $balance += $debitTotal - $creditTotal;
                } else {
                    $balance += $creditTotal - $debitTotal;
                }
                
                return [
                    'id' => $account->id,
                    'name' => $account->name,
                    'code' => $account->code,
                    'balance' => abs($balance),
                    'account' => $account,
                    'is_cash' => $account->is_cash_account ?? false,
                    'is_bank' => $account->is_bank_account ?? false
                ];
            })
            ->filter(function ($item) {
                return $item['balance'] > 0;
            });
    }

    private function getAccountBalancesForPeriod($type, $startDate, $endDate)
    {
        return Account::where('type', $type)
            ->where('is_active', true)
            ->get()
            ->map(function ($account) use ($startDate, $endDate) {
                $balance = $account->journalEntryLines()
                    ->whereHas('journalEntry', function ($query) use ($startDate, $endDate) {
                        $query->whereBetween('entry_date', [$startDate, $endDate])
                              ->where('status', 'posted');
                    })
                    ->sum('credit_amount') - $account->journalEntryLines()
                    ->whereHas('journalEntry', function ($query) use ($startDate, $endDate) {
                        $query->whereBetween('entry_date', [$startDate, $endDate])
                              ->where('status', 'posted');
                    })
                    ->sum('debit_amount');

                if ($account->debit_credit === 'debit') {
                    $balance = -$balance;
                }

                return [
                    'account' => $account,
                    'balance' => abs($balance)
                ];
            })
            ->filter(function ($item) {
                return $item['balance'] > 0;
            });
    }

    private function calculateNetIncome($asOfDate)
    {
        $revenue = $this->getAccountBalancesForPeriod('revenue', now()->startOfYear()->format('Y-m-d'), $asOfDate)->sum('balance');
        $expenses = $this->getAccountBalancesForPeriod('expense', now()->startOfYear()->format('Y-m-d'), $asOfDate)->sum('balance');
        
        return $revenue - $expenses;
    }

    private function calculateOperatingCashFlow($startDate, $endDate)
    {
        // Simplified calculation - would need more detailed implementation
        return 0;
    }

    private function calculateInvestingCashFlow($startDate, $endDate)
    {
        // Simplified calculation - would need more detailed implementation
        return 0;
    }

    private function calculateFinancingCashFlow($startDate, $endDate)
    {
        // Simplified calculation - would need more detailed implementation
        return 0;
    }
}