<?php

namespace App\Http\Controllers;

use App\Models\File;
use App\Models\Folder;
use Illuminate\Http\Request;
use Illuminate\Http\Response;
use Illuminate\Http\RedirectResponse;
use Illuminate\View\View;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Str;
use ZipArchive;

class FileManagerController extends Controller
{
    /**
     * عرض صفحة إدارة الملفات الرئيسية
     */
    public function index(Request $request): View
    {
        $folderId = $request->get('folder');
        $search = $request->get('search');
        $category = $request->get('category');
        $type = $request->get('type');

        // الحصول على المجلد الحالي
        $currentFolder = null;
        if ($folderId) {
            $currentFolder = Folder::findOrFail($folderId);
        }

        // الحصول على المجلدات الفرعية
        $folders = Folder::when($currentFolder, function ($q) use ($currentFolder) {
            $q->where('parent_id', $currentFolder->id);
        }, function ($q) {
            $q->whereNull('parent_id');
        })->ordered()->get();

        // الحصول على الملفات
        $filesQuery = File::with(['uploader', 'folder'])
            ->when($currentFolder, function ($q) use ($currentFolder) {
                $q->where('folder_id', $currentFolder->id);
            }, function ($q) {
                $q->whereNull('folder_id');
            })
            ->when($search, function ($q) use ($search) {
                $q->search($search);
            })
            ->when($category, function ($q) use ($category) {
                $q->inCategory($category);
            })
            ->when($type, function ($q) use ($type) {
                $q->ofType($type);
            })
            ->active()
            ->notExpired()
            ->latest();

        $files = $filesQuery->paginate(20)->appends($request->query());

        // إحصائيات
        $stats = [
            'total_files' => File::active()->count(),
            'total_size' => File::active()->sum('size'),
            'total_folders' => Folder::count(),
            'categories' => File::active()->distinct('category')->pluck('category'),
        ];

        return view('file-manager.index', compact(
            'files',
            'folders',
            'currentFolder',
            'stats'
        ));
    }

    /**
     * رفع ملف جديد
     */
    public function upload(Request $request): RedirectResponse
    {
        $validator = Validator::make($request->all(), [
            'files.*' => 'required|file|max:51200', // 50MB
            'folder_id' => 'nullable|exists:folders,id',
            'category' => 'nullable|string|max:50',
            'description' => 'nullable|string|max:500',
            'is_public' => 'boolean',
            'tags' => 'nullable|string',
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        $uploadedFiles = [];
        $errors = [];

        foreach ($request->file('files') as $uploadedFile) {
            try {
                // تحقق من نوع الملف
                if (!$this->isAllowedFileType($uploadedFile)) {
                    $errors[] = "نوع الملف {$uploadedFile->getClientOriginalName()} غير مسموح";
                    continue;
                }

                // توليد اسم فريد للملف
                $filename = time() . '_' . Str::random(10) . '.' . $uploadedFile->getClientOriginalExtension();
                
                // حفظ الملف
                $path = $uploadedFile->storeAs('files', $filename, 'public');
                
                // حساب hash
                $hash = File::generateHash($uploadedFile->getRealPath());
                
                // تحقق من عدم وجود ملف مطابق
                if (File::where('hash', $hash)->exists()) {
                    Storage::delete($path);
                    $errors[] = "الملف {$uploadedFile->getClientOriginalName()} موجود مسبقاً";
                    continue;
                }

                // إنشاء سجل الملف
                $file = File::create([
                    'name' => $uploadedFile->getClientOriginalName(),
                    'filename' => $filename,
                    'path' => $path,
                    'extension' => $uploadedFile->getClientOriginalExtension(),
                    'mime_type' => $uploadedFile->getMimeType(),
                    'size' => $uploadedFile->getSize(),
                    'description' => $request->description,
                    'category' => $request->category ?: 'general',
                    'is_public' => $request->boolean('is_public'),
                    'uploaded_by' => auth()->id(),
                    'folder_id' => $request->folder_id,
                    'tags' => $request->tags ? explode(',', $request->tags) : null,
                    'hash' => $hash,
                ]);

                $uploadedFiles[] = $file;

                // تحديث إحصائيات المجلد
                if ($file->folder) {
                    $file->folder->updateStats();
                }

            } catch (\Exception $e) {
                $errors[] = "خطأ في رفع الملف {$uploadedFile->getClientOriginalName()}: " . $e->getMessage();
            }
        }

        $message = count($uploadedFiles) . ' ملف تم رفعها بنجاح';
        if (!empty($errors)) {
            $message .= '. أخطاء: ' . implode(', ', $errors);
        }

        return redirect()->back()->with('success', $message);
    }

    /**
     * تحميل ملف
     */
    public function download(File $file)
    {
        if (!Storage::exists($file->path)) {
            abort(404, 'الملف غير موجود');
        }

        // تحديث عداد التحميلات
        $file->incrementDownloadCount();

        return Storage::download($file->path, $file->name);
    }

    /**
     * معاينة ملف
     */
    public function preview(File $file)
    {
        if (!Storage::exists($file->path)) {
            abort(404, 'الملف غير موجود');
        }

        $filePath = Storage::path($file->path);
        
        return response()->file($filePath, [
            'Content-Type' => $file->mime_type,
            'Content-Disposition' => 'inline; filename="' . $file->name . '"'
        ]);
    }

    /**
     * تحديث معلومات الملف
     */
    public function update(Request $request, File $file): RedirectResponse
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'description' => 'nullable|string|max:500',
            'category' => 'nullable|string|max:50',
            'is_public' => 'boolean',
            'tags' => 'nullable|string',
            'folder_id' => 'nullable|exists:folders,id',
        ]);

        $validated['tags'] = $request->tags ? explode(',', $request->tags) : null;

        $file->update($validated);

        // تحديث إحصائيات المجلدات المتأثرة
        if ($file->folder) {
            $file->folder->updateStats();
        }

        return redirect()->back()->with('success', 'تم تحديث معلومات الملف بنجاح');
    }

    /**
     * حذف ملف
     */
    public function destroy(File $file): RedirectResponse
    {
        $folder = $file->folder;
        
        // حذف الملف من التخزين
        if (Storage::exists($file->path)) {
            Storage::delete($file->path);
        }

        $file->delete();

        // تحديث إحصائيات المجلد
        if ($folder) {
            $folder->updateStats();
        }

        return redirect()->back()->with('success', 'تم حذف الملف بنجاح');
    }

    /**
     * إنشاء مجلد جديد
     */
    public function createFolder(Request $request): RedirectResponse
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'description' => 'nullable|string|max:500',
            'parent_id' => 'nullable|exists:folders,id',
            'color' => 'nullable|string|max:7',
            'is_public' => 'boolean',
        ]);

        $parent = ($validated['parent_id'] ?? null) ? Folder::find($validated['parent_id']) : null;
        
        $folder = Folder::create([
            'name' => $validated['name'],
            'slug' => Folder::generateUniqueSlug($validated['name'], $validated['parent_id'] ?? null),
            'description' => $validated['description'],
            'parent_id' => $validated['parent_id'] ?? null,
            'path' => $parent ? $parent->path . '/' . Str::slug($validated['name']) : Str::slug($validated['name']),
            'level' => $parent ? $parent->level + 1 : 0,
            'color' => $validated['color'] ?: '#3498db',
            'is_public' => $request->boolean('is_public'),
            'created_by' => auth()->id(),
        ]);

        // تحديث إحصائيات المجلد الأب
        if ($parent) {
            $parent->updateStats();
        }

        return redirect()->back()->with('success', 'تم إنشاء المجلد بنجاح');
    }

    /**
     * تحديث مجلد
     */
    public function updateFolder(Request $request, Folder $folder): RedirectResponse
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'description' => 'nullable|string|max:500',
            'color' => 'nullable|string|max:7',
            'is_public' => 'boolean',
        ]);

        // تحديث slug إذا تغير الاسم
        if ($folder->name !== $validated['name']) {
            $validated['slug'] = Folder::generateUniqueSlug($validated['name'], $folder->parent_id, $folder->id);
            $validated['path'] = $folder->parent ? $folder->parent->path . '/' . $validated['slug'] : $validated['slug'];
        }

        $folder->update($validated);

        return redirect()->back()->with('success', 'تم تحديث المجلد بنجاح');
    }

    /**
     * حذف مجلد
     */
    public function deleteFolder(Folder $folder): RedirectResponse
    {
        if ($folder->is_system) {
            return redirect()->back()->with('error', 'لا يمكن حذف مجلدات النظام');
        }

        $parent = $folder->parent;
        
        $folder->deleteWithContents();

        // تحديث إحصائيات المجلد الأب
        if ($parent) {
            $parent->updateStats();
        }

        return redirect()->back()->with('success', 'تم حذف المجلد ومحتوياته بنجاح');
    }

    /**
     * تحميل مجلد كملف مضغوط
     */
    public function downloadFolder(Folder $folder)
    {
        $zipFileName = Str::slug($folder->name) . '-' . date('Y-m-d') . '.zip';
        $zipPath = storage_path('app/temp/' . $zipFileName);

        // إنشاء مجلد temp إذا لم يكن موجوداً
        if (!file_exists(dirname($zipPath))) {
            mkdir(dirname($zipPath), 0755, true);
        }

        $zip = new ZipArchive;
        if ($zip->open($zipPath, ZipArchive::CREATE) === TRUE) {
            $this->addFolderToZip($zip, $folder);
            $zip->close();

            return response()->download($zipPath)->deleteFileAfterSend(true);
        }

        return redirect()->back()->with('error', 'فشل في إنشاء الملف المضغوط');
    }

    /**
     * إضافة مجلد إلى الملف المضغوط
     */
    private function addFolderToZip(ZipArchive $zip, Folder $folder, string $parentPath = ''): void
    {
        $folderPath = $parentPath . $folder->name . '/';
        $zip->addEmptyDir(rtrim($folderPath, '/'));

        // إضافة الملفات
        foreach ($folder->files as $file) {
            if (Storage::exists($file->path)) {
                $zip->addFile(Storage::path($file->path), $folderPath . $file->name);
            }
        }

        // إضافة المجلدات الفرعية
        foreach ($folder->children as $child) {
            $this->addFolderToZip($zip, $child, $folderPath);
        }
    }

    /**
     * نقل ملفات/مجلدات
     */
    public function move(Request $request): RedirectResponse
    {
        $validated = $request->validate([
            'items' => 'required|array',
            'items.*.type' => 'required|in:file,folder',
            'items.*.id' => 'required|integer',
            'destination_folder_id' => 'nullable|exists:folders,id',
        ]);

        $destination = $validated['destination_folder_id'] ? Folder::find($validated['destination_folder_id']) : null;
        $moved = 0;

        foreach ($validated['items'] as $item) {
            try {
                if ($item['type'] === 'file') {
                    $file = File::find($item['id']);
                    if ($file) {
                        $oldFolder = $file->folder;
                        $file->update(['folder_id' => $destination?->id]);
                        
                        // تحديث إحصائيات المجلدات المتأثرة
                        if ($oldFolder) $oldFolder->updateStats();
                        if ($destination) $destination->updateStats();
                        
                        $moved++;
                    }
                } else {
                    $folder = Folder::find($item['id']);
                    if ($folder && !$folder->is_system) {
                        $folder->moveTo($destination);
                        $moved++;
                    }
                }
            } catch (\Exception $e) {
                // تجاهل الأخطاء والمتابعة
            }
        }

        return redirect()->back()->with('success', "تم نقل {$moved} عنصر بنجاح");
    }

    /**
     * الحصول على بيانات الملف للتعديل
     */
    public function getFileData(File $file)
    {
        return response()->json([
            'id' => $file->id,
            'name' => $file->name,
            'description' => $file->description,
            'category' => $file->category,
            'tags' => $file->tags,
            'folder_id' => $file->folder_id,
            'is_public' => $file->is_public,
            'formatted_size' => $file->formatted_size,
            'extension' => $file->extension,
            'created_at' => $file->created_at,
            'download_count' => $file->download_count,
        ]);
    }

    /**
     * الحصول على بيانات المجلد للتعديل
     */
    public function getFolderData(Folder $folder)
    {
        return response()->json([
            'id' => $folder->id,
            'name' => $folder->name,
            'description' => $folder->description,
            'color' => $folder->color,
            'is_public' => $folder->is_public,
            'files_count' => $folder->files_count,
            'subfolders_count' => $folder->subfolders_count,
            'formatted_total_size' => $folder->formatted_total_size,
            'created_at' => $folder->created_at,
        ]);
    }

    /**
     * الحصول على قائمة المجلدات
     */
    public function getFoldersList()
    {
        $folders = Folder::orderBy('path')->get()->map(function ($folder) {
            return [
                'id' => $folder->id,
                'name' => $folder->name,
                'full_path' => $folder->full_path,
                'level' => $folder->level,
            ];
        });

        return response()->json($folders);
    }

    /**
     * البحث في الملفات
     */
    public function search(Request $request): View
    {
        $search = $request->get('q');
        $category = $request->get('category');
        $type = $request->get('type');

        $files = File::with(['uploader', 'folder'])
            ->when($search, function ($q) use ($search) {
                $q->search($search);
            })
            ->when($category, function ($q) use ($category) {
                $q->inCategory($category);
            })
            ->when($type, function ($q) use ($type) {
                $q->ofType($type);
            })
            ->active()
            ->notExpired()
            ->latest()
            ->paginate(20)
            ->appends($request->query());

        return view('file-manager.search', compact('files'));
    }

    /**
     * التحقق من نوع الملف المسموح
     */
    private function isAllowedFileType($file): bool
    {
        $allowedMimes = [
            // صور
            'image/jpeg', 'image/jpg', 'image/png', 'image/gif', 'image/webp',
            // مستندات
            'application/pdf', 'application/msword', 'application/vnd.openxmlformats-officedocument.wordprocessingml.document',
            'application/vnd.ms-excel', 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet',
            'application/vnd.ms-powerpoint', 'application/vnd.openxmlformats-officedocument.presentationml.presentation',
            'text/plain', 'text/csv',
            // فيديو
            'video/mp4', 'video/avi', 'video/mov', 'video/wmv',
            // صوت
            'audio/mp3', 'audio/wav', 'audio/ogg',
            // أرشيف
            'application/zip', 'application/x-rar-compressed', 'application/x-7z-compressed',
        ];

        return in_array($file->getMimeType(), $allowedMimes);
    }
}
