<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Invoice;
use App\Models\Customer;
use App\Models\Product;
use App\Models\ZatcaSettings;
use Carbon\Carbon;

class DashboardController extends Controller
{
    public function index()
    {
        // إحصائيات عامة مع تطوير أكثر تفصيلاً
        $invoicesCount = Invoice::count();
        $customersCount = Customer::count();
        $productsCount = Product::count();
        $totalSales = Invoice::whereIn('status', ['cleared', 'reported'])->sum('total_amount');
        
        // إحصائيات الفواتير حسب النوع
        $standardInvoicesCount = Invoice::where('type', 'standard')->count();
        $simplifiedInvoicesCount = Invoice::where('type', 'simplified')->count();
        $clearedInvoicesCount = Invoice::where('status', 'cleared')->count();
        $pendingInvoicesCount = Invoice::where('status', 'draft')->count();
        
        // إحصائيات متقدمة للوحة التحكم الجديدة
        $currentMonth = Carbon::now()->startOfMonth();
        $lastMonth = Carbon::now()->subMonth()->startOfMonth();
        
        // مبيعات الشهر الحالي والسابق للمقارنة
        $currentMonthSales = Invoice::whereIn('status', ['cleared', 'reported'])
            ->whereBetween('created_at', [$currentMonth, Carbon::now()])
            ->sum('total_amount');
        
        $lastMonthSales = Invoice::whereIn('status', ['cleared', 'reported'])
            ->whereBetween('created_at', [$lastMonth, $currentMonth])
            ->sum('total_amount');
        
        // حساب نسبة النمو
        $salesGrowthPercentage = $lastMonthSales > 0
            ? (($currentMonthSales - $lastMonthSales) / $lastMonthSales) * 100
            : 0;
        
        // عملاء جدد هذا الشهر
        $newCustomersThisMonth = Customer::whereBetween('created_at', [$currentMonth, Carbon::now()])->count();
        $newCustomersLastMonth = Customer::whereBetween('created_at', [$lastMonth, $currentMonth])->count();
        $customersGrowthPercentage = $newCustomersLastMonth > 0
            ? (($newCustomersThisMonth - $newCustomersLastMonth) / $newCustomersLastMonth) * 100
            : 0;
        
        // طلبات جديدة هذا الشهر
        $ordersThisMonth = Invoice::whereBetween('created_at', [$currentMonth, Carbon::now()])->count();
        $ordersLastMonth = Invoice::whereBetween('created_at', [$lastMonth, $currentMonth])->count();
        $ordersGrowthPercentage = $ordersLastMonth > 0
            ? (($ordersThisMonth - $ordersLastMonth) / $ordersLastMonth) * 100
            : 0;
        
        // متوسط قيمة الطلب
        $averageOrderValue = $invoicesCount > 0 ? $totalSales / $invoicesCount : 0;
        
        // بيانات الرسوم البيانية - مبيعات آخر 6 أشهر
        $monthlyRevenue = [];
        $monthlyLabels = [];
        
        for ($i = 5; $i >= 0; $i--) {
            $monthStart = Carbon::now()->subMonths($i)->startOfMonth();
            $monthEnd = Carbon::now()->subMonths($i)->endOfMonth();
            $monthRevenue = Invoice::whereIn('status', ['cleared', 'reported'])
                ->whereBetween('created_at', [$monthStart, $monthEnd])
                ->sum('total_amount');
            
            $monthlyRevenue[] = $monthRevenue;
            $monthlyLabels[] = $monthStart->format('M Y');
        }
        
        // توزيع المبيعات حسب نوع المنتج (مثال)
        $salesDistribution = [
            ['label' => 'المنتجات الإلكترونية', 'value' => 45],
            ['label' => 'الملابس', 'value' => 30],
            ['label' => 'الأجهزة المنزلية', 'value' => 25]
        ];
        
        // آخر الفواتير مع تفاصيل أكثر
        $recentInvoices = Invoice::with(['customer'])
            ->orderBy('created_at', 'desc')
            ->limit(10)
            ->get()
            ->map(function ($invoice) {
                return [
                    'id' => $invoice->id,
                    'invoice_number' => $invoice->invoice_number,
                    'customer_name' => $invoice->customer ? $invoice->customer->name : 'عميل محذوف',
                    'total_amount' => $invoice->total_amount,
                    'status' => $invoice->status,
                    'created_at' => $invoice->created_at->format('Y-m-d'),
                    'status_label' => $this->getStatusLabel($invoice->status)
                ];
            });
        
        // حالة ZATCA
        $zatcaSettings = ZatcaSettings::first();
        $zatcaStatus = $zatcaSettings ? $zatcaSettings->is_active : false;
        $zatcaEnvironment = $zatcaSettings ? $zatcaSettings->environment : null;
        $lastConnectionTest = $zatcaSettings && $zatcaSettings->last_connection_test
            ? $zatcaSettings->last_connection_test->diffForHumans()
            : null;
        
        // تنبيهات النظام
        $systemAlerts = $this->getSystemAlerts($zatcaSettings);
        
        // مؤشرات الأداء الرئيسية
        $kpis = [
            'conversion_rate' => 24.8, // معدل التحويل
            'avg_order_value' => $averageOrderValue,
            'return_rate' => 2.1, // معدل الإرجاع
            'customer_satisfaction' => 4.8 // رضا العملاء
        ];
        
        return view('dashboard', compact(
            'invoicesCount',
            'customersCount',
            'productsCount',
            'totalSales',
            'currentMonthSales',
            'salesGrowthPercentage',
            'newCustomersThisMonth',
            'customersGrowthPercentage',
            'ordersThisMonth',
            'ordersGrowthPercentage',
            'averageOrderValue',
            'monthlyRevenue',
            'monthlyLabels',
            'salesDistribution',
            'standardInvoicesCount',
            'simplifiedInvoicesCount',
            'clearedInvoicesCount',
            'pendingInvoicesCount',
            'recentInvoices',
            'zatcaStatus',
            'zatcaEnvironment',
            'lastConnectionTest',
            'systemAlerts',
            'kpis'
        ));
    }
    
    private function getStatusLabel($status)
    {
        $labels = [
            'draft' => 'مسودة',
            'pending' => 'قيد المعالجة',
            'cleared' => 'مكتمل',
            'reported' => 'مبلغ عنه',
            'cancelled' => 'ملغي'
        ];
        
        return $labels[$status] ?? $status;
    }
    
    private function getSystemAlerts($zatcaSettings)
    {
        $alerts = [];
        
        // تحقق من إعدادات ZATCA
        if (!$zatcaSettings || !$zatcaSettings->is_active) {
            $alerts[] = [
                'type' => 'warning',
                'icon' => 'exclamation-triangle',
                'message' => 'لم يتم إعداد الاتصال مع ZATCA بعد. يرجى الذهاب إلى إعدادات ZATCA لإكمال الإعداد.'
            ];
        }
        
        // تحقق من انتهاء صلاحية الشهادة
        if ($zatcaSettings && $zatcaSettings->certificate_expires_at) {
            $daysToExpiry = Carbon::now()->diffInDays($zatcaSettings->certificate_expires_at, false);
            if ($daysToExpiry <= 30 && $daysToExpiry > 0) {
                $alerts[] = [
                    'type' => 'warning',
                    'icon' => 'clock',
                    'message' => "ستنتهي صلاحية شهادة ZATCA خلال {$daysToExpiry} يوم. يرجى تجديدها قريباً."
                ];
            } elseif ($daysToExpiry <= 0) {
                $alerts[] = [
                    'type' => 'danger',
                    'icon' => 'times-circle',
                    'message' => 'انتهت صلاحية شهادة ZATCA. يرجى تجديدها فوراً.'
                ];
            }
        }
        
        // تحقق من المنتجات منخفضة المخزون
        $lowStockProducts = Product::lowStock()->count();
        if ($lowStockProducts > 0) {
            $alerts[] = [
                'type' => 'info',
                'icon' => 'info-circle',
                'message' => "يوجد {$lowStockProducts} منتج منخفض المخزون."
            ];
        }
        
        return $alerts;
    }
}
