<?php

namespace App\Http\Controllers;

use App\Models\Customer;
use App\Models\Invoice;
use Illuminate\Http\Request;
use Carbon\Carbon;
use Illuminate\Support\Facades\DB;

class CustomerReportController extends Controller
{
    /**
     * Display customer reports dashboard
     */
    public function index()
    {
        return view('system-reports.customer.index');
    }

    /**
     * Customer summary report
     */
    public function summary(Request $request)
    {
        $totalCustomers = Customer::count();
        $activeCustomers = Customer::where('is_active', true)->count();
        $inactiveCustomers = Customer::where('is_active', false)->count();
        
        // حساب العملاء الجدد هذا الشهر
        $newCustomersThisMonth = Customer::whereMonth('created_at', now()->month)
            ->whereYear('created_at', now()->year)
            ->count();
        
        // حساب متوسط قيمة العميل - طريقة مبسطة
        $totalCustomers = Customer::count();
        $totalInvoiceAmount = Invoice::sum('total_amount') ?? 0;
        $averageCustomerValue = $totalCustomers > 0 ? ($totalInvoiceAmount / $totalCustomers) : 0;
        
        $customerStats = Customer::selectRaw('
                COUNT(*) as total_customers,
                COUNT(CASE WHEN is_active = 1 THEN 1 END) as active_customers,
                COUNT(CASE WHEN is_active = 0 THEN 1 END) as inactive_customers
            ')
            ->first();

        // إضافة البيانات المحسوبة للكائن
        $customerStats->new_customers_this_month = $newCustomersThisMonth;
        $customerStats->average_customer_value = $averageCustomerValue;

        // بيانات نمو العملاء لآخر 12 شهر - تم إصلاح المنطق
        $customerGrowthData = [];
        
        for ($i = 11; $i >= 0; $i--) {
            $date = now()->copy()->subMonths($i);
            $monthStart = $date->copy()->startOfMonth();
            $monthEnd = $date->copy()->endOfMonth();
            
            // عدد العملاء الجدد في هذا الشهر
            $newCustomers = Customer::whereBetween('created_at', [$monthStart, $monthEnd])->count();
            
            // إجمالي العملاء حتى نهاية هذا الشهر
            $cumulativeTotal = Customer::where('created_at', '<=', $monthEnd)->count();
            
            $customerGrowthData[] = (object) [
                'month' => $date->format('Y-m'),
                'month_name_ar' => $date->format('F Y'),
                'new_customers' => $newCustomers,
                'cumulative_total' => $cumulativeTotal
            ];
        }

        // بيانات توزيع العملاء حسب النشاط - طريقة مبسطة
        $threeMonthsAgo = now()->subMonths(3);
        $oneMonthAgo = now()->subMonth();
        
        $activeCustomers = Customer::where('is_active', true)
            ->whereHas('invoices', function($query) use ($threeMonthsAgo) {
                $query->where('created_at', '>=', $threeMonthsAgo);
            })->count();
            
        $inactiveCustomers = Customer::where('is_active', false)->count();
        
        $newCustomers = Customer::where('created_at', '>=', $oneMonthAgo)->count();
        
        $customerActivityData = [
            'active' => $activeCustomers,
            'inactive' => $inactiveCustomers,
            'new' => $newCustomers
        ];

        // أفضل العملاء
        $topCustomers = Customer::leftJoin('invoices', 'customers.id', '=', 'invoices.customer_id')
            ->selectRaw('
                customers.id,
                customers.name,
                customers.email,
                customers.phone,
                COUNT(invoices.id) as total_invoices,
                COALESCE(SUM(invoices.total_amount), 0) as total_sales
            ')
            ->groupBy('customers.id', 'customers.name', 'customers.email', 'customers.phone')
            ->orderByDesc('total_sales')
            ->limit(10)
            ->get();

        // مؤشرات الأداء الرئيسية
        $kpis = [
            'retention_rate' => $this->calculateRetentionRate(),
            'return_rate' => $this->calculateReturnRate(),
            'average_customer_age' => $this->calculateAverageCustomerAge(),
            'average_orders_per_customer' => $this->calculateAverageOrdersPerCustomer()
        ];

        return view('system-reports.customer.summary', compact(
            'customerStats',
            'topCustomers',
            'customerGrowthData',
            'customerActivityData',
            'kpis'
        ));
    }

    /**
     * حساب معدل الاحتفاظ بالعملاء
     */
    private function calculateRetentionRate()
    {
        $totalCustomers = Customer::count();
        if ($totalCustomers == 0) return 0;
        
        // العملاء الذين لديهم أكثر من فاتورة واحدة
        $returningCustomers = Customer::whereHas('invoices', function($query) {
            // فاتورتين أو أكثر
        }, '>=', 2)->count();
        
        return round(($returningCustomers / $totalCustomers) * 100, 1);
    }

    /**
     * حساب معدل الإعادة
     */
    private function calculateReturnRate()
    {
        $totalCustomers = Customer::count();
        if ($totalCustomers == 0) return 0;
        
        // العملاء الذين تسوقوا في آخر 6 أشهر وأيضاً قبل ذلك
        $sixMonthsAgo = now()->subMonths(6);
        
        $returningCustomers = Customer::where('created_at', '<', $sixMonthsAgo)
            ->whereHas('invoices', function($query) use ($sixMonthsAgo) {
                $query->where('created_at', '>=', $sixMonthsAgo);
            })->count();
        
        return round(($returningCustomers / $totalCustomers) * 100, 1);
    }

    /**
     * حساب متوسط عمر العميل
     */
    private function calculateAverageCustomerAge()
    {
        $customers = Customer::all();
        if ($customers->count() == 0) return 0;
        
        $totalDays = 0;
        foreach ($customers as $customer) {
            $totalDays += now()->diffInDays($customer->created_at);
        }
        
        return round($totalDays / $customers->count());
    }

    /**
     * حساب متوسط الطلبات لكل عميل
     */
    private function calculateAverageOrdersPerCustomer()
    {
        $totalCustomers = Customer::count();
        if ($totalCustomers == 0) return 0;
        
        $totalInvoices = Invoice::count();
        
        return round($totalInvoices / $totalCustomers, 1);
    }

    /**
     * Customer activity report
     */
    public function activity(Request $request)
    {
        $startDate = $request->get('start_date', now()->startOfMonth());
        $endDate = $request->get('end_date', now()->endOfMonth());

        $customerActivity = Customer::leftJoin('invoices', 'customers.id', '=', 'invoices.customer_id')
            ->whereBetween('invoices.issue_date', [$startDate, $endDate])
            ->selectRaw('
                customers.id,
                customers.name,
                customers.email,
                customers.phone,
                customers.created_at as registration_date,
                COUNT(invoices.id) as total_invoices,
                COALESCE(SUM(invoices.total_amount), 0) as total_sales,
                MAX(invoices.issue_date) as last_purchase_date,
                DATEDIFF(NOW(), MAX(invoices.issue_date)) as days_since_last_purchase
            ')
            ->groupBy('customers.id', 'customers.name', 'customers.email', 'customers.phone', 'customers.created_at')
            ->orderByDesc('total_sales')
            ->paginate(15);

        return view('system-reports.customer.activity', compact('customerActivity', 'startDate', 'endDate'));
    }

    /**
     * Customer segmentation report
     */
    public function segmentation(Request $request)
    {
        // Customer segments based on purchase behavior
        $segments = DB::select("
            SELECT 
                CASE 
                    WHEN total_sales >= 10000 THEN 'VIP'
                    WHEN total_sales >= 5000 THEN 'Premium'
                    WHEN total_sales >= 1000 THEN 'Regular'
                    WHEN total_sales > 0 THEN 'New'
                    ELSE 'Inactive'
                END as segment,
                COUNT(*) as customer_count,
                SUM(total_sales) as segment_sales,
                AVG(total_sales) as avg_sales_per_customer
            FROM (
                SELECT 
                    customers.id,
                    COALESCE(SUM(invoices.total_amount), 0) as total_sales
                FROM customers
                LEFT JOIN invoices ON customers.id = invoices.customer_id
                GROUP BY customers.id
            ) as customer_totals
            GROUP BY segment
            ORDER BY segment_sales DESC
        ");

        // Geographic distribution
        $geographicData = Customer::selectRaw('
                city,
                COUNT(*) as customer_count,
                COALESCE(SUM(total_sales.total), 0) as total_sales
            ')
            ->leftJoinSub(
                Invoice::selectRaw('customer_id, SUM(total_amount) as total')->groupBy('customer_id'),
                'total_sales',
                'customers.id',
                '=',
                'total_sales.customer_id'
            )
            ->groupBy('city')
            ->orderByDesc('customer_count')
            ->get();

        return view('system-reports.customer.segmentation', compact('segments', 'geographicData'));
    }

    /**
     * Customer lifetime value report
     */
    public function lifetimeValue(Request $request)
    {
        $customerLTV = Customer::leftJoin('invoices', 'customers.id', '=', 'invoices.customer_id')
            ->selectRaw('
                customers.id,
                customers.name,
                customers.email,
                customers.created_at as registration_date,
                COUNT(invoices.id) as total_orders,
                COALESCE(SUM(invoices.total_amount), 0) as total_spent,
                COALESCE(AVG(invoices.total_amount), 0) as avg_order_value,
                DATEDIFF(NOW(), customers.created_at) as customer_age_days,
                CASE 
                    WHEN DATEDIFF(NOW(), customers.created_at) > 0 
                    THEN COALESCE(SUM(invoices.total_amount), 0) / (DATEDIFF(NOW(), customers.created_at) / 30)
                    ELSE 0 
                END as monthly_value
            ')
            ->groupBy('customers.id', 'customers.name', 'customers.email', 'customers.created_at')
            ->orderByDesc('total_spent')
            ->paginate(15);

        $ltvStats = DB::select("
            SELECT 
                COUNT(*) as total_customers,
                AVG(total_spent) as avg_ltv,
                MAX(total_spent) as max_ltv,
                MIN(total_spent) as min_ltv,
                AVG(total_orders) as avg_orders_per_customer,
                AVG(avg_order_value) as overall_avg_order_value
            FROM (
                SELECT 
                    customers.id,
                    COUNT(invoices.id) as total_orders,
                    COALESCE(SUM(invoices.total_amount), 0) as total_spent,
                    COALESCE(AVG(invoices.total_amount), 0) as avg_order_value
                FROM customers
                LEFT JOIN invoices ON customers.id = invoices.customer_id
                GROUP BY customers.id
            ) as customer_stats
        ")[0];

        return view('system-reports.customer.lifetime-value', compact('customerLTV', 'ltvStats'));
    }

    /**
     * Customer retention report
     */
    public function retention(Request $request)
    {
        $months = $request->get('months', 12);
        $startDate = now()->subMonths($months)->startOfMonth();

        // Customer retention by cohort
        $retentionData = DB::select("
            SELECT 
                DATE_FORMAT(first_purchase, '%Y-%m') as cohort,
                COUNT(DISTINCT customer_id) as customers,
                COUNT(DISTINCT CASE WHEN months_diff >= 1 THEN customer_id END) as month_1,
                COUNT(DISTINCT CASE WHEN months_diff >= 3 THEN customer_id END) as month_3,
                COUNT(DISTINCT CASE WHEN months_diff >= 6 THEN customer_id END) as month_6,
                COUNT(DISTINCT CASE WHEN months_diff >= 12 THEN customer_id END) as month_12
            FROM (
                SELECT 
                    customer_id,
                    MIN(issue_date) as first_purchase,
                    TIMESTAMPDIFF(MONTH, MIN(issue_date), MAX(issue_date)) as months_diff
                FROM invoices
                WHERE issue_date >= ?
                GROUP BY customer_id
            ) as customer_cohorts
            GROUP BY cohort
            ORDER BY cohort DESC
        ", [$startDate]);

        return view('system-reports.customer.retention', compact('retentionData', 'months'));
    }

    /**
     * Export customer report
     */
    public function export(Request $request)
    {
        $type = $request->get('type', 'summary');
        $format = $request->get('format', 'excel');
        
        // Implementation for export functionality
        return redirect()->back()->with('success', 'تم تصدير التقرير بنجاح');
    }
}