<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Customer;

class CustomerController extends Controller
{
    /**
     * عرض قائمة العملاء
     */
    public function index(Request $request)
    {
        $query = Customer::query();
        
        // فلترة حسب النوع
        if ($request->filled('type')) {
            $query->where('customer_type', $request->type);
        }
        
        // فلترة حسب الحالة
        if ($request->filled('status')) {
            if ($request->status == 'active') {
                $query->where('is_active', true);
            } else {
                $query->where('is_active', false);
            }
        }
        
        // البحث
        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function($q) use ($search) {
                $q->where('name', 'like', "%{$search}%")
                  ->orWhere('email', 'like', "%{$search}%")
                  ->orWhere('phone', 'like', "%{$search}%")
                  ->orWhere('vat_number', 'like', "%{$search}%")
                  ->orWhere('national_id', 'like', "%{$search}%");
            });
        }
        
        $customers = $query->orderBy('created_at', 'desc')->paginate(15);
        
        return view('customers.index', compact('customers'));
    }

    /**
     * عرض نموذج إنشاء عميل جديد
     */
    public function create()
    {
        return view('customers.create');
    }

    /**
     * حفظ عميل جديد
     */
    public function store(Request $request)
    {
        $rules = [
            'name' => 'required|string|max:255',
            'customer_type' => 'required|in:individual,business',
            'email' => 'nullable|email|unique:customers,email',
            'phone' => 'nullable|string|max:20',
            'is_active' => 'boolean',
        ];

        // قواعد إضافية حسب نوع العميل
        if ($request->customer_type === 'individual') {
            // للأفراد: جميع الحقول اختيارية
            $rules['national_id'] = 'nullable|string|size:10|unique:customers,national_id';
            $rules['city'] = 'nullable|string|max:255';
            $rules['district'] = 'nullable|string|max:255';
            $rules['street'] = 'nullable|string|max:255';
            $rules['building_number'] = 'nullable|string|max:10';
            $rules['postal_code'] = 'nullable|string|size:5';
            $rules['additional_number'] = 'nullable|string|max:10';
            $rules['unit_number'] = 'nullable|string|max:10';
        } else {
            // للشركات: العنوان الوطني مطلوب
            $rules['vat_number'] = 'required|string|size:15|unique:customers,vat_number';
            $rules['cr_number'] = 'nullable|string|max:20';
            $rules['city'] = 'required|string|max:255';
            $rules['district'] = 'required|string|max:255';
            $rules['street'] = 'required|string|max:255';
            $rules['building_number'] = 'required|string|max:10';
            $rules['postal_code'] = 'required|string|size:5';
            $rules['additional_number'] = 'required|string|max:10';
            $rules['unit_number'] = 'nullable|string|max:10';
        }

        $request->validate($rules);

        $customer = Customer::create($request->all());

        return redirect()->route('customers.index')
            ->with('success', 'تم إنشاء العميل بنجاح');
    }

    /**
     * عرض تفاصيل العميل
     */
    public function show(Customer $customer)
    {
        $customer->load('invoices');
        return view('customers.show', compact('customer'));
    }

    /**
     * عرض نموذج تعديل العميل
     */
    public function edit(Customer $customer)
    {
        return view('customers.edit', compact('customer'));
    }

    /**
     * تحديث العميل
     */
    public function update(Request $request, Customer $customer)
    {
        $rules = [
            'name' => 'required|string|max:255',
            'customer_type' => 'required|in:individual,business',
            'email' => 'nullable|email|unique:customers,email,' . $customer->id,
            'phone' => 'nullable|string|max:20',
            'is_active' => 'boolean',
        ];

        // قواعد إضافية حسب نوع العميل
        if ($request->customer_type === 'individual') {
            // للأفراد: جميع الحقول اختيارية
            $rules['national_id'] = 'nullable|string|size:10|unique:customers,national_id,' . $customer->id;
            $rules['city'] = 'nullable|string|max:255';
            $rules['district'] = 'nullable|string|max:255';
            $rules['street'] = 'nullable|string|max:255';
            $rules['building_number'] = 'nullable|string|max:10';
            $rules['postal_code'] = 'nullable|string|size:5';
            $rules['additional_number'] = 'nullable|string|max:10';
            $rules['unit_number'] = 'nullable|string|max:10';
        } else {
            // للشركات: العنوان الوطني مطلوب
            $rules['vat_number'] = 'required|string|size:15|unique:customers,vat_number,' . $customer->id;
            $rules['cr_number'] = 'nullable|string|max:20';
            $rules['city'] = 'required|string|max:255';
            $rules['district'] = 'required|string|max:255';
            $rules['street'] = 'required|string|max:255';
            $rules['building_number'] = 'required|string|max:10';
            $rules['postal_code'] = 'required|string|size:5';
            $rules['additional_number'] = 'required|string|max:10';
            $rules['unit_number'] = 'nullable|string|max:10';
        }

        $request->validate($rules);

        $customer->update($request->all());

        return redirect()->route('customers.index')
            ->with('success', 'تم تحديث العميل بنجاح');
    }

    /**
     * حذف العميل
     */
    public function destroy(Customer $customer)
    {
        // التحقق من عدم استخدام العميل في فواتير
        if ($customer->invoices()->count() > 0) {
            return redirect()->route('customers.index')
                ->with('error', 'لا يمكن حذف العميل لأنه مرتبط بفواتير');
        }

        $customer->delete();

        return redirect()->route('customers.index')
            ->with('success', 'تم حذف العميل بنجاح');
    }
}
