<?php

namespace App\Http\Controllers;

use App\Models\CostCenter;
use App\Models\JournalEntryLine;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class CostCenterController extends Controller
{
    /**
     * Display a listing of cost centers.
     */
    public function index()
    {
        $costCenters = CostCenter::orderBy('code')->get();
        return view('cost-centers.index', compact('costCenters'));
    }

    /**
     * Show the form for creating a new cost center.
     */
    public function create()
    {
        // Get all active cost centers that can be used as parents
        $parentCostCenters = CostCenter::where('is_active', true)
            ->orderBy('code')
            ->get();
            
        return view('cost-centers.create', compact('parentCostCenters'));
    }

    /**
     * Store a newly created cost center in storage.
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'code' => 'required|string|max:20|unique:cost_centers',
            'name' => 'required|string|max:255',
            'type' => 'required|string|in:department,project,branch,activity,service',
            'parent_id' => 'nullable|exists:cost_centers,id',
            'manager_name' => 'nullable|string|max:255',
            'phone' => 'nullable|string|max:20',
            'description' => 'nullable|string',
            'budget_amount' => 'nullable|numeric|min:0',
            'budget_period' => 'nullable|string|in:monthly,quarterly,annual',
            'actual_amount' => 'nullable|numeric|min:0',
            'variance_threshold' => 'nullable|numeric|min:0|max:100',
            'is_active' => 'boolean',
            'allow_budget_exceeded' => 'boolean',
            'requires_approval' => 'boolean',
            'track_performance' => 'boolean'
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        $data = $request->all();
        
        // Handle boolean fields
        $data['is_active'] = $request->has('is_active');
        $data['allow_budget_exceeded'] = $request->has('allow_budget_exceeded');
        $data['requires_approval'] = $request->has('requires_approval');
        $data['track_performance'] = $request->has('track_performance');

        CostCenter::create($data);

        // Check if user wants to create another cost center
        if ($request->has('save_and_new')) {
            return redirect()->route('accounting.cost-centers.create')
                ->with('success', 'تم إنشاء مركز التكلفة بنجاح. يمكنك إضافة مركز تكلفة جديد.');
        }

        return redirect()->route('accounting.cost-centers.index')
            ->with('success', 'تم إنشاء مركز التكلفة بنجاح');
    }

    /**
     * Display the specified cost center.
     */
    public function show(CostCenter $costCenter)
    {
        // Get cost center transactions for current year
        $currentYear = date('Y');
        $transactions = JournalEntryLine::where('cost_center_id', $costCenter->id)
            ->whereHas('journalEntry', function($query) use ($currentYear) {
                $query->whereYear('entry_date', $currentYear)
                      ->where('status', 'posted');
            })
            ->with('journalEntry', 'account')
            ->orderBy('created_at', 'desc')
            ->paginate(20);

        // Calculate totals
        $totalDebits = JournalEntryLine::where('cost_center_id', $costCenter->id)
            ->whereHas('journalEntry', function($query) use ($currentYear) {
                $query->whereYear('entry_date', $currentYear)
                      ->where('status', 'posted');
            })
            ->sum('debit_amount');

        $totalCredits = JournalEntryLine::where('cost_center_id', $costCenter->id)
            ->whereHas('journalEntry', function($query) use ($currentYear) {
                $query->whereYear('entry_date', $currentYear)
                      ->where('status', 'posted');
            })
            ->sum('credit_amount');

        $actualAmount = $totalDebits - $totalCredits;
        $budgetVariance = $costCenter->budget_amount ? $costCenter->budget_amount - $actualAmount : null;
        $budgetUtilization = $costCenter->budget_amount && $costCenter->budget_amount > 0
            ? ($actualAmount / $costCenter->budget_amount) * 100
            : 0;

        // Get total transactions count
        $totalTransactions = JournalEntryLine::where('cost_center_id', $costCenter->id)
            ->whereHas('journalEntry', function($query) use ($currentYear) {
                $query->whereYear('entry_date', $currentYear)
                      ->where('status', 'posted');
            })
            ->count();
        
        // Get recent transactions (limit to 10 for the show page)
        $recentTransactions = JournalEntryLine::where('cost_center_id', $costCenter->id)
            ->whereHas('journalEntry', function($query) use ($currentYear) {
                $query->whereYear('entry_date', $currentYear)
                      ->where('status', 'posted');
            })
            ->with('journalEntry', 'account')
            ->orderBy('created_at', 'desc')
            ->limit(10)
            ->get();
        
        // Get monthly data for the current year
        $monthlyData = collect();
        
        for ($month = 1; $month <= 12; $month++) {
            $startDate = date('Y-m-01', strtotime("$currentYear-$month-01"));
            $endDate = date('Y-m-t', strtotime("$currentYear-$month-01"));
            
            $monthlyDebits = JournalEntryLine::where('cost_center_id', $costCenter->id)
                ->whereHas('journalEntry', function($query) use ($startDate, $endDate) {
                    $query->whereBetween('entry_date', [$startDate, $endDate])
                          ->where('status', 'posted');
                })
                ->sum('debit_amount');

            $monthlyCredits = JournalEntryLine::where('cost_center_id', $costCenter->id)
                ->whereHas('journalEntry', function($query) use ($startDate, $endDate) {
                    $query->whereBetween('entry_date', [$startDate, $endDate])
                          ->where('status', 'posted');
                })
                ->sum('credit_amount');

            $monthlyActual = $monthlyDebits - $monthlyCredits;
            $monthlyBudget = $costCenter->budget_amount ? $costCenter->budget_amount / 12 : 0;
            
            $monthlyData->push((object)[
                'month' => $month,
                'month_name' => date('F', strtotime("$currentYear-$month-01")),
                'total_amount' => $monthlyActual,
                'actual' => $monthlyActual,
                'budget' => $monthlyBudget,
                'variance' => $monthlyBudget - $monthlyActual,
                'debits' => $monthlyDebits,
                'credits' => $monthlyCredits,
            ]);
        }

        return view('cost-centers.show', compact(
            'costCenter',
            'transactions',
            'totalDebits',
            'totalCredits',
            'actualAmount',
            'budgetVariance',
            'budgetUtilization',
            'totalTransactions',
            'recentTransactions',
            'monthlyData'
        ));
    }

    /**
     * Show the form for editing the specified cost center.
     */
    public function edit(CostCenter $costCenter)
    {
        // Check if cost center has transactions
        $transactionCount = JournalEntryLine::where('cost_center_id', $costCenter->id)->count();
        $hasTransactions = $transactionCount > 0;
        
        return view('cost-centers.edit', compact('costCenter', 'transactionCount', 'hasTransactions'));
    }

    /**
     * Update the specified cost center in storage.
     */
    public function update(Request $request, CostCenter $costCenter)
    {
        $validator = Validator::make($request->all(), [
            'code' => 'required|string|max:20|unique:cost_centers,code,' . $costCenter->id,
            'name' => 'required|string|max:255',
            'description' => 'nullable|string',
            'budget_amount' => 'nullable|numeric|min:0',
            'manager' => 'nullable|string|max:255',
            'is_active' => 'boolean'
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        $costCenter->update($request->all());

        return redirect()->route('accounting.cost-centers.index')
            ->with('success', 'تم تحديث مركز التكلفة بنجاح');
    }

    /**
     * Remove the specified cost center from storage.
     */
    public function destroy(CostCenter $costCenter)
    {
        // Check if cost center has transactions
        $transactionCount = JournalEntryLine::where('cost_center_id', $costCenter->id)->count();
        
        if ($transactionCount > 0) {
            return redirect()->back()
                ->with('error', 'لا يمكن حذف مركز التكلفة لوجود حركات مرتبطة به');
        }

        $costCenter->delete();

        return redirect()->route('accounting.cost-centers.index')
            ->with('success', 'تم حذف مركز التكلفة بنجاح');
    }

    /**
     * Toggle cost center status
     */
    public function toggleStatus(CostCenter $costCenter)
    {
        $costCenter->update(['is_active' => !$costCenter->is_active]);
        
        $status = $costCenter->is_active ? 'تفعيل' : 'إلغاء تفعيل';
        return redirect()->back()
            ->with('success', "تم {$status} مركز التكلفة بنجاح");
    }

    /**
     * Cost center performance report
     */
    public function performance(Request $request)
    {
        $startDate = $request->input('start_date', now()->startOfYear());
        $endDate = $request->input('end_date', now()->endOfYear());

        $costCenters = CostCenter::where('is_active', true)->get();
        
        $performanceData = [];
        
        foreach ($costCenters as $costCenter) {
            $totalDebits = JournalEntryLine::where('cost_center_id', $costCenter->id)
                ->whereHas('journalEntry', function($query) use ($startDate, $endDate) {
                    $query->whereBetween('entry_date', [$startDate, $endDate])
                          ->where('status', 'posted');
                })
                ->sum('debit_amount');

            $totalCredits = JournalEntryLine::where('cost_center_id', $costCenter->id)
                ->whereHas('journalEntry', function($query) use ($startDate, $endDate) {
                    $query->whereBetween('entry_date', [$startDate, $endDate])
                          ->where('status', 'posted');
                })
                ->sum('credit_amount');

            $actualAmount = $totalDebits - $totalCredits;
            $budgetVariance = $costCenter->budget_amount ? $costCenter->budget_amount - $actualAmount : null;
            $budgetUtilization = $costCenter->budget_amount && $costCenter->budget_amount > 0 
                ? ($actualAmount / $costCenter->budget_amount) * 100 
                : 0;

            $performanceData[] = [
                'cost_center' => $costCenter,
                'actual_amount' => $actualAmount,
                'budget_amount' => $costCenter->budget_amount,
                'budget_variance' => $budgetVariance,
                'budget_utilization' => $budgetUtilization,
                'total_debits' => $totalDebits,
                'total_credits' => $totalCredits,
            ];
        }

        return view('cost-centers.performance', compact(
            'performanceData', 
            'startDate', 
            'endDate'
        ));
    }

    /**
     * Cost center budget analysis
     */
    public function budgetAnalysis(CostCenter $costCenter, Request $request)
    {
        $year = $request->input('year', date('Y'));
        
        // Calculate total actual amount for the year
        $totalDebits = JournalEntryLine::where('cost_center_id', $costCenter->id)
            ->whereHas('journalEntry', function($query) use ($year) {
                $query->whereYear('entry_date', $year)
                      ->where('status', 'posted');
            })
            ->sum('debit_amount');

        $totalCredits = JournalEntryLine::where('cost_center_id', $costCenter->id)
            ->whereHas('journalEntry', function($query) use ($year) {
                $query->whereYear('entry_date', $year)
                      ->where('status', 'posted');
            })
            ->sum('credit_amount');

        $actualAmount = $totalDebits - $totalCredits;
        
        // Get total transactions count
        $totalTransactions = JournalEntryLine::where('cost_center_id', $costCenter->id)
            ->whereHas('journalEntry', function($query) use ($year) {
                $query->whereYear('entry_date', $year)
                      ->where('status', 'posted');
            })
            ->count();
        
        // Get recent transactions
        $recentTransactions = JournalEntryLine::where('cost_center_id', $costCenter->id)
            ->whereHas('journalEntry', function($query) use ($year) {
                $query->whereYear('entry_date', $year)
                      ->where('status', 'posted');
            })
            ->with('journalEntry', 'account')
            ->orderBy('created_at', 'desc')
            ->limit(10)
            ->get();
        
        // Get monthly data
        $monthlyData = collect();
        
        for ($month = 1; $month <= 12; $month++) {
            $startDate = date('Y-m-01', strtotime("$year-$month-01"));
            $endDate = date('Y-m-t', strtotime("$year-$month-01"));
            
            $monthlyDebits = JournalEntryLine::where('cost_center_id', $costCenter->id)
                ->whereHas('journalEntry', function($query) use ($startDate, $endDate) {
                    $query->whereBetween('entry_date', [$startDate, $endDate])
                          ->where('status', 'posted');
                })
                ->sum('debit_amount');

            $monthlyCredits = JournalEntryLine::where('cost_center_id', $costCenter->id)
                ->whereHas('journalEntry', function($query) use ($startDate, $endDate) {
                    $query->whereBetween('entry_date', [$startDate, $endDate])
                          ->where('status', 'posted');
                })
                ->sum('credit_amount');

            $monthlyActual = $monthlyDebits - $monthlyCredits;
            $monthlyBudget = $costCenter->budget_amount ? $costCenter->budget_amount / 12 : 0;
            
            $monthlyData->push((object)[
                'month' => $month,
                'month_name' => date('F', strtotime("$year-$month-01")),
                'total_amount' => $monthlyActual,
                'actual' => $monthlyActual,
                'budget' => $monthlyBudget,
                'variance' => $monthlyBudget - $monthlyActual,
                'debits' => $monthlyDebits,
                'credits' => $monthlyCredits,
            ]);
        }

        return view('cost-centers.budget-analysis', compact(
            'costCenter',
            'monthlyData',
            'year',
            'actualAmount',
            'totalTransactions',
            'recentTransactions'
        ));
    }
}