<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Company;

class CompanyController extends Controller
{
    /**
     * عرض قائمة الشركات
     */
    public function index()
    {
        $companies = Company::orderBy('created_at', 'desc')->paginate(10);
        
        return view('companies.index', compact('companies'));
    }

    /**
     * عرض نموذج إنشاء شركة جديدة
     */
    public function create()
    {
        return view('companies.create');
    }

    /**
     * حفظ شركة جديدة
     */
    public function store(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'name_en' => 'nullable|string|max:255',
            'logo' => 'nullable|image|mimes:jpeg,png,jpg,gif,svg|max:2048',
            'vat_numbers' => 'required|array|min:1',
            'vat_numbers.*' => 'required|string|size:15|distinct',
            'cr_number' => 'required|string|max:20|unique:companies,cr_number',
            'city' => 'required|string|max:255',
            'district' => 'required|string|max:255',
            'street' => 'required|string|max:255',
            'building_number' => 'required|string|max:10',
            'postal_code' => 'required|string|size:5',
            'additional_number' => 'required|string|max:10',
            'unit_number' => 'nullable|string|max:10',
            'email' => 'nullable|email|unique:companies,email',
            'phone' => 'nullable|string|max:20',
            'website' => 'nullable|url',
            'is_active' => 'boolean',
        ]);

        // التحقق من عدم تكرار أي رقم ضريبي في قاعدة البيانات
        foreach ($request->vat_numbers as $vatNumber) {
            $existingCompany = Company::whereJsonContains('vat_numbers', $vatNumber)->first();
            if ($existingCompany) {
                return back()->withErrors([
                    'vat_numbers' => "الرقم الضريبي {$vatNumber} مستخدم مسبقاً في شركة: {$existingCompany->name}"
                ])->withInput();
            }
        }

        // إعداد البيانات للحفظ
        $data = $request->all();
        
        // التعامل مع رفع الشعار
        if ($request->hasFile('logo')) {
            $logo = $request->file('logo');
            $logoName = time() . '_' . uniqid() . '.' . $logo->getClientOriginalExtension();
            $logo->move(public_path('uploads/logos'), $logoName);
            $data['logo'] = 'uploads/logos/' . $logoName;
        }

        $company = Company::create($data);

        return redirect()->route('companies.index')
            ->with('success', 'تم إنشاء الشركة بنجاح');
    }

    /**
     * عرض تفاصيل الشركة
     */
    public function show(Company $company)
    {
        $company->load(['invoices', 'zatcaSettings']);
        return view('companies.show', compact('company'));
    }

    /**
     * عرض نموذج تعديل الشركة
     */
    public function edit(Company $company)
    {
        return view('companies.edit', compact('company'));
    }

    /**
     * تحديث الشركة
     */
    public function update(Request $request, Company $company)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'name_en' => 'nullable|string|max:255',
            'logo' => 'nullable|image|mimes:jpeg,png,jpg,gif,svg|max:2048',
            'vat_numbers' => 'required|array|min:1',
            'vat_numbers.*' => 'required|string|size:15|distinct',
            'cr_number' => 'required|string|max:20|unique:companies,cr_number,' . $company->id,
            'city' => 'required|string|max:255',
            'district' => 'required|string|max:255',
            'street' => 'required|string|max:255',
            'building_number' => 'required|string|max:10',
            'postal_code' => 'required|string|size:5',
            'additional_number' => 'required|string|max:10',
            'unit_number' => 'nullable|string|max:10',
            'email' => 'nullable|email|unique:companies,email,' . $company->id,
            'phone' => 'nullable|string|max:20',
            'website' => 'nullable|url',
            'is_active' => 'boolean',
        ]);

        // التحقق من عدم تكرار أي رقم ضريبي في قاعدة البيانات (باستثناء الشركة الحالية)
        foreach ($request->vat_numbers as $vatNumber) {
            $existingCompany = Company::where('id', '!=', $company->id)
                ->whereJsonContains('vat_numbers', $vatNumber)
                ->first();
            if ($existingCompany) {
                return back()->withErrors([
                    'vat_numbers' => "الرقم الضريبي {$vatNumber} مستخدم مسبقاً في شركة: {$existingCompany->name}"
                ])->withInput();
            }
        }

        // إعداد البيانات للتحديث
        $data = $request->except(['logo']);
        
        // التعامل مع رفع الشعار الجديد
        if ($request->hasFile('logo')) {
            // حذف الشعار القديم إذا كان موجوداً
            if ($company->logo && file_exists(public_path($company->logo))) {
                unlink(public_path($company->logo));
            }
            
            $logo = $request->file('logo');
            $logoName = time() . '_' . uniqid() . '.' . $logo->getClientOriginalExtension();
            $logo->move(public_path('uploads/logos'), $logoName);
            $data['logo'] = 'uploads/logos/' . $logoName;
        }

        $company->update($data);

        return redirect()->route('companies.index')
            ->with('success', 'تم تحديث الشركة بنجاح');
    }

    /**
     * حذف الشركة
     */
    public function destroy(Company $company)
    {
        // التحقق من عدم وجود فواتير مرتبطة
        if ($company->invoices()->count() > 0) {
            return redirect()->route('companies.index')
                ->with('error', 'لا يمكن حذف الشركة لوجود فواتير مرتبطة بها');
        }

        // حذف إعدادات ZATCA المرتبطة
        $company->zatcaSettings()->delete();

        $company->delete();

        return redirect()->route('companies.index')
            ->with('success', 'تم حذف الشركة بنجاح');
    }
}
