<?php

namespace App\Http\Controllers;

use App\Models\Category;
use Illuminate\Http\Request;
use Illuminate\Http\RedirectResponse;
use Illuminate\View\View;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;
use Illuminate\Validation\Rule;

class CategoryController extends Controller
{
    /**
     * Display a listing of categories
     */
    public function index(Request $request): View
    {
        $query = Category::with(['parent', 'creator', 'updater']);

        // Apply filters
        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('name', 'like', "%{$search}%")
                  ->orWhere('name_en', 'like', "%{$search}%")
                  ->orWhere('code', 'like', "%{$search}%")
                  ->orWhere('description', 'like', "%{$search}%");
            });
        }

        if ($request->filled('type')) {
            $query->where('type', $request->type);
        }

        if ($request->filled('status')) {
            $query->where('is_active', $request->status === 'active');
        }

        if ($request->filled('parent_id')) {
            $query->where('parent_id', $request->parent_id);
        }

        if ($request->filled('level')) {
            $query->where('level', $request->level);
        }

        // Sort
        $sortBy = $request->get('sort', 'sort_order');
        $sortDirection = $request->get('direction', 'asc');
        
        if ($sortBy === 'hierarchy') {
            $query->orderBy('level')->orderBy('sort_order')->orderBy('name');
        } else {
            $query->orderBy($sortBy, $sortDirection);
        }

        $categories = $query->paginate(20)->withQueryString();

        // Get parent categories for filter
        $parentCategories = Category::whereNull('parent_id')
            ->active()
            ->ordered()
            ->get();

        // Get statistics
        $stats = [
            'total' => Category::count(),
            'active' => Category::active()->count(),
            'inactive' => Category::where('is_active', false)->count(),
            'featured' => Category::featured()->count(),
            'products' => Category::where('type', 'product')->count(),
            'services' => Category::where('type', 'service')->count(),
            'both' => Category::where('type', 'both')->count(),
        ];

        return view('categories.index', compact('categories', 'parentCategories', 'stats'));
    }

    /**
     * Show the form for creating a new category
     */
    public function create(): View
    {
        return view('categories.create');
    }

    /**
     * Store a newly created category
     */
    public function store(Request $request): RedirectResponse
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255|unique:categories,name',
        ]);

        // Set default values for required fields
        $categoryData = [
            'name' => $validated['name'],
            'slug' => Str::slug($validated['name']),
            'code' => Category::generateUniqueCode($validated['name']),
            'type' => 'both', // Default type
            'color' => '#6c757d', // Default color
            'icon' => 'fas fa-folder', // Default icon
            'is_active' => true,
            'is_featured' => false,
            'show_on_menu' => true,
            'requires_inventory' => false,
            'sort_order' => 0,
        ];

        $category = Category::create($categoryData);

        return redirect()
            ->route('categories.index')
            ->with('success', 'تم إنشاء الفئة بنجاح');
    }

    /**
     * Display the specified category
     */
    public function show(Category $category): View
    {
        $category->load(['parent', 'children.children', 'products', 'creator', 'updater']);
        
        // Increment views
        $category->incrementViews();

        // Get category statistics
        $stats = [
            'products_count' => $category->products()->count(),
            'active_products' => $category->products()->where('is_active', true)->count(),
            'children_count' => $category->children()->count(),
            'descendants_count' => $this->getDescendantsCount($category),
            'total_sales' => $category->total_sales,
            'views_count' => $category->views_count,
        ];

        return view('categories.show', compact('category', 'stats'));
    }

    /**
     * Show the form for editing the specified category
     */
    public function edit(Category $category): View
    {
        return view('categories.edit', compact('category'));
    }

    /**
     * Update the specified category
     */
    public function update(Request $request, Category $category): RedirectResponse
    {
        $validated = $request->validate([
            'name' => [
                'required',
                'string',
                'max:255',
                Rule::unique('categories', 'name')->ignore($category->id)
            ],
        ]);

        // Generate new slug if name changed
        if ($validated['name'] !== $category->name) {
            $validated['slug'] = Str::slug($validated['name']);
        }

        $category->update($validated);

        return redirect()
            ->route('categories.index')
            ->with('success', 'تم تحديث الفئة بنجاح');
    }

    /**
     * Remove the specified category
     */
    public function destroy(Category $category): RedirectResponse
    {
        // Check if category has products
        if ($category->products()->exists()) {
            return back()->with('error', 'لا يمكن حذف الفئة لأنها تحتوي على منتجات');
        }

        // Check if category has children
        if ($category->hasChildren()) {
            return back()->with('error', 'لا يمكن حذف الفئة لأنها تحتوي على فئات فرعية');
        }

        // Delete image if exists
        if ($category->image) {
            Storage::disk('public')->delete($category->image);
        }

        $category->delete();

        return redirect()
            ->route('categories.index')
            ->with('success', 'تم حذف الفئة بنجاح');
    }

    /**
     * Toggle category status
     */
    public function toggleStatus(Category $category): RedirectResponse
    {
        $category->update(['is_active' => !$category->is_active]);
        
        $status = $category->is_active ? 'تم تفعيل' : 'تم إلغاء تفعيل';
        
        return back()->with('success', "{$status} الفئة بنجاح");
    }

    /**
     * Duplicate category
     */
    public function duplicate(Category $category): RedirectResponse
    {
        $newCategory = $category->duplicate();
        
        return redirect()
            ->route('categories.edit', $newCategory)
            ->with('success', 'تم نسخ الفئة بنجاح');
    }

    /**
     * Get categories tree (AJAX)
     */
    public function tree(): \Illuminate\Http\JsonResponse
    {
        $categories = Category::getCategoriesTree();
        
        return response()->json($categories);
    }

    /**
     * Move category to new parent (AJAX)
     */
    public function move(Request $request, Category $category): \Illuminate\Http\JsonResponse
    {
        $validated = $request->validate([
            'parent_id' => 'nullable|exists:categories,id',
            'sort_order' => 'nullable|integer|min:0',
        ]);

        // Validation
        if ($validated['parent_id'] == $category->id) {
            return response()->json(['error' => 'لا يمكن أن تكون الفئة أساسية لنفسها'], 422);
        }

        if ($validated['parent_id'] && Category::find($validated['parent_id'])->isDescendantOf($category)) {
            return response()->json(['error' => 'لا يمكن جعل فئة فرعية كفئة أساسية'], 422);
        }

        $category->update($validated);

        return response()->json(['success' => 'تم نقل الفئة بنجاح']);
    }

    /**
     * Update sort order (AJAX)
     */
    public function updateSort(Request $request): \Illuminate\Http\JsonResponse
    {
        $validated = $request->validate([
            'categories' => 'required|array',
            'categories.*.id' => 'required|exists:categories,id',
            'categories.*.sort_order' => 'required|integer|min:0',
        ]);

        foreach ($validated['categories'] as $categoryData) {
            Category::where('id', $categoryData['id'])
                ->update(['sort_order' => $categoryData['sort_order']]);
        }

        return response()->json(['success' => 'تم تحديث الترتيب بنجاح']);
    }

    /**
     * Export categories
     */
    public function export(Request $request)
    {
        $format = $request->get('format', 'excel');
        
        // This would typically use an export class
        // For now, return a simple response
        return back()->with('info', 'سيتم تطوير وظيفة التصدير قريباً');
    }

    /**
     * Get category statistics
     */
    public function statistics(): \Illuminate\Http\JsonResponse
    {
        $stats = [
            'total_categories' => Category::count(),
            'active_categories' => Category::active()->count(),
            'featured_categories' => Category::featured()->count(),
            'product_categories' => Category::where('type', 'product')->count(),
            'service_categories' => Category::where('type', 'service')->count(),
            'root_categories' => Category::root()->count(),
            'categories_with_products' => Category::has('products')->count(),
            'most_viewed' => Category::orderBy('views_count', 'desc')->take(5)->get(),
            'best_selling' => Category::orderBy('total_sales', 'desc')->take(5)->get(),
        ];

        return response()->json($stats);
    }

    /**
     * Search categories (AJAX)
     */
    public function search(Request $request): \Illuminate\Http\JsonResponse
    {
        $query = $request->get('q');
        
        $categories = Category::active()
            ->where(function ($q) use ($query) {
                $q->where('name', 'like', "%{$query}%")
                  ->orWhere('name_en', 'like', "%{$query}%")
                  ->orWhere('code', 'like', "%{$query}%");
            })
            ->limit(20)
            ->get(['id', 'name', 'name_en', 'code', 'color', 'icon']);

        return response()->json($categories);
    }

    /**
     * Get descendants count recursively
     */
    private function getDescendantsCount(Category $category): int
    {
        $count = $category->children()->count();
        
        foreach ($category->children as $child) {
            $count += $this->getDescendantsCount($child);
        }
        
        return $count;
    }
}
