<?php

namespace App\Http\Controllers;

use App\Models\CashTransaction;
use App\Models\Account;
use App\Models\Customer;
use App\Models\Supplier;
use App\Models\CostCenter;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use App\Exports\CashTransactionsExport;
use Maatwebsite\Excel\Facades\Excel;
use Barryvdh\DomPDF\Facade\Pdf;

class CashTransactionController extends Controller
{
    public function index(Request $request)
    {
        $query = CashTransaction::with(['cashAccount', 'creator']);

        if ($request->filled('type')) {
            $query->where('type', $request->type);
        }

        if ($request->filled('cash_account_id')) {
            $query->where('cash_account_id', $request->cash_account_id);
        }

        if ($request->filled('start_date') && $request->filled('end_date')) {
            $query->whereBetween('transaction_date', [$request->start_date, $request->end_date]);
        }

        if ($request->filled('search')) {
            $query->where(function ($q) use ($request) {
                $q->where('transaction_number', 'like', '%' . $request->search . '%')
                  ->orWhere('description', 'like', '%' . $request->search . '%')
                  ->orWhere('reference', 'like', '%' . $request->search . '%');
            });
        }

        $transactions = $query->orderBy('transaction_date', 'desc')
            ->orderBy('id', 'desc')
            ->paginate(20);

        $cashAccounts = Account::cashAccounts()->active()->get();

        // Calculate statistics for the same filtered data
        $allTransactions = $query->get(); // Get all without pagination for stats
        $totalReceipts = $allTransactions->where('type', 'receipt')->sum('amount');
        $totalPayments = $allTransactions->where('type', 'payment')->sum('amount');
        $netCashFlow = $totalReceipts - $totalPayments;

        return view('accounting.cash-transactions.index', compact(
            'transactions',
            'cashAccounts',
            'totalReceipts',
            'totalPayments',
            'netCashFlow'
        ));
    }

    public function create()
    {
        $cashAccounts = Account::cashAccounts()->active()->orderBy('code')->get();
        $allAccounts = Account::active()->orderBy('code')->get();
        $costCenters = CostCenter::where('is_active', true)->orderBy('code')->get();
        $customers = Customer::where('is_active', true)->orderBy('name')->get();
        $suppliers = Supplier::where('is_active', true)->orderBy('name')->get();

        return view('accounting.cash-transactions.create', compact('cashAccounts', 'allAccounts', 'costCenters', 'customers', 'suppliers'));
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'date' => 'required|date',
            'account_id' => 'required|exists:accounts,id',
            'type' => 'required|in:receipt,payment',
            'amount' => 'required|numeric|min:0.01',
            'reference' => 'nullable|string|max:255',
            'description' => 'required|string',
            'contra_account_id' => 'required|exists:accounts,id|different:account_id',
            'cost_center_id' => 'nullable|exists:cost_centers,id',
            'notes' => 'nullable|string',
            'create_journal_entry' => 'nullable|boolean',
            'is_reconciled' => 'nullable|boolean',
            'received_from' => 'nullable|string|max:255',
            'paid_to' => 'nullable|string|max:255',
            'payment_method' => 'nullable|string|max:255',
            'check_number' => 'nullable|string|max:255',
            'check_date' => 'nullable|date',
            'bank_name' => 'nullable|string|max:255',
            'attachments' => 'nullable|array'
        ]);

        DB::beginTransaction();
        try {
            $transaction = CashTransaction::create([
                'transaction_number' => CashTransaction::generateTransactionNumber($validated['type']),
                'transaction_date' => $validated['date'],
                'cash_account_id' => $validated['account_id'],
                'type' => $validated['type'],
                'amount' => $validated['amount'],
                'reference' => $validated['reference'],
                'description' => $validated['description'],
                'contra_account_id' => $validated['contra_account_id'],
                'cost_center_id' => $validated['cost_center_id'],
                'notes' => $validated['notes'],
                'is_reconciled' => $validated['is_reconciled'] ?? false,
                'received_from' => $validated['received_from'],
                'paid_to' => $validated['paid_to'],
                'payment_method' => $validated['payment_method'],
                'check_number' => $validated['check_number'],
                'check_date' => $validated['check_date'],
                'bank_name' => $validated['bank_name'],
                'created_by' => Auth::id(),
                'attachments' => $validated['attachments'] ?? []
            ]);

            // Create journal entry
            $transaction->createJournalEntry();

            DB::commit();

            return redirect()->route('cash-transactions.show', $transaction)
                ->with('success', 'تم إنشاء الحركة النقدية بنجاح');
        } catch (\Exception $e) {
            DB::rollback();
            return back()->withErrors(['error' => 'حدث خطأ أثناء إنشاء الحركة النقدية']);
        }
    }

    public function show(CashTransaction $cashTransaction)
    {
        $cashTransaction->load(['cashAccount', 'journalEntry.lines.account', 'creator']);
        
        return view('accounting.cash-transactions.show', compact('cashTransaction'));
    }

    public function edit(CashTransaction $cashTransaction)
    {
        if ($cashTransaction->journalEntry && $cashTransaction->journalEntry->status === 'posted') {
            return back()->withErrors(['error' => 'لا يمكن تعديل حركة نقدية مرحلة']);
        }

        $cashAccounts = Account::cashAccounts()->active()->orderBy('code')->get();
        $allAccounts = Account::active()->orderBy('code')->get();
        $costCenters = CostCenter::where('is_active', true)->orderBy('code')->get();
        $customers = Customer::where('is_active', true)->orderBy('name')->get();
        $suppliers = Supplier::where('is_active', true)->orderBy('name')->get();

        return view('accounting.cash-transactions.edit', compact('cashTransaction', 'cashAccounts', 'allAccounts', 'costCenters', 'customers', 'suppliers'));
    }

    public function update(Request $request, CashTransaction $cashTransaction)
    {
        if ($cashTransaction->journalEntry && $cashTransaction->journalEntry->status === 'posted') {
            return back()->withErrors(['error' => 'لا يمكن تعديل حركة نقدية مرحلة']);
        }

        $validated = $request->validate([
            'date' => 'required|date',
            'account_id' => 'required|exists:accounts,id',
            'amount' => 'required|numeric|min:0.01',
            'reference' => 'nullable|string|max:255',
            'description' => 'required|string',
            'contra_account_id' => 'required|exists:accounts,id|different:account_id',
            'cost_center_id' => 'nullable|exists:cost_centers,id',
            'notes' => 'nullable|string',
            'create_journal_entry' => 'nullable|boolean',
            'is_reconciled' => 'nullable|boolean',
            'received_from' => 'nullable|string|max:255',
            'paid_to' => 'nullable|string|max:255',
            'payment_method' => 'nullable|string|max:255',
            'check_number' => 'nullable|string|max:255',
            'check_date' => 'nullable|date',
            'bank_name' => 'nullable|string|max:255',
            'attachments' => 'nullable|array'
        ]);

        DB::beginTransaction();
        try {
            $cashTransaction->update([
                'transaction_date' => $validated['date'],
                'cash_account_id' => $validated['account_id'],
                'amount' => $validated['amount'],
                'reference' => $validated['reference'],
                'description' => $validated['description'],
                'contra_account_id' => $validated['contra_account_id'],
                'cost_center_id' => $validated['cost_center_id'],
                'notes' => $validated['notes'],
                'is_reconciled' => $validated['is_reconciled'] ?? false,
                'received_from' => $validated['received_from'],
                'paid_to' => $validated['paid_to'],
                'payment_method' => $validated['payment_method'],
                'check_number' => $validated['check_number'],
                'check_date' => $validated['check_date'],
                'bank_name' => $validated['bank_name'],
                'attachments' => $validated['attachments'] ?? []
            ]);

            // Update journal entry if exists
            if ($cashTransaction->journalEntry) {
                $cashTransaction->journalEntry->update([
                    'entry_date' => $validated['date'],
                    'description' => $validated['description'],
                    'total_debit' => $validated['amount'],
                    'total_credit' => $validated['amount']
                ]);

                // Update journal entry lines
                $cashTransaction->journalEntry->lines()->update([
                    'description' => $validated['description'],
                    'debit_amount' => $cashTransaction->type === 'receipt' ? $validated['amount'] : 0,
                    'credit_amount' => $cashTransaction->type === 'payment' ? $validated['amount'] : 0
                ]);
            }

            DB::commit();

            return redirect()->route('cash-transactions.show', $cashTransaction)
                ->with('success', 'تم تحديث الحركة النقدية بنجاح');
        } catch (\Exception $e) {
            DB::rollback();
            return back()->withErrors(['error' => 'حدث خطأ أثناء تحديث الحركة النقدية']);
        }
    }

    public function destroy(CashTransaction $cashTransaction)
    {
        if ($cashTransaction->journalEntry && $cashTransaction->journalEntry->status === 'posted') {
            return back()->withErrors(['error' => 'لا يمكن حذف حركة نقدية مرحلة']);
        }

        DB::beginTransaction();
        try {
            // Delete journal entry if exists
            if ($cashTransaction->journalEntry) {
                $cashTransaction->journalEntry->delete();
            }

            $cashTransaction->delete();

            DB::commit();

            return redirect()->route('cash-transactions.index')
                ->with('success', 'تم حذف الحركة النقدية بنجاح');
        } catch (\Exception $e) {
            DB::rollback();
            return back()->withErrors(['error' => 'حدث خطأ أثناء حذف الحركة النقدية']);
        }
    }

    public function print(CashTransaction $cashTransaction)
    {
        $cashTransaction->load(['cashAccount', 'creator']);
        
        return view('accounting.cash-transactions.print', compact('cashTransaction'));
    }

    public function cashBook(Request $request)
    {
        $startDate = $request->start_date ?? now()->startOfMonth()->format('Y-m-d');
        $endDate = $request->end_date ?? now()->endOfMonth()->format('Y-m-d');
        $cashAccountId = $request->cash_account_id;

        $cashAccounts = Account::cashAccounts()->active()->get();
        $transactions = collect();
        $openingBalance = 0;
        $closingBalance = 0;

        if ($cashAccountId) {
            $cashAccount = Account::findOrFail($cashAccountId);
            
            // Get opening balance
            $openingBalance = $cashAccount->journalEntryLines()
                ->whereHas('journalEntry', function ($query) use ($startDate) {
                    $query->where('entry_date', '<', $startDate)
                          ->where('status', 'posted');
                })
                ->sum('debit_amount') - $cashAccount->journalEntryLines()
                ->whereHas('journalEntry', function ($query) use ($startDate) {
                    $query->where('entry_date', '<', $startDate)
                          ->where('status', 'posted');
                })
                ->sum('credit_amount');

            $openingBalance += $cashAccount->opening_balance;

            // Get transactions for the period
            $transactions = CashTransaction::where('cash_account_id', $cashAccountId)
                ->whereBetween('transaction_date', [$startDate, $endDate])
                ->orderBy('transaction_date')
                ->orderBy('id')
                ->get();

            // Calculate closing balance
            $receipts = $transactions->where('type', 'receipt')->sum('amount');
            $payments = $transactions->where('type', 'payment')->sum('amount');
            $closingBalance = $openingBalance + $receipts - $payments;
        }

        return view('accounting.cash-transactions.cash-book', compact(
            'cashAccounts', 'transactions', 'startDate', 'endDate', 
            'cashAccountId', 'openingBalance', 'closingBalance'
        ));
    }

    public function exportExcel(Request $request)
    {
        $query = CashTransaction::with(['cashAccount', 'creator']);

        // Apply same filters as index method
        if ($request->filled('type')) {
            $query->where('type', $request->type);
        }

        if ($request->filled('cash_account_id')) {
            $query->where('cash_account_id', $request->cash_account_id);
        }

        if ($request->filled('start_date') && $request->filled('end_date')) {
            $query->whereBetween('transaction_date', [$request->start_date, $request->end_date]);
        }

        if ($request->filled('search')) {
            $query->where(function ($q) use ($request) {
                $q->where('transaction_number', 'like', '%' . $request->search . '%')
                  ->orWhere('description', 'like', '%' . $request->search . '%')
                  ->orWhere('reference', 'like', '%' . $request->search . '%');
            });
        }

        $transactions = $query->orderBy('transaction_date', 'desc')
            ->orderBy('id', 'desc')
            ->get();

        $filename = 'cash_transactions_' . now()->format('Y_m_d_H_i_s') . '.xlsx';

        return Excel::download(
            new CashTransactionsExport($transactions, $request->all()),
            $filename
        );
    }

    public function exportPdf(Request $request)
    {
        $query = CashTransaction::with(['cashAccount', 'creator']);

        // Apply same filters as index method
        if ($request->filled('type')) {
            $query->where('type', $request->type);
        }

        if ($request->filled('cash_account_id')) {
            $query->where('cash_account_id', $request->cash_account_id);
        }

        if ($request->filled('start_date') && $request->filled('end_date')) {
            $query->whereBetween('transaction_date', [$request->start_date, $request->end_date]);
        }

        if ($request->filled('search')) {
            $query->where(function ($q) use ($request) {
                $q->where('transaction_number', 'like', '%' . $request->search . '%')
                  ->orWhere('description', 'like', '%' . $request->search . '%')
                  ->orWhere('reference', 'like', '%' . $request->search . '%');
            });
        }

        $transactions = $query->orderBy('transaction_date', 'desc')
            ->orderBy('id', 'desc')
            ->get();

        $cashAccounts = Account::cashAccounts()->active()->get();

        // Calculate totals
        $totalReceipts = $transactions->where('type', 'receipt')->sum('amount');
        $totalPayments = $transactions->where('type', 'payment')->sum('amount');
        $netCashFlow = $totalReceipts - $totalPayments;

        $pdf = Pdf::loadView('accounting.cash-transactions.pdf', compact(
            'transactions',
            'cashAccounts',
            'totalReceipts',
            'totalPayments',
            'netCashFlow',
            'request'
        ));

        $filename = 'cash_transactions_' . now()->format('Y_m_d_H_i_s') . '.pdf';

        return $pdf->download($filename);
    }
}