<?php

namespace App\Http\Controllers;

use App\Models\Budget;
use App\Models\FiscalYear;
use App\Models\CostCenter;
use App\Models\Account;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;

class BudgetController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth');
        $this->middleware('accounting.permissions:view_budgets')->only(['index', 'show']);
        $this->middleware('accounting.permissions:create_budgets')->only(['create', 'store']);
        $this->middleware('accounting.permissions:edit_budgets')->only(['edit', 'update']);
        $this->middleware('accounting.permissions:delete_budgets')->only(['destroy']);
        $this->middleware('accounting.permissions:approve_budgets')->only(['approve', 'activate']);
    }

    /**
     * Display a listing of budgets
     */
    public function index(Request $request)
    {
        $query = Budget::with(['fiscalYear', 'costCenter', 'account', 'creator']);

        // البحث
        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('name', 'like', "%{$search}%")
                  ->orWhere('code', 'like', "%{$search}%")
                  ->orWhere('description', 'like', "%{$search}%");
            });
        }

        // فلترة حسب السنة المالية
        if ($request->filled('fiscal_year_id')) {
            $query->where('fiscal_year_id', $request->fiscal_year_id);
        }

        // فلترة حسب مركز التكلفة
        if ($request->filled('cost_center_id')) {
            $query->where('cost_center_id', $request->cost_center_id);
        }

        // فلترة حسب الحالة
        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        // فلترة حسب التنبيهات
        if ($request->filled('alert_level')) {
            switch ($request->alert_level) {
                case 'critical':
                    $query->overCriticalThreshold();
                    break;
                case 'warning':
                    $query->overWarningThreshold();
                    break;
            }
        }

        $budgets = $query->orderBy('created_at', 'desc')->paginate(15);

        // البيانات المساعدة
        $fiscalYears = FiscalYear::orderBy('start_date', 'desc')->get();
        $costCenters = CostCenter::where('is_active', true)->orderBy('name')->get();

        // إحصائيات
        $stats = [
            'total' => Budget::count(),
            'active' => Budget::where('status', 'active')->count(),
            'over_budget' => Budget::overWarningThreshold()->count(),
            'critical' => Budget::overCriticalThreshold()->count(),
            'total_budgeted' => Budget::where('is_active', true)->sum('budgeted_amount'),
            'total_actual' => Budget::where('is_active', true)->sum('actual_amount'),
        ];

        return view('budgets.index', compact('budgets', 'fiscalYears', 'costCenters', 'stats'));
    }

    /**
     * Show the form for creating a new budget
     */
    public function create()
    {
        $fiscalYears = FiscalYear::orderBy('start_date', 'desc')->get();
        $costCenters = CostCenter::where('is_active', true)->orderBy('name')->get();
        $accounts = Account::where('is_active', true)
                          ->where('type', 'expense')
                          ->orderBy('code')
                          ->get();

        // السنة المالية الحالية
        $currentFiscalYear = FiscalYear::current()->first();

        return view('budgets.create', compact('fiscalYears', 'costCenters', 'accounts', 'currentFiscalYear'));
    }

    /**
     * Store a newly created budget
     */
    public function store(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'fiscal_year_id' => 'required|exists:fiscal_years,id',
            'cost_center_id' => 'nullable|exists:cost_centers,id',
            'account_id' => 'nullable|exists:accounts,id',
            'period_type' => 'required|in:monthly,quarterly,annual',
            'start_date' => 'required|date',
            'end_date' => 'required|date|after:start_date',
            'budgeted_amount' => 'required|numeric|min:0',
            'description' => 'nullable|string|max:1000',
            'warning_threshold' => 'nullable|numeric|min:0|max:100',
            'critical_threshold' => 'nullable|numeric|min:0|max:100',
            'auto_calculate' => 'boolean',
        ]);

        DB::beginTransaction();
        
        try {
            $budget = Budget::create([
                'name' => $request->name,
                'fiscal_year_id' => $request->fiscal_year_id,
                'cost_center_id' => $request->cost_center_id,
                'account_id' => $request->account_id,
                'period_type' => $request->period_type,
                'start_date' => $request->start_date,
                'end_date' => $request->end_date,
                'budgeted_amount' => $request->budgeted_amount,
                'description' => $request->description,
                'warning_threshold' => $request->warning_threshold ?? 80,
                'critical_threshold' => $request->critical_threshold ?? 95,
                'auto_calculate' => $request->boolean('auto_calculate', true),
                'created_by' => auth()->id(),
            ]);

            // إنشاء التفصيل الشهري
            if ($request->boolean('create_breakdown')) {
                $budget->generateMonthlyBreakdown();
            }

            DB::commit();

            return redirect()->route('budgets.show', $budget)
                           ->with('success', 'تم إنشاء الميزانية بنجاح');

        } catch (\Exception $e) {
            DB::rollback();
            return back()->withErrors(['error' => 'حدث خطأ: ' . $e->getMessage()])
                        ->withInput();
        }
    }

    /**
     * Display the specified budget
     */
    public function show(Budget $budget)
    {
        $budget->load(['fiscalYear', 'costCenter', 'account', 'creator', 'approver']);

        // حساب المبلغ الفعلي إذا كان التحديث تلقائي
        if ($budget->auto_calculate) {
            $budget->calculateActualAmount();
        }

        // الإحصائيات
        $stats = [
            'consumption_percentage' => $budget->consumption_percentage,
            'remaining_amount' => $budget->remaining_amount,
            'remaining_days' => $budget->remaining_days,
            'daily_burn_rate' => $budget->daily_burn_rate,
            'actual_daily_burn_rate' => $budget->actual_daily_burn_rate,
            'period_elapsed_percentage' => $budget->period_elapsed_percentage,
        ];

        // التفصيل الشهري
        $monthlyBreakdown = $budget->monthly_breakdown;
        if ($monthlyBreakdown && $budget->auto_calculate) {
            $budget->updateMonthlyBreakdown();
            $monthlyBreakdown = $budget->monthly_breakdown;
        }

        return view('budgets.show', compact('budget', 'stats', 'monthlyBreakdown'));
    }

    /**
     * Show the form for editing the budget
     */
    public function edit(Budget $budget)
    {
        if (!$budget->canBeModified()) {
            return redirect()->route('budgets.show', $budget)
                           ->with('error', 'لا يمكن تعديل هذه الميزانية');
        }

        $fiscalYears = FiscalYear::orderBy('start_date', 'desc')->get();
        $costCenters = CostCenter::where('is_active', true)->orderBy('name')->get();
        $accounts = Account::where('is_active', true)
                          ->where('type', 'expense')
                          ->orderBy('code')
                          ->get();

        return view('budgets.edit', compact('budget', 'fiscalYears', 'costCenters', 'accounts'));
    }

    /**
     * Update the specified budget
     */
    public function update(Request $request, Budget $budget)
    {
        if (!$budget->canBeModified()) {
            return redirect()->route('budgets.show', $budget)
                           ->with('error', 'لا يمكن تعديل هذه الميزانية');
        }

        $request->validate([
            'name' => 'required|string|max:255',
            'fiscal_year_id' => 'required|exists:fiscal_years,id',
            'cost_center_id' => 'nullable|exists:cost_centers,id',
            'account_id' => 'nullable|exists:accounts,id',
            'period_type' => 'required|in:monthly,quarterly,annual',
            'start_date' => 'required|date',
            'end_date' => 'required|date|after:start_date',
            'budgeted_amount' => 'required|numeric|min:0',
            'description' => 'nullable|string|max:1000',
            'warning_threshold' => 'nullable|numeric|min:0|max:100',
            'critical_threshold' => 'nullable|numeric|min:0|max:100',
            'auto_calculate' => 'boolean',
            'is_active' => 'boolean',
        ]);

        DB::beginTransaction();
        
        try {
            $budget->update([
                'name' => $request->name,
                'fiscal_year_id' => $request->fiscal_year_id,
                'cost_center_id' => $request->cost_center_id,
                'account_id' => $request->account_id,
                'period_type' => $request->period_type,
                'start_date' => $request->start_date,
                'end_date' => $request->end_date,
                'budgeted_amount' => $request->budgeted_amount,
                'description' => $request->description,
                'warning_threshold' => $request->warning_threshold ?? 80,
                'critical_threshold' => $request->critical_threshold ?? 95,
                'auto_calculate' => $request->boolean('auto_calculate', true),
                'is_active' => $request->boolean('is_active', true),
            ]);

            DB::commit();

            return redirect()->route('budgets.show', $budget)
                           ->with('success', 'تم تحديث الميزانية بنجاح');

        } catch (\Exception $e) {
            DB::rollback();
            return back()->withErrors(['error' => 'حدث خطأ: ' . $e->getMessage()])
                        ->withInput();
        }
    }

    /**
     * Approve the budget
     */
    public function approve(Budget $budget)
    {
        if ($budget->status !== 'draft') {
            return redirect()->route('budgets.show', $budget)
                           ->with('error', 'الميزانية معتمدة مسبقاً أو في حالة غير قابلة للاعتماد');
        }

        try {
            $budget->approve(auth()->id());

            return redirect()->route('budgets.show', $budget)
                           ->with('success', 'تم اعتماد الميزانية بنجاح');

        } catch (\Exception $e) {
            return redirect()->route('budgets.show', $budget)
                           ->with('error', 'حدث خطأ: ' . $e->getMessage());
        }
    }

    /**
     * Activate the budget
     */
    public function activate(Budget $budget)
    {
        if ($budget->status !== 'approved') {
            return redirect()->route('budgets.show', $budget)
                           ->with('error', 'يجب اعتماد الميزانية أولاً قبل التفعيل');
        }

        try {
            $budget->activate();

            return redirect()->route('budgets.show', $budget)
                           ->with('success', 'تم تفعيل الميزانية بنجاح');

        } catch (\Exception $e) {
            return redirect()->route('budgets.show', $budget)
                           ->with('error', 'حدث خطأ: ' . $e->getMessage());
        }
    }

    /**
     * Close the budget
     */
    public function close(Budget $budget)
    {
        try {
            $budget->close();

            return redirect()->route('budgets.show', $budget)
                           ->with('success', 'تم إغلاق الميزانية بنجاح');

        } catch (\Exception $e) {
            return redirect()->route('budgets.show', $budget)
                           ->with('error', 'حدث خطأ: ' . $e->getMessage());
        }
    }

    /**
     * Calculate actual amounts for budgets
     */
    public function calculateActual(Request $request)
    {
        $budgetIds = $request->input('budget_ids', []);
        
        if (empty($budgetIds)) {
            return back()->with('error', 'لم يتم تحديد أي ميزانيات');
        }

        $calculated = 0;
        $errors = [];

        foreach ($budgetIds as $budgetId) {
            try {
                $budget = Budget::find($budgetId);
                if ($budget && $budget->auto_calculate) {
                    $budget->calculateActualAmount();
                    $calculated++;
                }
            } catch (\Exception $e) {
                $errors[] = "خطأ في الميزانية {$budgetId}: " . $e->getMessage();
            }
        }

        $message = "تم حساب {$calculated} ميزانية";
        if (!empty($errors)) {
            $message .= ". أخطاء: " . implode(', ', $errors);
        }

        return back()->with('success', $message);
    }

    /**
     * Budget comparison report
     */
    public function comparison(Request $request)
    {
        $fiscalYearId = $request->input('fiscal_year_id');
        $costCenterId = $request->input('cost_center_id');

        $query = Budget::with(['costCenter', 'account']);

        if ($fiscalYearId) {
            $query->where('fiscal_year_id', $fiscalYearId);
        }

        if ($costCenterId) {
            $query->where('cost_center_id', $costCenterId);
        }

        $budgets = $query->where('is_active', true)->get();

        // تجميع حسب مركز التكلفة
        $budgetsByCenter = $budgets->groupBy('cost_center_id');

        // إحصائيات إجمالية
        $totalStats = [
            'total_budgeted' => $budgets->sum('budgeted_amount'),
            'total_actual' => $budgets->sum('actual_amount'),
            'total_variance' => $budgets->sum('variance_amount'),
            'average_consumption' => $budgets->avg('consumption_percentage'),
        ];

        $fiscalYears = FiscalYear::orderBy('start_date', 'desc')->get();
        $costCenters = CostCenter::where('is_active', true)->orderBy('name')->get();

        return view('budgets.comparison', compact(
            'budgets', 'budgetsByCenter', 'totalStats', 
            'fiscalYears', 'costCenters', 'fiscalYearId', 'costCenterId'
        ));
    }

    /**
     * Budget alerts dashboard
     */
    public function alerts()
    {
        // الميزانيات المتجاوزة للحد الحرج
        $criticalBudgets = Budget::overCriticalThreshold()
            ->with(['costCenter', 'account'])
            ->orderBy('variance_percentage', 'desc')
            ->get();

        // الميزانيات المتجاوزة لحد التحذير
        $warningBudgets = Budget::overWarningThreshold()
            ->whereNotIn('id', $criticalBudgets->pluck('id'))
            ->with(['costCenter', 'account'])
            ->orderBy('variance_percentage', 'desc')
            ->get();

        // الميزانيات قريبة الانتهاء
        $expiringSoon = Budget::where('is_active', true)
            ->where('end_date', '<=', now()->addDays(30))
            ->where('end_date', '>=', now())
            ->with(['costCenter', 'account'])
            ->orderBy('end_date')
            ->get();

        return view('budgets.alerts', compact('criticalBudgets', 'warningBudgets', 'expiringSoon'));
    }

    /**
     * Remove the specified budget
     */
    public function destroy(Budget $budget)
    {
        if (!$budget->canBeModified()) {
            return redirect()->route('budgets.index')
                           ->with('error', 'لا يمكن حذف هذه الميزانية');
        }

        try {
            DB::beginTransaction();

            $budget->delete();

            DB::commit();

            return redirect()->route('budgets.index')
                           ->with('success', 'تم حذف الميزانية بنجاح');

        } catch (\Exception $e) {
            DB::rollback();
            return redirect()->route('budgets.index')
                           ->with('error', 'حدث خطأ أثناء حذف الميزانية: ' . $e->getMessage());
        }
    }
}