<?php

namespace App\Http\Controllers;

use App\Models\BankTransaction;
use App\Models\BankAccount;
use App\Models\JournalEntry;
use App\Models\JournalEntryLine;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;

class BankTransactionController extends Controller
{
    /**
     * Display a listing of bank transactions.
     */
    public function index()
    {
        $bankTransactions = BankTransaction::with('bankAccount')
            ->orderBy('transaction_date', 'desc')
            ->paginate(20);
        
        return view('bank-transactions.index', compact('bankTransactions'));
    }

    /**
     * Show the form for creating a new bank transaction.
     */
    public function create()
    {
        $bankAccounts = BankAccount::where('is_active', true)->get();
        return view('bank-transactions.create', compact('bankAccounts'));
    }

    /**
     * Store a newly created bank transaction in storage.
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'bank_account_id' => 'required|exists:bank_accounts,id',
            'type' => 'required|in:deposit,withdrawal,transfer',
            'amount' => 'required|numeric|min:0.01',
            'description' => 'required|string|max:500',
            'reference_number' => 'nullable|string|max:100',
            'transaction_date' => 'required|date',
            'status' => 'required|in:pending,cleared,cancelled',
            'to_bank_account_id' => 'nullable|exists:bank_accounts,id|different:bank_account_id',
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        DB::beginTransaction();
        try {
            $bankAccount = BankAccount::findOrFail($request->bank_account_id);
            
            // Create bank transaction
            $transaction = BankTransaction::create($request->all());

            // Create corresponding journal entry if cleared
            if ($request->status === 'cleared') {
                $this->createJournalEntry($transaction);
            }

            DB::commit();

            return redirect()->route('bank-transactions.index')
                ->with('success', 'تم إنشاء الحركة البنكية بنجاح');

        } catch (\Exception $e) {
            DB::rollback();
            return redirect()->back()
                ->with('error', 'حدث خطأ أثناء إنشاء الحركة البنكية')
                ->withInput();
        }
    }

    /**
     * Display the specified bank transaction.
     */
    public function show(BankTransaction $bankTransaction)
    {
        $bankTransaction->load('bankAccount', 'journalEntry');
        return view('bank-transactions.show', compact('bankTransaction'));
    }

    /**
     * Show the form for editing the specified bank transaction.
     */
    public function edit(BankTransaction $bankTransaction)
    {
        $bankAccounts = BankAccount::where('is_active', true)->get();
        return view('bank-transactions.edit', compact('bankTransaction', 'bankAccounts'));
    }

    /**
     * Update the specified bank transaction in storage.
     */
    public function update(Request $request, BankTransaction $bankTransaction)
    {
        $validator = Validator::make($request->all(), [
            'bank_account_id' => 'required|exists:bank_accounts,id',
            'type' => 'required|in:deposit,withdrawal,transfer',
            'amount' => 'required|numeric|min:0.01',
            'description' => 'required|string|max:500',
            'reference_number' => 'nullable|string|max:100',
            'transaction_date' => 'required|date',
            'status' => 'required|in:pending,cleared,cancelled',
            'to_bank_account_id' => 'nullable|exists:bank_accounts,id|different:bank_account_id',
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        DB::beginTransaction();
        try {
            $oldStatus = $bankTransaction->status;
            $bankTransaction->update($request->all());

            // Handle journal entry based on status change
            if ($oldStatus !== $request->status) {
                if ($request->status === 'cleared' && !$bankTransaction->journal_entry_id) {
                    $this->createJournalEntry($bankTransaction);
                } elseif ($request->status !== 'cleared' && $bankTransaction->journal_entry_id) {
                    $this->deleteJournalEntry($bankTransaction);
                }
            }

            DB::commit();

            return redirect()->route('bank-transactions.index')
                ->with('success', 'تم تحديث الحركة البنكية بنجاح');

        } catch (\Exception $e) {
            DB::rollback();
            return redirect()->back()
                ->with('error', 'حدث خطأ أثناء تحديث الحركة البنكية')
                ->withInput();
        }
    }

    /**
     * Remove the specified bank transaction from storage.
     */
    public function destroy(BankTransaction $bankTransaction)
    {
        DB::beginTransaction();
        try {
            // Delete related journal entry if exists
            if ($bankTransaction->journal_entry_id) {
                $this->deleteJournalEntry($bankTransaction);
            }

            $bankTransaction->delete();
            
            DB::commit();

            return redirect()->route('bank-transactions.index')
                ->with('success', 'تم حذف الحركة البنكية بنجاح');

        } catch (\Exception $e) {
            DB::rollback();
            return redirect()->back()
                ->with('error', 'حدث خطأ أثناء حذف الحركة البنكية');
        }
    }

    /**
     * Clear a pending transaction
     */
    public function clear(BankTransaction $bankTransaction)
    {
        if ($bankTransaction->status !== 'pending') {
            return redirect()->back()
                ->with('error', 'لا يمكن تصفية هذه الحركة');
        }

        DB::beginTransaction();
        try {
            $bankTransaction->update(['status' => 'cleared']);
            
            if (!$bankTransaction->journal_entry_id) {
                $this->createJournalEntry($bankTransaction);
            }

            DB::commit();

            return redirect()->back()
                ->with('success', 'تم تصفية الحركة البنكية بنجاح');

        } catch (\Exception $e) {
            DB::rollback();
            return redirect()->back()
                ->with('error', 'حدث خطأ أثناء تصفية الحركة البنكية');
        }
    }

    /**
     * Cancel a transaction
     */
    public function cancel(BankTransaction $bankTransaction)
    {
        if ($bankTransaction->status === 'cancelled') {
            return redirect()->back()
                ->with('error', 'الحركة ملغاة مسبقاً');
        }

        DB::beginTransaction();
        try {
            $bankTransaction->update(['status' => 'cancelled']);
            
            if ($bankTransaction->journal_entry_id) {
                $this->deleteJournalEntry($bankTransaction);
            }

            DB::commit();

            return redirect()->back()
                ->with('success', 'تم إلغاء الحركة البنكية بنجاح');

        } catch (\Exception $e) {
            DB::rollback();
            return redirect()->back()
                ->with('error', 'حدث خطأ أثناء إلغاء الحركة البنكية');
        }
    }

    /**
     * Create journal entry for bank transaction
     */
    private function createJournalEntry(BankTransaction $transaction)
    {
        $bankAccount = $transaction->bankAccount;
        
        // Generate entry number
        $entryNumber = 'JE-' . date('Ym') . '-' . str_pad(JournalEntry::count() + 1, 6, '0', STR_PAD_LEFT);

        $journalEntry = JournalEntry::create([
            'entry_number' => $entryNumber,
            'description' => 'حركة بنكية: ' . $transaction->description,
            'entry_date' => $transaction->transaction_date,
            'status' => 'posted',
            'created_by' => auth()->id(),
        ]);

        // Create journal entry lines based on transaction type
        if ($transaction->type === 'deposit') {
            // Debit bank account, Credit source (assume cash or another account)
            JournalEntryLine::create([
                'journal_entry_id' => $journalEntry->id,
                'account_id' => $bankAccount->account_id,
                'debit_amount' => $transaction->amount,
                'credit_amount' => 0,
                'description' => $transaction->description,
            ]);

            // Credit cash account (assuming cash deposit)
            $cashAccount = \App\Models\Account::where('code', '1110')->first();
            if ($cashAccount) {
                JournalEntryLine::create([
                    'journal_entry_id' => $journalEntry->id,
                    'account_id' => $cashAccount->id,
                    'debit_amount' => 0,
                    'credit_amount' => $transaction->amount,
                    'description' => $transaction->description,
                ]);
            }

        } elseif ($transaction->type === 'withdrawal') {
            // Credit bank account, Debit expense or cash
            JournalEntryLine::create([
                'journal_entry_id' => $journalEntry->id,
                'account_id' => $bankAccount->account_id,
                'debit_amount' => 0,
                'credit_amount' => $transaction->amount,
                'description' => $transaction->description,
            ]);

            // Debit cash account (assuming cash withdrawal)
            $cashAccount = \App\Models\Account::where('code', '1110')->first();
            if ($cashAccount) {
                JournalEntryLine::create([
                    'journal_entry_id' => $journalEntry->id,
                    'account_id' => $cashAccount->id,
                    'debit_amount' => $transaction->amount,
                    'credit_amount' => 0,
                    'description' => $transaction->description,
                ]);
            }

        } elseif ($transaction->type === 'transfer' && $transaction->to_bank_account_id) {
            // Transfer between bank accounts
            $toBankAccount = BankAccount::findOrFail($transaction->to_bank_account_id);

            // Credit from bank account
            JournalEntryLine::create([
                'journal_entry_id' => $journalEntry->id,
                'account_id' => $bankAccount->account_id,
                'debit_amount' => 0,
                'credit_amount' => $transaction->amount,
                'description' => $transaction->description,
            ]);

            // Debit to bank account
            JournalEntryLine::create([
                'journal_entry_id' => $journalEntry->id,
                'account_id' => $toBankAccount->account_id,
                'debit_amount' => $transaction->amount,
                'credit_amount' => 0,
                'description' => $transaction->description,
            ]);
        }

        $transaction->update(['journal_entry_id' => $journalEntry->id]);
    }

    /**
     * Delete journal entry for bank transaction
     */
    private function deleteJournalEntry(BankTransaction $transaction)
    {
        if ($transaction->journal_entry_id) {
            $journalEntry = JournalEntry::find($transaction->journal_entry_id);
            if ($journalEntry) {
                $journalEntry->lines()->delete();
                $journalEntry->delete();
            }
            $transaction->update(['journal_entry_id' => null]);
        }
    }
}