<?php

namespace App\Http\Controllers;

use App\Models\BankAccount;
use App\Models\Account;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class BankAccountController extends Controller
{
    /**
     * Display a listing of bank accounts.
     */
    public function index()
    {
        $bankAccounts = BankAccount::with('account')->get();
        return view('bank-accounts.index', compact('bankAccounts'));
    }

    /**
     * Show the form for creating a new bank account.
     */
    public function create()
    {
        $bankingAccounts = Account::where('type', 'banking')->get();
        return view('bank-accounts.create', compact('bankingAccounts'));
    }

    /**
     * Store a newly created bank account in storage.
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'account_id' => 'required|exists:accounts,id',
            'bank_name' => 'required|string|max:255',
            'account_number' => 'required|string|max:100|unique:bank_accounts',
            'iban' => 'nullable|string|max:34',
            'swift_code' => 'nullable|string|max:11',
            'branch' => 'nullable|string|max:255',
            'currency' => 'required|string|max:3|default:SAR',
            'is_active' => 'boolean'
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        BankAccount::create($request->all());

        return redirect()->route('bank-accounts.index')
            ->with('success', 'تم إنشاء الحساب البنكي بنجاح');
    }

    /**
     * Display the specified bank account.
     */
    public function show(BankAccount $bankAccount)
    {
        $bankAccount->load('account', 'transactions');
        return view('bank-accounts.show', compact('bankAccount'));
    }

    /**
     * Show the form for editing the specified bank account.
     */
    public function edit(BankAccount $bankAccount)
    {
        $bankingAccounts = Account::where('type', 'banking')->get();
        return view('bank-accounts.edit', compact('bankAccount', 'bankingAccounts'));
    }

    /**
     * Update the specified bank account in storage.
     */
    public function update(Request $request, BankAccount $bankAccount)
    {
        $validator = Validator::make($request->all(), [
            'account_id' => 'required|exists:accounts,id',
            'bank_name' => 'required|string|max:255',
            'account_number' => 'required|string|max:100|unique:bank_accounts,account_number,' . $bankAccount->id,
            'iban' => 'nullable|string|max:34',
            'swift_code' => 'nullable|string|max:11',
            'branch' => 'nullable|string|max:255',
            'currency' => 'required|string|max:3',
            'is_active' => 'boolean'
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        $bankAccount->update($request->all());

        return redirect()->route('bank-accounts.index')
            ->with('success', 'تم تحديث الحساب البنكي بنجاح');
    }

    /**
     * Remove the specified bank account from storage.
     */
    public function destroy(BankAccount $bankAccount)
    {
        // Check if bank account has transactions
        if ($bankAccount->transactions()->count() > 0) {
            return redirect()->back()
                ->with('error', 'لا يمكن حذف الحساب البنكي لوجود حركات مرتبطة به');
        }

        $bankAccount->delete();

        return redirect()->route('bank-accounts.index')
            ->with('success', 'تم حذف الحساب البنكي بنجاح');
    }

    /**
     * Toggle bank account status
     */
    public function toggleStatus(BankAccount $bankAccount)
    {
        $bankAccount->update(['is_active' => !$bankAccount->is_active]);
        
        $status = $bankAccount->is_active ? 'تفعيل' : 'إلغاء تفعيل';
        return redirect()->back()
            ->with('success', "تم {$status} الحساب البنكي بنجاح");
    }

    /**
     * Get bank account statement
     */
    public function statement(BankAccount $bankAccount, Request $request)
    {
        $startDate = $request->input('start_date', now()->startOfMonth());
        $endDate = $request->input('end_date', now()->endOfMonth());

        $transactions = $bankAccount->transactions()
            ->whereBetween('transaction_date', [$startDate, $endDate])
            ->orderBy('transaction_date', 'desc')
            ->get();

        $openingBalance = $bankAccount->transactions()
            ->where('transaction_date', '<', $startDate)
            ->sum('amount');

        return view('bank-accounts.statement', compact(
            'bankAccount', 
            'transactions', 
            'openingBalance', 
            'startDate', 
            'endDate'
        ));
    }
}