<?php

namespace App\Http\Controllers;

use App\Models\AuditLog;
use App\Models\User;
use Illuminate\Http\Request;
use Carbon\Carbon;
use Illuminate\Support\Facades\DB;

class AuditLogController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth');
        $this->middleware('accounting.permissions:view_audit_logs');
    }

    /**
     * Display a listing of audit logs
     */
    public function index(Request $request)
    {
        $query = AuditLog::with(['user']);

        // البحث
        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('event', 'like', "%{$search}%")
                  ->orWhere('auditable_type', 'like', "%{$search}%")
                  ->orWhere('description', 'like', "%{$search}%")
                  ->orWhere('url', 'like', "%{$search}%");
            });
        }

        // فلترة حسب نوع الحدث
        if ($request->filled('event')) {
            $query->where('event', $request->event);
        }

        // فلترة حسب نوع النموذج
        if ($request->filled('auditable_type')) {
            $query->where('auditable_type', $request->auditable_type);
        }

        // فلترة حسب المستخدم
        if ($request->filled('user_id')) {
            $query->where('user_id', $request->user_id);
        }

        // فلترة حسب التاريخ
        if ($request->filled('date_from')) {
            $query->where('created_at', '>=', $request->date_from);
        }

        if ($request->filled('date_to')) {
            $query->where('created_at', '<=', $request->date_to . ' 23:59:59');
        }

        // فلترة الأحداث المحاسبية فقط
        if ($request->boolean('accounting_only')) {
            $query->accountingEvents();
        }

        // فلترة الأحداث الحساسة فقط
        if ($request->boolean('critical_only')) {
            $query->criticalEvents();
        }

        // فلترة حسب عنوان IP
        if ($request->filled('ip_address')) {
            $query->where('ip_address', $request->ip_address);
        }

        $auditLogs = $query->orderBy('created_at', 'desc')->paginate(20);

        // البيانات المساعدة للفلاتر
        $events = AuditLog::distinct('event')->pluck('event')->sort();
        $auditableTypes = AuditLog::distinct('auditable_type')->pluck('auditable_type')->sort();
        $users = User::orderBy('name')->get();

        // إحصائيات سريعة
        $stats = [
            'total_today' => AuditLog::whereDate('created_at', today())->count(),
            'total_week' => AuditLog::whereBetween('created_at', [now()->startOfWeek(), now()->endOfWeek()])->count(),
            'total_month' => AuditLog::whereMonth('created_at', now()->month)->count(),
            'critical_today' => AuditLog::criticalEvents()->whereDate('created_at', today())->count(),
        ];

        return view('audit-logs.index', compact(
            'auditLogs', 'events', 'auditableTypes', 'users', 'stats'
        ));
    }

    /**
     * Display the specified audit log
     */
    public function show(AuditLog $auditLog)
    {
        $auditLog->load(['user', 'auditable']);

        // الحصول على السجلات المرتبطة بنفس النموذج
        $relatedLogs = AuditLog::where('auditable_type', $auditLog->auditable_type)
            ->where('auditable_id', $auditLog->auditable_id)
            ->where('id', '!=', $auditLog->id)
            ->with('user')
            ->orderBy('created_at', 'desc')
            ->limit(10)
            ->get();

        return view('audit-logs.show', compact('auditLog', 'relatedLogs'));
    }

    /**
     * Show audit statistics dashboard
     */
    public function dashboard(Request $request)
    {
        $dateRange = $request->input('range', '30'); // آخر 30 يوم افتراضياً
        $startDate = now()->subDays($dateRange);
        $endDate = now();

        // إحصائيات عامة
        $generalStats = AuditLog::betweenDates($startDate, $endDate)->selectRaw('
            COUNT(*) as total_events,
            COUNT(DISTINCT user_id) as unique_users,
            COUNT(DISTINCT ip_address) as unique_ips,
            COUNT(DISTINCT auditable_type) as unique_models
        ')->first();

        // الأحداث حسب النوع
        $eventsByType = AuditLog::betweenDates($startDate, $endDate)
            ->groupBy('event')
            ->selectRaw('event, COUNT(*) as count')
            ->orderBy('count', 'desc')
            ->get();

        // الأحداث حسب النموذج
        $eventsByModel = AuditLog::betweenDates($startDate, $endDate)
            ->groupBy('auditable_type')
            ->selectRaw('auditable_type, COUNT(*) as count')
            ->orderBy('count', 'desc')
            ->limit(10)
            ->get();

        // النشاط حسب المستخدم
        $activityByUser = AuditLog::betweenDates($startDate, $endDate)
            ->whereNotNull('user_id')
            ->with('user:id,name')
            ->groupBy('user_id')
            ->selectRaw('user_id, COUNT(*) as count')
            ->orderBy('count', 'desc')
            ->limit(10)
            ->get();

        // النشاط اليومي
        $dailyActivity = AuditLog::betweenDates($startDate, $endDate)
            ->groupBy(DB::raw('DATE(created_at)'))
            ->selectRaw('DATE(created_at) as date, COUNT(*) as count')
            ->orderBy('date')
            ->get();

        // الأحداث الحساسة الأخيرة
        $recentCriticalEvents = AuditLog::criticalEvents()
            ->betweenDates($startDate, $endDate)
            ->with(['user', 'auditable'])
            ->orderBy('created_at', 'desc')
            ->limit(10)
            ->get();

        // نشاط الساعات (آخر 24 ساعة)
        $hourlyActivity = AuditLog::where('created_at', '>=', now()->subDay())
            ->groupBy(DB::raw('HOUR(created_at)'))
            ->selectRaw('HOUR(created_at) as hour, COUNT(*) as count')
            ->orderBy('hour')
            ->get();

        // أكثر عناوين IP نشاطاً
        $topIPs = AuditLog::betweenDates($startDate, $endDate)
            ->whereNotNull('ip_address')
            ->groupBy('ip_address')
            ->selectRaw('ip_address, COUNT(*) as count')
            ->orderBy('count', 'desc')
            ->limit(10)
            ->get();

        return view('audit-logs.dashboard', compact(
            'generalStats', 'eventsByType', 'eventsByModel', 'activityByUser',
            'dailyActivity', 'recentCriticalEvents', 'hourlyActivity', 'topIPs',
            'dateRange', 'startDate', 'endDate'
        ));
    }

    /**
     * Show audit trail for specific model
     */
    public function trail(Request $request)
    {
        $modelType = $request->input('model_type');
        $modelId = $request->input('model_id');

        if (!$modelType || !$modelId) {
            return redirect()->route('audit-logs.index')
                           ->with('error', 'يجب تحديد نوع النموذج والمعرف');
        }

        $auditTrail = AuditLog::forModel($modelType, $modelId)
            ->with('user')
            ->orderBy('created_at', 'desc')
            ->paginate(20);

        // معلومات النموذج
        $modelInfo = null;
        if (class_exists($modelType)) {
            try {
                $modelInfo = $modelType::find($modelId);
            } catch (\Exception $e) {
                // النموذج قد يكون محذوف
            }
        }

        return view('audit-logs.trail', compact('auditTrail', 'modelType', 'modelId', 'modelInfo'));
    }

    /**
     * Export audit logs
     */
    public function export(Request $request)
    {
        $request->validate([
            'format' => 'required|in:excel,csv,pdf',
            'date_from' => 'required|date',
            'date_to' => 'required|date|after_or_equal:date_from',
        ]);

        $query = AuditLog::with(['user'])
            ->betweenDates($request->date_from, $request->date_to . ' 23:59:59');

        // تطبيق الفلاتر
        if ($request->filled('event')) {
            $query->where('event', $request->event);
        }

        if ($request->filled('auditable_type')) {
            $query->where('auditable_type', $request->auditable_type);
        }

        if ($request->filled('user_id')) {
            $query->where('user_id', $request->user_id);
        }

        if ($request->boolean('accounting_only')) {
            $query->accountingEvents();
        }

        if ($request->boolean('critical_only')) {
            $query->criticalEvents();
        }

        $auditLogs = $query->orderBy('created_at', 'desc')->get();

        // تحديد فئة التصدير حسب التنسيق
        $exportClass = match($request->format) {
            'excel' => 'App\Exports\AuditLogsExport',
            'csv' => 'App\Exports\AuditLogsCsvExport',
            'pdf' => 'App\Exports\AuditLogsPdfExport',
        };

        $fileName = 'audit_logs_' . $request->date_from . '_to_' . $request->date_to;

        return (new $exportClass($auditLogs))->download($fileName . '.' . $request->format);
    }

    /**
     * Clean up old audit logs
     */
    public function cleanup(Request $request)
    {
        $request->validate([
            'days_to_keep' => 'required|integer|min:30|max:3650', // من 30 يوم إلى 10 سنوات
            'confirm' => 'required|accepted',
        ]);

        try {
            $deletedCount = AuditLog::cleanupOldLogs($request->days_to_keep);

            return redirect()->route('audit-logs.index')
                           ->with('success', "تم حذف {$deletedCount} سجل قديم بنجاح");

        } catch (\Exception $e) {
            return redirect()->route('audit-logs.index')
                           ->with('error', 'حدث خطأ أثناء تنظيف السجلات: ' . $e->getMessage());
        }
    }

    /**
     * Show security incidents
     */
    public function security(Request $request)
    {
        $dateRange = $request->input('range', '7'); // آخر 7 أيام
        $startDate = now()->subDays($dateRange);

        // محاولات الدخول المتعددة من نفس IP
        $suspiciousIPs = AuditLog::where('created_at', '>=', $startDate)
            ->whereNotNull('ip_address')
            ->groupBy('ip_address')
            ->havingRaw('COUNT(*) > 100') // أكثر من 100 عملية
            ->selectRaw('ip_address, COUNT(*) as count')
            ->orderBy('count', 'desc')
            ->get();

        // عمليات الحذف المتعددة
        $deletionEvents = AuditLog::where('created_at', '>=', $startDate)
            ->where('event', 'deleted')
            ->with(['user', 'auditable'])
            ->orderBy('created_at', 'desc')
            ->limit(20)
            ->get();

        // تعديلات خارج ساعات العمل
        $afterHoursActivity = AuditLog::where('created_at', '>=', $startDate)
            ->whereRaw('HOUR(created_at) NOT BETWEEN 8 AND 18') // خارج 8 صباحاً - 6 مساءً
            ->criticalEvents()
            ->with(['user'])
            ->orderBy('created_at', 'desc')
            ->limit(20)
            ->get();

        // نشاط المستخدمين غير المعتادين
        $unusualActivity = AuditLog::where('created_at', '>=', $startDate)
            ->whereNotNull('user_id')
            ->groupBy('user_id')
            ->havingRaw('COUNT(*) > 200') // أكثر من 200 عملية
            ->with('user')
            ->selectRaw('user_id, COUNT(*) as count')
            ->orderBy('count', 'desc')
            ->get();

        return view('audit-logs.security', compact(
            'suspiciousIPs', 'deletionEvents', 'afterHoursActivity', 
            'unusualActivity', 'dateRange'
        ));
    }

    /**
     * Show compliance report
     */
    public function compliance(Request $request)
    {
        $startDate = $request->input('start_date', now()->subMonth()->toDateString());
        $endDate = $request->input('end_date', now()->toDateString());

        // إحصائيات الامتثال
        $complianceStats = AuditLog::getAuditStats($startDate, $endDate);

        // العمليات المحاسبية المُراجعة
        $accountingEvents = AuditLog::accountingEvents()
            ->betweenDates($startDate, $endDate)
            ->groupBy('event')
            ->selectRaw('event, COUNT(*) as count')
            ->orderBy('count', 'desc')
            ->get();

        // المستخدمون النشطون
        $activeUsers = AuditLog::betweenDates($startDate, $endDate)
            ->whereNotNull('user_id')
            ->with('user')
            ->groupBy('user_id')
            ->selectRaw('user_id, COUNT(*) as activity_count, MIN(created_at) as first_activity, MAX(created_at) as last_activity')
            ->orderBy('activity_count', 'desc')
            ->get();

        // العمليات الحساسة
        $sensitiveOperations = AuditLog::criticalEvents()
            ->betweenDates($startDate, $endDate)
            ->with(['user', 'auditable'])
            ->orderBy('created_at', 'desc')
            ->get();

        return view('audit-logs.compliance', compact(
            'complianceStats', 'accountingEvents', 'activeUsers', 
            'sensitiveOperations', 'startDate', 'endDate'
        ));
    }
}