<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Invoice;
use App\Models\ZatcaSettings;
use App\Services\ZatcaService;
use App\Jobs\SyncZatcaInvoicesJob;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Validator;

class ZatcaSyncController extends Controller
{
    /**
     * الحصول على إحصائيات المزامنة
     */
    public function getStats(): JsonResponse
    {
        try {
            $stats = [
                'total_sent' => Invoice::whereNotNull('sent_to_zatca_at')->count(),
                'cleared' => Invoice::where('status', 'cleared')->count(),
                'reported' => Invoice::where('status', 'reported')->count(),
                'rejected' => Invoice::where('status', 'rejected')->count(),
                'pending_sync' => Invoice::whereNotNull('sent_to_zatca_at')
                    ->where('updated_at', '>', now()->subHours(24))
                    ->count(),
                'last_sync' => $this->getLastSyncInfo()
            ];

            return response()->json([
                'success' => true,
                'data' => $stats
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'فشل في جلب الإحصائيات: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * مزامنة فاتورة واحدة
     */
    public function syncSingle(Request $request, Invoice $invoice): JsonResponse
    {
        if (!$invoice->sent_to_zatca_at) {
            return response()->json([
                'success' => false,
                'message' => 'لم يتم إرسال هذه الفاتورة إلى ZATCA بعد'
            ], 400);
        }

        try {
            $zatcaSettings = ZatcaSettings::where('company_id', $invoice->company_id)
                ->where('is_active', true)
                ->first();

            if (!$zatcaSettings) {
                return response()->json([
                    'success' => false,
                    'message' => 'لم يتم إعداد اتصال ZATCA للشركة'
                ], 400);
            }

            $zatcaService = new ZatcaService($zatcaSettings);
            $result = $zatcaService->checkInvoiceStatus($invoice);

            return response()->json([
                'success' => true,
                'message' => 'تم تحديث حالة الفاتورة بنجاح',
                'data' => [
                    'invoice_id' => $invoice->id,
                    'invoice_number' => $invoice->invoice_number,
                    'old_status' => $invoice->getOriginal('status'),
                    'new_status' => $invoice->status,
                    'zatca_response' => $result
                ]
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'فشل في مزامنة الفاتورة: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * مزامنة شاملة
     */
    public function syncAll(Request $request): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'company_id' => 'nullable|exists:companies,id',
            'environment' => 'nullable|in:sandbox,production',
            'background' => 'boolean',
            'force' => 'boolean'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'بيانات غير صالحة',
                'errors' => $validator->errors()
            ], 422);
        }

        try {
            $query = ZatcaSettings::where('is_active', true);

            if ($request->company_id) {
                $query->where('company_id', $request->company_id);
            }

            if ($request->environment) {
                $query->where('environment', $request->environment);
            }

            $zatcaSettings = $query->get();

            if ($zatcaSettings->isEmpty()) {
                return response()->json([
                    'success' => false,
                    'message' => 'لم يتم العثور على إعدادات ZATCA نشطة'
                ], 400);
            }

            if ($request->boolean('background')) {
                // تشغيل في الخلفية
                foreach ($zatcaSettings as $settings) {
                    SyncZatcaInvoicesJob::dispatch($settings->id, null, auth()->id());
                }

                return response()->json([
                    'success' => true,
                    'message' => 'تم إرسال مهام المزامنة للخلفية',
                    'data' => [
                        'companies_count' => $zatcaSettings->count(),
                        'background' => true
                    ]
                ]);
            }

            // تشغيل مباشر
            $results = [
                'total_companies' => $zatcaSettings->count(),
                'success' => 0,
                'failed' => 0,
                'details' => []
            ];

            foreach ($zatcaSettings as $settings) {
                try {
                    $invoices = Invoice::where('company_id', $settings->company_id)
                        ->whereNotNull('sent_to_zatca_at')
                        ->when(!$request->boolean('force'), function($query) {
                            return $query->where('updated_at', '>', now()->subDays(7));
                        })
                        ->get();

                    if ($invoices->isEmpty()) {
                        $results['details'][$settings->company->name] = [
                            'success' => 0,
                            'failed' => 0,
                            'message' => 'لا توجد فواتير للمزامنة'
                        ];
                        continue;
                    }

                    $zatcaService = new ZatcaService($settings);
                    $companyResults = $zatcaService->syncMultipleInvoices($invoices);
                    
                    $results['success'] += $companyResults['success'];
                    $results['failed'] += $companyResults['failed'];
                    $results['details'][$settings->company->name] = $companyResults;

                } catch (\Exception $e) {
                    $results['failed']++;
                    $results['details'][$settings->company->name] = [
                        'success' => 0,
                        'failed' => 1,
                        'error' => $e->getMessage()
                    ];
                }
            }

            return response()->json([
                'success' => true,
                'message' => "تمت المزامنة - نجح: {$results['success']}, فشل: {$results['failed']}",
                'data' => $results
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'فشل في المزامنة الشاملة: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * الحصول على سجل العمليات
     */
    public function getOperationLog(Request $request): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'company_id' => 'nullable|exists:companies,id',
            'operation' => 'nullable|string',
            'limit' => 'nullable|integer|min:1|max:100'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'بيانات غير صالحة',
                'errors' => $validator->errors()
            ], 422);
        }

        try {
            $query = ZatcaSettings::where('is_active', true);

            if ($request->company_id) {
                $query->where('company_id', $request->company_id);
            }

            $settings = $query->with('company')->get();
            $logs = [];

            foreach ($settings as $setting) {
                if ($setting->operation_log) {
                    $companyLogs = collect($setting->operation_log);

                    if ($request->operation) {
                        $companyLogs = $companyLogs->where('operation', $request->operation);
                    }

                    $companyLogs = $companyLogs->map(function ($log) use ($setting) {
                        return array_merge($log, [
                            'company_name' => $setting->company->name,
                            'company_id' => $setting->company_id
                        ]);
                    });

                    $logs = array_merge($logs, $companyLogs->toArray());
                }
            }

            // ترتيب حسب التاريخ
            usort($logs, function ($a, $b) {
                return strtotime($b['timestamp']) - strtotime($a['timestamp']);
            });

            // تحديد العدد
            $limit = $request->limit ?? 50;
            $logs = array_slice($logs, 0, $limit);

            return response()->json([
                'success' => true,
                'data' => [
                    'logs' => $logs,
                    'total' => count($logs),
                    'limit' => $limit
                ]
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'فشل في جلب سجل العمليات: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * فحص حالة ZATCA
     */
    public function checkStatus(): JsonResponse
    {
        try {
            $activeSettings = ZatcaSettings::where('is_active', true)->with('company')->get();
            $status = [];

            foreach ($activeSettings as $setting) {
                $companyStatus = [
                    'company_id' => $setting->company_id,
                    'company_name' => $setting->company->name,
                    'environment' => $setting->environment,
                    'is_valid' => $setting->isValid(),
                    'certificate_expires_at' => $setting->certificate_expires_at,
                    'certificate_expired' => $setting->isCertificateExpired(),
                    'token_expired' => $setting->isTokenExpired(),
                    'last_connection_test' => $setting->last_connection_test,
                    'connection_status' => $setting->connection_status,
                    'last_error' => $setting->last_error
                ];

                $status[] = $companyStatus;
            }

            return response()->json([
                'success' => true,
                'data' => [
                    'companies' => $status,
                    'total_companies' => count($status),
                    'healthy_companies' => collect($status)->where('is_valid', true)->count(),
                    'expired_certificates' => collect($status)->where('certificate_expired', true)->count()
                ]
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'فشل في فحص حالة ZATCA: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * الحصول على معلومات آخر مزامنة
     */
    private function getLastSyncInfo(): ?array
    {
        $settings = ZatcaSettings::where('is_active', true)->get();
        $lastSync = null;

        foreach ($settings as $setting) {
            if ($setting->operation_log) {
                $logs = collect($setting->operation_log)
                    ->where('operation', 'sync')
                    ->sortByDesc('timestamp');

                if ($logs->isNotEmpty()) {
                    $latest = $logs->first();
                    if (!$lastSync || strtotime($latest['timestamp']) > strtotime($lastSync['timestamp'])) {
                        $lastSync = array_merge($latest, [
                            'company_name' => $setting->company->name
                        ]);
                    }
                }
            }
        }

        return $lastSync;
    }
}