<?php

namespace App\Http\Controllers;

use App\Models\AccountingSetting;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;

class AccountingSettingController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth');
        $this->middleware('accounting.permissions:view_settings')->only(['index', 'show']);
        $this->middleware('accounting.permissions:edit_settings')->only(['edit', 'update']);
        $this->middleware('accounting.permissions:create_settings')->only(['create', 'store']);
        $this->middleware('accounting.permissions:delete_settings')->only(['destroy']);
    }

    /**
     * Display a listing of accounting settings
     */
    public function index(Request $request)
    {
        $query = AccountingSetting::query();

        // البحث
        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('key', 'like', "%{$search}%")
                  ->orWhere('label', 'like', "%{$search}%")
                  ->orWhere('description', 'like', "%{$search}%")
                  ->orWhere('value', 'like', "%{$search}%");
            });
        }

        // فلترة حسب المجموعة
        if ($request->filled('group')) {
            $query->where('group', $request->group);
        }

        // فلترة حسب النوع
        if ($request->filled('type')) {
            $query->where('type', $request->type);
        }

        // فلترة القابلة للتعديل فقط
        if ($request->boolean('editable_only')) {
            $query->editable();
        }

        // فلترة العامة فقط
        if ($request->boolean('public_only')) {
            $query->public();
        }

        $settings = $query->ordered()->paginate(20);

        // تجميع الإعدادات حسب المجموعة
        $settingsByGroup = $settings->getCollection()->groupBy('group');

        // الحصول على المجموعات والأنواع للفلاتر
        $groups = AccountingSetting::distinct('group')->pluck('group')->sort();
        $types = AccountingSetting::distinct('type')->pluck('type')->sort();

        // إحصائيات
        $stats = [
            'total' => AccountingSetting::count(),
            'editable' => AccountingSetting::editable()->count(),
            'public' => AccountingSetting::public()->count(),
            'groups' => AccountingSetting::distinct('group')->count(),
        ];

        return view('accounting-settings.index', compact(
            'settings', 'settingsByGroup', 'groups', 'types', 'stats'
        ));
    }

    /**
     * Show the form for creating a new setting
     */
    public function create()
    {
        $groups = AccountingSetting::distinct('group')->pluck('group')->sort();
        $types = ['text', 'number', 'boolean', 'json', 'array', 'date', 'datetime'];

        return view('accounting-settings.create', compact('groups', 'types'));
    }

    /**
     * Store a newly created setting
     */
    public function store(Request $request)
    {
        $request->validate([
            'key' => 'required|string|max:100|unique:accounting_settings|regex:/^[a-z_]+$/',
            'label' => 'required|string|max:255',
            'group' => 'required|string|max:50',
            'type' => 'required|string|in:text,number,integer,float,boolean,json,array,date,datetime',
            'value' => 'nullable|string',
            'default_value' => 'nullable|string',
            'description' => 'nullable|string|max:1000',
            'validation_rules' => 'nullable|string',
            'options' => 'nullable|json',
            'is_public' => 'boolean',
            'is_editable' => 'boolean',
            'requires_restart' => 'boolean',
            'sort_order' => 'nullable|integer|min:0',
        ]);

        try {
            DB::beginTransaction();

            $setting = AccountingSetting::create([
                'key' => $request->key,
                'label' => $request->label,
                'group' => $request->group,
                'type' => $request->type,
                'value' => $request->value,
                'default_value' => $request->default_value,
                'description' => $request->description,
                'validation_rules' => $request->validation_rules,
                'options' => $request->options ? json_decode($request->options, true) : null,
                'is_public' => $request->boolean('is_public', false),
                'is_editable' => $request->boolean('is_editable', true),
                'requires_restart' => $request->boolean('requires_restart', false),
                'sort_order' => $request->sort_order ?? 0,
            ]);

            DB::commit();

            return redirect()->route('accounting-settings.index')
                           ->with('success', 'تم إنشاء الإعداد بنجاح');

        } catch (\Exception $e) {
            DB::rollback();
            return back()->withErrors(['error' => 'حدث خطأ: ' . $e->getMessage()])
                        ->withInput();
        }
    }

    /**
     * Display the specified setting
     */
    public function show(AccountingSetting $accountingSetting)
    {
        return view('accounting-settings.show', compact('accountingSetting'));
    }

    /**
     * Show the form for editing the setting
     */
    public function edit(AccountingSetting $accountingSetting)
    {
        if (!$accountingSetting->is_editable) {
            return redirect()->route('accounting-settings.show', $accountingSetting)
                           ->with('error', 'هذا الإعداد غير قابل للتعديل');
        }

        $groups = AccountingSetting::distinct('group')->pluck('group')->sort();
        $types = ['text', 'number', 'boolean', 'json', 'array', 'date', 'datetime'];

        return view('accounting-settings.edit', compact('accountingSetting', 'groups', 'types'));
    }

    /**
     * Update the specified setting
     */
    public function update(Request $request, AccountingSetting $accountingSetting)
    {
        if (!$accountingSetting->is_editable) {
            return redirect()->route('accounting-settings.show', $accountingSetting)
                           ->with('error', 'هذا الإعداد غير قابل للتعديل');
        }

        $request->validate([
            'key' => 'required|string|max:100|unique:accounting_settings,key,' . $accountingSetting->id . '|regex:/^[a-z_]+$/',
            'label' => 'required|string|max:255',
            'group' => 'required|string|max:50',
            'type' => 'required|string|in:text,number,integer,float,boolean,json,array,date,datetime',
            'value' => 'nullable|string',
            'default_value' => 'nullable|string',
            'description' => 'nullable|string|max:1000',
            'validation_rules' => 'nullable|string',
            'options' => 'nullable|json',
            'is_public' => 'boolean',
            'is_editable' => 'boolean',
            'requires_restart' => 'boolean',
            'sort_order' => 'nullable|integer|min:0',
        ]);

        try {
            DB::beginTransaction();

            $accountingSetting->update([
                'key' => $request->key,
                'label' => $request->label,
                'group' => $request->group,
                'type' => $request->type,
                'value' => $request->value,
                'default_value' => $request->default_value,
                'description' => $request->description,
                'validation_rules' => $request->validation_rules,
                'options' => $request->options ? json_decode($request->options, true) : null,
                'is_public' => $request->boolean('is_public', false),
                'is_editable' => $request->boolean('is_editable', true),
                'requires_restart' => $request->boolean('requires_restart', false),
                'sort_order' => $request->sort_order ?? 0,
            ]);

            DB::commit();

            return redirect()->route('accounting-settings.show', $accountingSetting)
                           ->with('success', 'تم تحديث الإعداد بنجاح');

        } catch (\Exception $e) {
            DB::rollback();
            return back()->withErrors(['error' => 'حدث خطأ: ' . $e->getMessage()])
                        ->withInput();
        }
    }

    /**
     * Update setting value quickly
     */
    public function updateValue(Request $request, AccountingSetting $accountingSetting)
    {
        if (!$accountingSetting->is_editable) {
            return response()->json(['error' => 'هذا الإعداد غير قابل للتعديل'], 403);
        }

        $request->validate([
            'value' => 'required',
        ]);

        try {
            // التحقق من صحة القيمة حسب نوع الإعداد
            $validator = $this->validateSettingValue($accountingSetting, $request->value);
            
            if ($validator->fails()) {
                return response()->json(['errors' => $validator->errors()], 422);
            }

            $accountingSetting->updateValue($request->value);

            return response()->json([
                'success' => true,
                'message' => 'تم تحديث القيمة بنجاح',
                'new_value' => $accountingSetting->fresh()->typed_value,
            ]);

        } catch (\Exception $e) {
            return response()->json(['error' => $e->getMessage()], 500);
        }
    }

    /**
     * Bulk update settings
     */
    public function bulkUpdate(Request $request)
    {
        $request->validate([
            'settings' => 'required|array',
            'settings.*.id' => 'required|exists:accounting_settings,id',
            'settings.*.value' => 'required',
        ]);

        $updated = 0;
        $errors = [];

        DB::beginTransaction();

        try {
            foreach ($request->settings as $settingData) {
                $setting = AccountingSetting::find($settingData['id']);
                
                if (!$setting->is_editable) {
                    $errors[] = "الإعداد {$setting->label} غير قابل للتعديل";
                    continue;
                }

                try {
                    $setting->updateValue($settingData['value']);
                    $updated++;
                } catch (\Exception $e) {
                    $errors[] = "خطأ في {$setting->label}: " . $e->getMessage();
                }
            }

            DB::commit();

            $message = "تم تحديث {$updated} إعداد";
            if (!empty($errors)) {
                $message .= ". أخطاء: " . implode(', ', $errors);
            }

            return redirect()->route('accounting-settings.index')
                           ->with('success', $message);

        } catch (\Exception $e) {
            DB::rollback();
            return redirect()->route('accounting-settings.index')
                           ->with('error', 'حدث خطأ أثناء التحديث: ' . $e->getMessage());
        }
    }

    /**
     * Reset setting to default value
     */
    public function reset(AccountingSetting $accountingSetting)
    {
        if (!$accountingSetting->is_editable) {
            return redirect()->route('accounting-settings.show', $accountingSetting)
                           ->with('error', 'هذا الإعداد غير قابل للتعديل');
        }

        if (empty($accountingSetting->default_value)) {
            return redirect()->route('accounting-settings.show', $accountingSetting)
                           ->with('error', 'لا توجد قيمة افتراضية لهذا الإعداد');
        }

        try {
            $accountingSetting->updateValue($accountingSetting->default_value);

            return redirect()->route('accounting-settings.show', $accountingSetting)
                           ->with('success', 'تم استعادة القيمة الافتراضية بنجاح');

        } catch (\Exception $e) {
            return redirect()->route('accounting-settings.show', $accountingSetting)
                           ->with('error', 'حدث خطأ: ' . $e->getMessage());
        }
    }

    /**
     * Load default settings
     */
    public function loadDefaults()
    {
        try {
            AccountingSetting::loadDefaults();

            return redirect()->route('accounting-settings.index')
                           ->with('success', 'تم تحميل الإعدادات الافتراضية بنجاح');

        } catch (\Exception $e) {
            return redirect()->route('accounting-settings.index')
                           ->with('error', 'حدث خطأ أثناء تحميل الإعدادات: ' . $e->getMessage());
        }
    }

    /**
     * Clear all cache
     */
    public function clearCache()
    {
        try {
            AccountingSetting::clearAllCache();

            return redirect()->route('accounting-settings.index')
                           ->with('success', 'تم مسح ذاكرة التخزين المؤقت بنجاح');

        } catch (\Exception $e) {
            return redirect()->route('accounting-settings.index')
                           ->with('error', 'حدث خطأ أثناء مسح الذاكرة: ' . $e->getMessage());
        }
    }

    /**
     * Export settings
     */
    public function export(Request $request)
    {
        $request->validate([
            'format' => 'required|in:json,csv,excel',
            'group' => 'nullable|string',
        ]);

        $query = AccountingSetting::query();

        if ($request->filled('group')) {
            $query->where('group', $request->group);
        }

        $settings = $query->ordered()->get();

        $fileName = 'accounting_settings_' . now()->format('Y-m-d_H-i-s');

        switch ($request->format) {
            case 'json':
                $data = $settings->toArray();
                return response()->json($data)
                    ->header('Content-Disposition', 'attachment; filename="' . $fileName . '.json"');

            case 'csv':
                return (new \App\Exports\AccountingSettingsCsvExport($settings))
                    ->download($fileName . '.csv');

            case 'excel':
                return (new \App\Exports\AccountingSettingsExport($settings))
                    ->download($fileName . '.xlsx');
        }
    }

    /**
     * Import settings
     */
    public function import(Request $request)
    {
        $request->validate([
            'file' => 'required|file|mimes:json,csv,xlsx',
            'action' => 'required|in:merge,replace',
        ]);

        try {
            DB::beginTransaction();

            $imported = 0;
            $errors = [];

            // معالجة الملف حسب النوع
            $extension = $request->file('file')->getClientOriginalExtension();
            
            switch ($extension) {
                case 'json':
                    $data = json_decode(file_get_contents($request->file('file')->path()), true);
                    break;
                    
                case 'csv':
                case 'xlsx':
                    // استخدام مكتبة استيراد مخصصة
                    $data = (new \App\Imports\AccountingSettingsImport)->toArray($request->file('file'));
                    break;
            }

            foreach ($data as $settingData) {
                try {
                    if ($request->action === 'replace') {
                        AccountingSetting::updateOrCreate(
                            ['key' => $settingData['key']],
                            $settingData
                        );
                    } else {
                        AccountingSetting::firstOrCreate(
                            ['key' => $settingData['key']],
                            $settingData
                        );
                    }
                    $imported++;
                } catch (\Exception $e) {
                    $errors[] = "خطأ في {$settingData['key']}: " . $e->getMessage();
                }
            }

            DB::commit();

            $message = "تم استيراد {$imported} إعداد";
            if (!empty($errors)) {
                $message .= ". أخطاء: " . implode(', ', array_slice($errors, 0, 5));
                if (count($errors) > 5) {
                    $message .= " و" . (count($errors) - 5) . " أخطاء أخرى";
                }
            }

            return redirect()->route('accounting-settings.index')
                           ->with('success', $message);

        } catch (\Exception $e) {
            DB::rollback();
            return redirect()->route('accounting-settings.index')
                           ->with('error', 'حدث خطأ أثناء الاستيراد: ' . $e->getMessage());
        }
    }

    /**
     * Remove the specified setting
     */
    public function destroy(AccountingSetting $accountingSetting)
    {
        if (!$accountingSetting->is_editable) {
            return redirect()->route('accounting-settings.index')
                           ->with('error', 'هذا الإعداد غير قابل للحذف');
        }

        try {
            $accountingSetting->delete();

            return redirect()->route('accounting-settings.index')
                           ->with('success', 'تم حذف الإعداد بنجاح');

        } catch (\Exception $e) {
            return redirect()->route('accounting-settings.index')
                           ->with('error', 'حدث خطأ أثناء حذف الإعداد: ' . $e->getMessage());
        }
    }

    /**
     * Validate setting value based on type and rules
     */
    private function validateSettingValue(AccountingSetting $setting, $value)
    {
        $rules = ['value' => $setting->validation_rules_array];
        
        // إضافة قواعد حسب النوع
        switch ($setting->type) {
            case 'integer':
                $rules['value'][] = 'integer';
                break;
            case 'float':
            case 'number':
                $rules['value'][] = 'numeric';
                break;
            case 'boolean':
                $rules['value'][] = 'boolean';
                break;
            case 'date':
                $rules['value'][] = 'date';
                break;
            case 'datetime':
                $rules['value'][] = 'date_format:Y-m-d H:i:s';
                break;
            case 'json':
            case 'array':
                $rules['value'][] = 'json';
                break;
        }

        return Validator::make(['value' => $value], $rules);
    }
}