<?php

namespace App\Http\Controllers;

use App\Models\Account;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Maatwebsite\Excel\Facades\Excel;
use App\Exports\AccountsExport;
use App\Imports\AccountsImport;

class AccountController extends Controller
{
    public function index(Request $request)
    {
        $query = Account::with('parent');

        if ($request->filled('type')) {
            $query->where('type', $request->type);
        }

        if ($request->filled('search')) {
            $query->where(function ($q) use ($request) {
                $q->where('name', 'like', '%' . $request->search . '%')
                  ->orWhere('code', 'like', '%' . $request->search . '%');
            });
        }

        if ($request->filled('parent_id')) {
            $query->where('parent_id', $request->parent_id);
        }

        $accounts = $query->orderBy('code')->paginate(50);
        $accountTypes = Account::select('type')->distinct()->pluck('type');
        $parentAccounts = Account::whereNull('parent_id')->orderBy('name')->get();

        return view('accounting.accounts.index', compact('accounts', 'accountTypes', 'parentAccounts'));
    }

    public function tree()
    {
        $accounts = Account::getAccountsTree();
        return view('accounting.accounts.tree', compact('accounts'));
    }

    public function create()
    {
        $parentAccounts = Account::where('is_active', true)->orderBy('code')->get();
        
        return view('accounting.accounts.create', compact('parentAccounts'));
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'code' => 'required|string|max:20|unique:accounts',
            'name' => 'required|string|max:255',
            'name_en' => 'nullable|string|max:255',
            'type' => 'required|in:asset,liability,equity,revenue,expense',
            'subtype' => 'required|string',
            'parent_id' => 'nullable|exists:accounts,id',
            'debit_credit' => 'required|in:debit,credit',
            'opening_balance' => 'nullable|numeric',
            'is_cash_account' => 'boolean',
            'is_bank_account' => 'boolean',
            'is_control_account' => 'boolean',
            'tax_calculation_method' => 'required|in:inclusive,exclusive',
            'applies_tax' => 'boolean',
            'default_tax_rate' => 'nullable|numeric|min:0|max:100',
            'description' => 'nullable|string'
        ]);

        DB::beginTransaction();
        try {
            $account = Account::create($validated);

            // Set level and path
            if ($account->parent_id) {
                $parent = Account::find($account->parent_id);
                $account->level = $parent->level + 1;
                $account->path = ($parent->path ? $parent->path . ',' : '') . $account->id;
            } else {
                $account->level = 1;
                $account->path = $account->id;
            }

            $account->current_balance = $account->opening_balance ?? 0;
            $account->save();

            DB::commit();

            return redirect()->route('accounts.index')
                ->with('success', 'تم إنشاء الحساب بنجاح');
        } catch (\Exception $e) {
            DB::rollback();
            return back()->withErrors(['error' => 'حدث خطأ أثناء إنشاء الحساب']);
        }
    }

    public function show(Account $account)
    {
        $account->load(['parent', 'children', 'journalEntryLines.journalEntry']);
        
        $recentTransactions = $account->journalEntryLines()
            ->with('journalEntry')
            ->whereHas('journalEntry', function ($query) {
                $query->where('status', 'posted');
            })
            ->orderBy('created_at', 'desc')
            ->limit(10)
            ->get();

        return view('accounting.accounts.show', compact('account', 'recentTransactions'));
    }

    public function edit(Account $account)
    {
        $parentAccounts = Account::where('is_active', true)
            ->where('id', '!=', $account->id)
            ->orderBy('code')
            ->get();
        
        return view('accounting.accounts.edit', compact('account', 'parentAccounts'));
    }

    public function update(Request $request, Account $account)
    {
        $validated = $request->validate([
            'code' => 'required|string|max:20|unique:accounts,code,' . $account->id,
            'name' => 'required|string|max:255',
            'name_en' => 'nullable|string|max:255',
            'type' => 'required|in:asset,liability,equity,revenue,expense',
            'subtype' => 'required|string',
            'parent_id' => 'nullable|exists:accounts,id',
            'debit_credit' => 'required|in:debit,credit',
            'is_active' => 'boolean',
            'is_cash_account' => 'boolean',
            'is_bank_account' => 'boolean',
            'is_control_account' => 'boolean',
            'tax_calculation_method' => 'required|in:inclusive,exclusive',
            'applies_tax' => 'boolean',
            'default_tax_rate' => 'nullable|numeric|min:0|max:100',
            'description' => 'nullable|string'
        ]);

        $account->update($validated);

        return redirect()->route('accounts.index')
            ->with('success', 'تم تحديث الحساب بنجاح');
    }

    public function destroy(Account $account)
    {
        if ($account->journalEntryLines()->exists()) {
            return back()->withErrors(['error' => 'لا يمكن حذف حساب يحتوي على حركات']);
        }

        if ($account->children()->exists()) {
            return back()->withErrors(['error' => 'لا يمكن حذف حساب يحتوي على حسابات فرعية']);
        }

        $account->delete();

        return redirect()->route('accounts.index')
            ->with('success', 'تم حذف الحساب بنجاح');
    }

    public function ledger(Request $request, Account $account)
    {
        $startDate = $request->start_date ?? now()->startOfMonth()->format('Y-m-d');
        $endDate = $request->end_date ?? now()->endOfMonth()->format('Y-m-d');

        $transactions = $account->journalEntryLines()
            ->with(['journalEntry'])
            ->whereHas('journalEntry', function ($query) use ($startDate, $endDate) {
                $query->whereBetween('entry_date', [$startDate, $endDate])
                      ->where('status', 'posted');
            })
            ->orderBy('created_at')
            ->get();

        $openingBalance = $account->getBalanceForPeriod(
            now()->startOfYear()->format('Y-m-d'),
            \Carbon\Carbon::parse($startDate)->subDay()->format('Y-m-d')
        );

        return view('accounting.accounts.ledger', compact(
            'account', 'transactions', 'startDate', 'endDate', 'openingBalance'
        ));
    }

    public function export()
    {
        return Excel::download(new AccountsExport, 'accounts.xlsx');
    }

    public function import(Request $request)
    {
        $request->validate([
            'file' => 'required|mimes:xlsx,xls'
        ]);

        try {
            Excel::import(new AccountsImport, $request->file('file'));
            
            return back()->with('success', 'تم استيراد الحسابات بنجاح');
        } catch (\Exception $e) {
            return back()->withErrors(['error' => 'حدث خطأ أثناء استيراد الحسابات: ' . $e->getMessage()]);
        }
    }
}