<?php

namespace App\Helpers;

class ArabicNumberToWords
{
    private static $ones = [
        0 => '',
        1 => 'واحد',
        2 => 'اثنان',
        3 => 'ثلاثة',
        4 => 'أربعة',
        5 => 'خمسة',
        6 => 'ستة',
        7 => 'سبعة',
        8 => 'ثمانية',
        9 => 'تسعة',
    ];

    private static $tens = [
        10 => 'عشرة',
        11 => 'أحد عشر',
        12 => 'اثنا عشر',
        13 => 'ثلاثة عشر',
        14 => 'أربعة عشر',
        15 => 'خمسة عشر',
        16 => 'ستة عشر',
        17 => 'سبعة عشر',
        18 => 'ثمانية عشر',
        19 => 'تسعة عشر',
        20 => 'عشرون',
        30 => 'ثلاثون',
        40 => 'أربعون',
        50 => 'خمسون',
        60 => 'ستون',
        70 => 'سبعون',
        80 => 'ثمانون',
        90 => 'تسعون',
    ];

    private static $hundreds = [
        100 => 'مائة',
        200 => 'مائتان',
        300 => 'ثلاثمائة',
        400 => 'أربعمائة',
        500 => 'خمسمائة',
        600 => 'ستمائة',
        700 => 'سبعمائة',
        800 => 'ثمانمائة',
        900 => 'تسعمائة',
    ];

    private static $scales = [
        1000000000000 => ['تريليون', 'تريليونان', 'تريليونات'],
        1000000000 => ['مليار', 'ملياران', 'مليارات'],
        1000000 => ['مليون', 'مليونان', 'ملايين'],
        1000 => ['ألف', 'ألفان', 'آلاف'],
    ];

    /**
     * Convert number to Arabic words
     */
    public static function convert($number, $currency = 'ريال سعودي', $fraction_currency = 'هللة')
    {
        if ($number < 0) {
            return 'ناقص ' . self::convert(abs($number), $currency, $fraction_currency);
        }

        if ($number == 0) {
            return 'صفر ' . $currency;
        }

        $integerPart = intval($number);
        $fractionPart = intval(round(($number - $integerPart) * 100));

        $result = self::convertInteger($integerPart);
        
        if ($integerPart > 0) {
            $result .= ' ' . self::getCurrencyText($integerPart, $currency);
        }

        if ($fractionPart > 0) {
            $result .= ' و ' . self::convertInteger($fractionPart) . ' ' . self::getCurrencyText($fractionPart, $fraction_currency);
        }

        return $result . ' فقط لا غير';
    }

    /**
     * Convert integer part to words
     */
    private static function convertInteger($number)
    {
        if ($number == 0) {
            return '';
        }

        if ($number < 10) {
            return self::$ones[$number];
        }

        if ($number < 20) {
            return self::$tens[$number];
        }

        if ($number < 100) {
            $tens = intval($number / 10) * 10;
            $ones = $number % 10;
            
            if ($ones == 0) {
                return self::$tens[$tens];
            }
            
            return self::$ones[$ones] . ' و ' . self::$tens[$tens];
        }

        if ($number < 1000) {
            $hundreds = intval($number / 100) * 100;
            $remainder = $number % 100;
            
            $result = self::$hundreds[$hundreds];
            
            if ($remainder > 0) {
                $result .= ' ' . self::convertInteger($remainder);
            }
            
            return $result;
        }

        // Handle thousands, millions, billions, trillions
        foreach (self::$scales as $scale => $scaleNames) {
            if ($number >= $scale) {
                $quotient = intval($number / $scale);
                $remainder = $number % $scale;
                
                $result = '';
                
                if ($quotient == 1) {
                    $result = $scaleNames[0]; // singular
                } elseif ($quotient == 2) {
                    $result = $scaleNames[1]; // dual
                } elseif ($quotient < 11) {
                    $result = self::convertInteger($quotient) . ' ' . $scaleNames[2]; // plural for 3-10
                } else {
                    $result = self::convertInteger($quotient) . ' ' . $scaleNames[0]; // singular for 11+
                }
                
                if ($remainder > 0) {
                    $result .= ' ' . self::convertInteger($remainder);
                }
                
                return $result;
            }
        }

        return '';
    }

    /**
     * Get appropriate currency text based on number
     */
    private static function getCurrencyText($number, $currency)
    {
        if ($currency == 'ريال سعودي') {
            if ($number == 1) {
                return 'ريال سعودي';
            } elseif ($number == 2) {
                return 'ريالان سعوديان';
            } elseif ($number <= 10) {
                return 'ريالات سعودية';
            } else {
                return 'ريالاً سعودياً';
            }
        } elseif ($currency == 'هللة') {
            if ($number == 1) {
                return 'هللة';
            } elseif ($number == 2) {
                return 'هللتان';
            } elseif ($number <= 10) {
                return 'هللات';
            } else {
                return 'هللة';
            }
        }
        
        return $currency;
    }

    /**
     * Format number with currency for display
     */
    public static function formatCurrency($number, $currency = 'ر.س')
    {
        return number_format($number, 2) . ' ' . $currency;
    }
}