<?php

namespace App\Helpers;

use App\Models\Account;
use App\Models\JournalEntry;
use App\Models\CashTransaction;
use App\Models\CostCenter;
use Carbon\Carbon;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\DB;

class AccountingHelper
{
    /**
     * تنسيق المبلغ المالي
     */
    public static function formatMoney($amount, $showCurrency = true, $decimals = null)
    {
        if ($decimals === null) {
            $decimals = config('accounting.general.decimal_places', 2);
        }
        
        $thousandSeparator = config('accounting.general.thousand_separator', ',');
        $decimalSeparator = config('accounting.general.decimal_separator', '.');
        $currencySymbol = config('accounting.general.currency_symbol', 'ريال');
        
        $formatted = number_format($amount, $decimals, $decimalSeparator, $thousandSeparator);
        
        if ($showCurrency) {
            $formatted .= ' ' . $currencySymbol;
        }
        
        return $formatted;
    }

    /**
     * تنسيق رقم الحساب
     */
    public static function formatAccountCode($code)
    {
        // إضافة أصفار في المقدمة إذا لزم الأمر
        $codeLength = config('accounting.chart_of_accounts.code_length', 4);
        return str_pad($code, $codeLength, '0', STR_PAD_LEFT);
    }

    /**
     * توليد رقم مرجع تلقائي
     */
    public static function generateReference($type, $year = null)
    {
        $year = $year ?? now()->year;
        
        $prefixes = [
            'journal_entry' => config('accounting.journal_entries.reference_prefix', 'JE'),
            'cash_receipt' => config('accounting.cash_transactions.receipt_prefix', 'CR'),
            'cash_payment' => config('accounting.cash_transactions.payment_prefix', 'CP'),
        ];
        
        $prefix = $prefixes[$type] ?? 'REF';
        
        // الحصول على آخر رقم
        $lastNumber = Cache::remember("last_reference_{$type}_{$year}", 3600, function () use ($type, $year) {
            switch ($type) {
                case 'journal_entry':
                    return JournalEntry::whereYear('created_at', $year)
                        ->where('reference', 'like', "{$prefixes['journal_entry']}{$year}%")
                        ->count();
                        
                case 'cash_receipt':
                case 'cash_payment':
                    return CashTransaction::whereYear('created_at', $year)
                        ->where('reference', 'like', "{$prefix}{$year}%")
                        ->count();
                        
                default:
                    return 0;
            }
        });
        
        $nextNumber = $lastNumber + 1;
        
        return $prefix . $year . str_pad($nextNumber, 4, '0', STR_PAD_LEFT);
    }

    /**
     * التحقق من صحة رقم الحساب
     */
    public static function validateAccountCode($code, $accountType = null)
    {
        $errors = [];
        
        // التحقق من الطول
        $maxLength = config('accounting.constants.max_account_code_length', 20);
        if (strlen($code) > $maxLength) {
            $errors[] = "رقم الحساب يجب ألا يتجاوز {$maxLength} رقم";
        }
        
        // التحقق من التكرار
        if (Account::where('code', $code)->exists()) {
            $errors[] = 'رقم الحساب موجود مسبقاً';
        }
        
        // التحقق من تطابق النوع مع الرقم
        if ($accountType) {
            $accountTypes = config('accounting.chart_of_accounts.account_types');
            $expectedPrefix = $accountTypes[$accountType]['code_prefix'] ?? null;
            
            if ($expectedPrefix && !str_starts_with($code, $expectedPrefix)) {
                $typeName = $accountTypes[$accountType]['name_ar'] ?? $accountType;
                $errors[] = "رقم الحساب يجب أن يبدأ بـ {$expectedPrefix} لنوع {$typeName}";
            }
        }
        
        return $errors;
    }

    /**
     * حساب الرصيد الطبيعي للحساب
     */
    public static function calculateNormalBalance($accountType, $debit, $credit)
    {
        $accountTypes = config('accounting.chart_of_accounts.account_types');
        $normalBalance = $accountTypes[$accountType]['normal_balance'] ?? 'debit';
        
        if ($normalBalance === 'debit') {
            return $debit - $credit;
        } else {
            return $credit - $debit;
        }
    }

    /**
     * التحقق من نوع الحساب
     */
    public static function getAccountTypeInfo($type)
    {
        $accountTypes = config('accounting.chart_of_accounts.account_types');
        return $accountTypes[$type] ?? null;
    }

    /**
     * حساب النسب المالية
     */
    public static function calculateFinancialRatios($balanceSheetData)
    {
        $ratios = [];
        
        // نسبة السيولة الجارية
        if ($balanceSheetData['total_current_liabilities'] > 0) {
            $ratios['current_ratio'] = $balanceSheetData['total_current_assets'] / $balanceSheetData['total_current_liabilities'];
        } else {
            $ratios['current_ratio'] = 0;
        }
        
        // نسبة حقوق الملكية
        if ($balanceSheetData['total_assets'] > 0) {
            $ratios['equity_ratio'] = ($balanceSheetData['total_equity'] / $balanceSheetData['total_assets']) * 100;
        } else {
            $ratios['equity_ratio'] = 0;
        }
        
        // نسبة الدين إلى حقوق الملكية
        if ($balanceSheetData['total_equity'] > 0) {
            $ratios['debt_to_equity'] = $balanceSheetData['total_liabilities'] / $balanceSheetData['total_equity'];
        } else {
            $ratios['debt_to_equity'] = 0;
        }
        
        return $ratios;
    }

    /**
     * حساب هوامش الربح
     */
    public static function calculateProfitMargins($incomeStatementData)
    {
        $margins = [];
        $totalRevenue = $incomeStatementData['total_revenue'];
        
        if ($totalRevenue > 0) {
            // هامش الربح الإجمالي
            $margins['gross_profit_margin'] = (($incomeStatementData['gross_profit'] ?? 0) / $totalRevenue) * 100;
            
            // هامش الربح الصافي
            $margins['net_profit_margin'] = ($incomeStatementData['net_income'] / $totalRevenue) * 100;
            
            // نسبة المصروفات
            $margins['expense_ratio'] = ($incomeStatementData['total_expenses'] / $totalRevenue) * 100;
        } else {
            $margins['gross_profit_margin'] = 0;
            $margins['net_profit_margin'] = 0;
            $margins['expense_ratio'] = 0;
        }
        
        return $margins;
    }

    /**
     * تحويل التاريخ الميلادي إلى هجري
     */
    public static function toHijri($date)
    {
        // يمكن تطوير هذا لاحقاً باستخدام مكتبة التقويم الهجري
        return $date;
    }

    /**
     * تحويل التاريخ الهجري إلى ميلادي
     */
    public static function fromHijri($hijriDate)
    {
        // يمكن تطوير هذا لاحقاً باستخدام مكتبة التقويم الهجري
        return $hijriDate;
    }

    /**
     * حساب بداية ونهاية السنة المالية
     */
    public static function getFiscalYearDates($year = null)
    {
        $year = $year ?? now()->year;
        $fiscalStart = config('accounting.general.fiscal_year_start', '01-01');
        
        $startDate = Carbon::createFromFormat('Y-m-d', $year . '-' . $fiscalStart);
        $endDate = $startDate->copy()->addYear()->subDay();
        
        return [
            'start' => $startDate,
            'end' => $endDate
        ];
    }

    /**
     * التحقق من صحة التوازن المحاسبي
     */
    public static function validateAccountingEquation($assets, $liabilities, $equity)
    {
        $tolerance = config('accounting.journal_entries.balance_tolerance', 0.01);
        return abs($assets - ($liabilities + $equity)) <= $tolerance;
    }

    /**
     * تحويل النص إلى رقم
     */
    public static function parseAmount($amount)
    {
        // إزالة الرموز والمسافات
        $cleaned = preg_replace('/[^\d.,\-]/', '', $amount);
        
        // التعامل مع الفاصلة العشرية
        $thousandSeparator = config('accounting.general.thousand_separator', ',');
        $decimalSeparator = config('accounting.general.decimal_separator', '.');
        
        if ($thousandSeparator === ',' && $decimalSeparator === '.') {
            $cleaned = str_replace(',', '', $cleaned);
        } elseif ($thousandSeparator === '.' && $decimalSeparator === ',') {
            $cleaned = str_replace('.', '', $cleaned);
            $cleaned = str_replace(',', '.', $cleaned);
        }
        
        return floatval($cleaned);
    }

    /**
     * توليد ملخص الحساب
     */
    public static function generateAccountSummary($accountId, $startDate = null, $endDate = null)
    {
        $account = Account::find($accountId);
        if (!$account) {
            return null;
        }
        
        $startDate = $startDate ? Carbon::parse($startDate) : now()->startOfMonth();
        $endDate = $endDate ? Carbon::parse($endDate) : now()->endOfMonth();
        
        // حساب الأرصدة والحركات
        $openingBalance = app('App\Services\AccountingService')->getAccountBalance($accountId, $startDate->copy()->subDay());
        $closingBalance = app('App\Services\AccountingService')->getAccountBalance($accountId, $endDate);
        
        $movements = DB::table('journal_entry_lines')
            ->join('journal_entries', 'journal_entry_lines.journal_entry_id', '=', 'journal_entries.id')
            ->where('journal_entry_lines.account_id', $accountId)
            ->where('journal_entries.is_approved', true)
            ->whereBetween('journal_entries.date', [$startDate, $endDate])
            ->selectRaw('
                SUM(debit_amount) as total_debits,
                SUM(credit_amount) as total_credits,
                COUNT(*) as transaction_count
            ')
            ->first();
        
        return [
            'account' => $account,
            'period' => [
                'start' => $startDate,
                'end' => $endDate
            ],
            'opening_balance' => $openingBalance,
            'closing_balance' => $closingBalance,
            'total_debits' => $movements->total_debits ?? 0,
            'total_credits' => $movements->total_credits ?? 0,
            'transaction_count' => $movements->transaction_count ?? 0,
            'net_change' => ($movements->total_debits ?? 0) - ($movements->total_credits ?? 0),
        ];
    }

    /**
     * حساب معدل دوران الحساب
     */
    public static function calculateAccountTurnover($accountId, $period = 'annual')
    {
        $account = Account::find($accountId);
        if (!$account) {
            return 0;
        }
        
        $endDate = now();
        $startDate = match($period) {
            'monthly' => $endDate->copy()->subMonth(),
            'quarterly' => $endDate->copy()->subQuarter(),
            'annual' => $endDate->copy()->subYear(),
            default => $endDate->copy()->subYear()
        };
        
        $movements = DB::table('journal_entry_lines')
            ->join('journal_entries', 'journal_entry_lines.journal_entry_id', '=', 'journal_entries.id')
            ->where('journal_entry_lines.account_id', $accountId)
            ->where('journal_entries.is_approved', true)
            ->whereBetween('journal_entries.date', [$startDate, $endDate])
            ->sum(DB::raw('debit_amount + credit_amount'));
        
        $averageBalance = ($account->opening_balance + $account->current_balance) / 2;
        
        if ($averageBalance == 0) {
            return 0;
        }
        
        return $movements / abs($averageBalance);
    }

    /**
     * تحديد لون الحساب حسب النوع
     */
    public static function getAccountTypeColor($type)
    {
        $colors = [
            'asset' => '#28a745',      // أخضر
            'liability' => '#dc3545',  // أحمر
            'equity' => '#6f42c1',     // بنفسجي
            'revenue' => '#007bff',    // أزرق
            'expense' => '#fd7e14',    // برتقالي
        ];
        
        return $colors[$type] ?? '#6c757d'; // رمادي كافتراضي
    }

    /**
     * تحديد رمز الحساب حسب النوع
     */
    public static function getAccountTypeIcon($type)
    {
        $icons = [
            'asset' => 'fas fa-coins',
            'liability' => 'fas fa-credit-card',
            'equity' => 'fas fa-chart-pie',
            'revenue' => 'fas fa-arrow-up',
            'expense' => 'fas fa-arrow-down',
        ];
        
        return $icons[$type] ?? 'fas fa-file-invoice-dollar';
    }

    /**
     * التحقق من إمكانية تعديل/حذف الحساب
     */
    public static function canModifyAccount($accountId)
    {
        $account = Account::find($accountId);
        if (!$account) {
            return false;
        }
        
        // التحقق من وجود حركات
        $hasTransactions = DB::table('journal_entry_lines')
            ->where('account_id', $accountId)
            ->exists();
        
        // التحقق من وجود حسابات فرعية
        $hasChildren = Account::where('parent_id', $accountId)->exists();
        
        // التحقق من ارتباطه بمعاملات نقدية
        $hasCashTransactions = DB::table('cash_transactions')
            ->where(function($query) use ($accountId) {
                $query->where('account_id', $accountId)
                      ->orWhere('contra_account_id', $accountId);
            })
            ->exists();
        
        return [
            'can_edit' => !$hasTransactions || $account->allow_manual_entries,
            'can_delete' => !$hasTransactions && !$hasChildren && !$hasCashTransactions,
            'has_transactions' => $hasTransactions,
            'has_children' => $hasChildren,
            'has_cash_transactions' => $hasCashTransactions,
        ];
    }

    /**
     * حساب العمر (للديون والذمم)
     */
    public static function calculateAge($date)
    {
        return Carbon::parse($date)->diffInDays(now());
    }

    /**
     * تصنيف الديون حسب العمر
     */
    public static function classifyDebtAge($days)
    {
        if ($days <= 30) {
            return ['class' => 'current', 'label' => 'جاري (0-30 يوم)', 'color' => 'success'];
        } elseif ($days <= 60) {
            return ['class' => 'overdue_30', 'label' => 'متأخر 30-60 يوم', 'color' => 'warning'];
        } elseif ($days <= 90) {
            return ['class' => 'overdue_60', 'label' => 'متأخر 60-90 يوم', 'color' => 'danger'];
        } else {
            return ['class' => 'overdue_90', 'label' => 'متأخر أكثر من 90 يوم', 'color' => 'dark'];
        }
    }
}