<?php

namespace App\Exports;

use App\Models\Account;
use Maatwebsite\Excel\Concerns\FromCollection;
use Maatwebsite\Excel\Concerns\WithHeadings;
use Maatwebsite\Excel\Concerns\WithMapping;
use Maatwebsite\Excel\Concerns\WithStyles;
use Maatwebsite\Excel\Concerns\WithTitle;
use PhpOffice\PhpSpreadsheet\Worksheet\Worksheet;

class TrialBalanceExport implements FromCollection, WithHeadings, WithMapping, WithStyles, WithTitle
{
    protected $endDate;
    protected $startDate;

    public function __construct($endDate, $startDate = null)
    {
        $this->endDate = $endDate;
        $this->startDate = $startDate ?? now()->startOfYear()->format('Y-m-d');
    }

    public function collection()
    {
        $accounts = Account::with(['journalEntryLines' => function ($query) {
            $query->whereHas('journalEntry', function ($q) {
                $q->whereBetween('entry_date', [$this->startDate, $this->endDate])
                  ->where('status', 'posted');
            });
        }])->where('is_active', true)->orderBy('code')->get();

        $trialBalanceData = collect();

        foreach ($accounts as $account) {
            $debitTotal = $account->journalEntryLines->sum('debit_amount');
            $creditTotal = $account->journalEntryLines->sum('credit_amount');
            
            if ($debitTotal > 0 || $creditTotal > 0 || $account->opening_balance != 0) {
                $balance = $account->opening_balance;
                
                if ($account->debit_credit === 'debit') {
                    $balance += $debitTotal - $creditTotal;
                    $debitBalance = $balance > 0 ? $balance : 0;
                    $creditBalance = $balance < 0 ? abs($balance) : 0;
                } else {
                    $balance += $creditTotal - $debitTotal;
                    $creditBalance = $balance > 0 ? $balance : 0;
                    $debitBalance = $balance < 0 ? abs($balance) : 0;
                }

                $trialBalanceData->push((object) [
                    'account' => $account,
                    'debit_balance' => $debitBalance,
                    'credit_balance' => $creditBalance,
                    'debit_movements' => $debitTotal,
                    'credit_movements' => $creditTotal
                ]);
            }
        }

        return $trialBalanceData;
    }

    public function headings(): array
    {
        return [
            'رمز الحساب',
            'اسم الحساب',
            'إجمالي المدين',
            'إجمالي الدائن',
            'الرصيد المدين',
            'الرصيد الدائن'
        ];
    }

    public function map($row): array
    {
        return [
            $row->account->code,
            $row->account->name,
            number_format($row->debit_movements, 2),
            number_format($row->credit_movements, 2),
            number_format($row->debit_balance, 2),
            number_format($row->credit_balance, 2)
        ];
    }

    public function styles(Worksheet $sheet)
    {
        return [
            1 => ['font' => ['bold' => true]],
            'A:F' => ['alignment' => ['horizontal' => 'center']],
        ];
    }

    public function title(): string
    {
        return 'ميزان المراجعة - ' . $this->endDate;
    }
}