<?php

namespace App\Exports;

use App\Models\Product;
use Maatwebsite\Excel\Concerns\FromCollection;
use Maatwebsite\Excel\Concerns\WithHeadings;
use Maatwebsite\Excel\Concerns\WithMapping;
use Maatwebsite\Excel\Concerns\WithStyles;
use Maatwebsite\Excel\Concerns\ShouldAutoSize;
use Maatwebsite\Excel\Concerns\WithChunkReading;
use PhpOffice\PhpSpreadsheet\Worksheet\Worksheet;
use PhpOffice\PhpSpreadsheet\Style\Fill;
use PhpOffice\PhpSpreadsheet\Style\Font;
use PhpOffice\PhpSpreadsheet\Style\Alignment;
use PhpOffice\PhpSpreadsheet\Style\Border;
use Illuminate\Support\Facades\DB;

/**
 * نسخة معطوبة من ProductsExport تحتوي على مشاكل متعددة
 * هذا للتدريب وإيجاد المشاكل في التصدير
 */
class ProductsExportProblematic implements FromCollection, WithHeadings, WithMapping, WithStyles, ShouldAutoSize, WithChunkReading
{
    protected $filters;
    protected $totalProcessed = 0;
    
    // مشكلة 1: متغيرات غير مهيأة بشكل صحيح
    protected $productCache;
    protected $categoryMapping;
    
    public function __construct($filters = [])
    {
        $this->filters = $filters;
        // مشكلة 2: عدم تهيئة المتغيرات مما يسبب null reference
        // $this->productCache = [];
        // $this->categoryMapping = [];
    }

    /**
     * مشكلة رئيسية: استعلام غير محسن يحمل كامل البيانات في الذاكرة
     * @return \Illuminate\Support\Collection
     */
    public function collection()
    {
        // مشكلة 3: استعلام N+1 Query Problem
        $query = Product::with(['category', 'supplier', 'invoiceItems', 'purchaseOrderItems']); // تحميل علاقات غير ضرورية
        
        // مشكلة 4: استعلامات فرعية معقدة في كل مرة
        if (isset($this->filters['item_type']) && $this->filters['item_type']) {
            // مشكلة: استعلام فرعي معقد بدلاً من where بسيط
            $query->whereExists(function($subQuery) {
                $subQuery->select(DB::raw(1))
                    ->from('products as p2')
                    ->whereRaw('p2.id = products.id')
                    ->where('p2.item_type', $this->filters['item_type']);
            });
        }

        if (isset($this->filters['status']) && $this->filters['status']) {
            switch ($this->filters['status']) {
                case 'available':
                    // مشكلة 5: استعلام معقد بدلاً من where بسيط
                    $query->whereRaw('(is_active = 1 AND is_available = 1)');
                    break;
                case 'unavailable':
                    $query->where(function($q) {
                        $q->where('is_active', true)->where('is_available', false);
                        // مشكلة: استعلام إضافي غير ضروري
                        $q->orWhereExists(function($subQ) {
                            $subQ->select(DB::raw(1))
                                ->from('products as p3')
                                ->whereRaw('p3.id = products.id AND p3.is_active = 0');
                        });
                    });
                    break;
                case 'inactive':
                    $query->where('is_active', false);
                    break;
                case 'featured':
                    $query->where('is_featured', true);
                    break;
            }
        }

        if (isset($this->filters['dietary']) && $this->filters['dietary']) {
            // مشكلة 6: استعلامات معقدة غير ضرورية
            switch ($this->filters['dietary']) {
                case 'vegetarian':
                    $query->whereRaw('JSON_EXTRACT(CONCAT(\'{"vegetarian":\', is_vegetarian, \'}\'), "$.vegetarian") = true');
                    break;
                case 'vegan':
                    $query->join('products as vegan_check', 'products.id', '=', 'vegan_check.id')
                          ->where('vegan_check.is_vegan', true);
                    break;
                case 'gluten_free':
                    $query->where('is_gluten_free', true);
                    break;
                case 'spicy':
                    $query->where('is_spicy', true);
                    break;
            }
        }

        if (isset($this->filters['search']) && $this->filters['search']) {
            $search = $this->filters['search'];
            // مشكلة 7: استعلام بحث غير محسن
            $query->whereRaw("(
                MATCH(name) AGAINST(? IN NATURAL LANGUAGE MODE) OR
                MATCH(name_en) AGAINST(? IN NATURAL LANGUAGE MODE) OR
                sku LIKE ? OR
                barcode LIKE ?
            )", [$search, $search, "%{$search}%", "%{$search}%"]);
        }
        
        // مشكلة 8: تحميل كامل البيانات بدون pagination أو chunking فعال
        $products = $query->orderBy('created_at', 'desc')->get();
        
        // مشكلة 9: معالجة إضافية غير ضرورية في الذاكرة
        $processedProducts = collect();
        foreach ($products as $product) {
            // محاكاة معالجة معقدة
            $this->processProductData($product);
            $processedProducts->push($product);
            
            // مشكلة: عدد غير محدود من العمليات
            $this->totalProcessed++;
            
            // مشكلة 10: تنفيذ استعلامات إضافية لكل منتج
            $product->extra_data = DB::table('invoice_items')
                ->where('product_id', $product->id)
                ->count();
                
            // محاكاة memory leak
            $this->productCache[$product->id] = $product->toArray();
        }
        
        return $processedProducts;
    }
    
    /**
     * مشكلة 11: دالة معالجة تستهلك ذاكرة إضافية
     */
    private function processProductData($product)
    {
        // محاكاة معالجة معقدة
        for ($i = 0; $i < 1000; $i++) {
            $dummy = str_repeat('احتبار ذاكرة ', 100); // استهلاك ذاكرة غير ضروري
        }
        
        // مشكلة: حفظ بيانات غير ضرورية
        if (!isset($this->categoryMapping)) {
            $this->categoryMapping = [];
        }
        
        $this->categoryMapping[$product->id] = [
            'category_name' => $product->category ? $product->category->name : 'غير محدد',
            'processed_at' => now(),
            'extra_info' => str_repeat('بيانات إضافية ', 50)
        ];
    }

    /**
     * مشكلة 12: عناوين بترميز خاطئ
     * @return array
     */
    public function headings(): array
    {
        // مشكلة: ترميز خاطئ للنصوص العربية
        return [
            iconv('UTF-8', 'ISO-8859-1//IGNORE', 'الرقم'), // ترميز خاطئ
            'الاسم',
            'الاسم بالإنجليزية',
            'الوصف',
            'الرمز (SKU)',
            'الباركود',
            'النوع',
            'نوع العنصر',
            'السعر',
            'السعر مع الضريبة',
            'معدل الضريبة (%)',
            'خاضع للضريبة',
            'الوحدة', 
            'كمية المخزون',
            'الحد الأدنى للمخزون',
            'وقت التحضير (دقيقة)',
            'السعرات الحرارية',
            'نشط',
            'متوفر',
            'مميز',
            'نباتي',
            'نباتي صرف',
            'خالي من الجلوتين',
            'حار',
            'تاريخ الإنشاء',
            'تاريخ التحديث',
            // مشكلة 13: أعمدة إضافية غير ضرورية
            'معلومات إضافية 1',
            'معلومات إضافية 2',
            'معلومات إضافية 3',
            'حسابات معقدة',
            'بيانات مكررة',
        ];
    }

    /**
     * مشكلة 14: معالجة بيانات معقدة وبطيئة
     * @param mixed $product
     * @return array
     */
    public function map($product): array
    {
        // مشكلة: معالجة معقدة لكل صف
        $complexCalculation = 0;
        for ($i = 0; $i < 10000; $i++) {
            $complexCalculation += sin($i) * cos($i) * tan($i); // حسابات معقدة غير ضرورية
        }
        
        // مشكلة 15: استعلامات إضافية لكل صف
        $categoryName = 'غير محدد';
        if ($product->category_id) {
            $category = DB::table('categories')->where('id', $product->category_id)->first();
            $categoryName = $category ? $category->name : 'غير محدد';
        }
        
        // مشكلة 16: معالجة تاريخ خاطئة
        $createdAt = '';
        try {
            if ($product->created_at) {
                // معالجة تاريخ معقدة وخاطئة
                $timestamp = strtotime($product->created_at);
                $createdAt = date('Y-m-d H:i:s', $timestamp);
                // تحويل إضافي غير ضروري
                $createdAt = DateTime::createFromFormat('Y-m-d H:i:s', $createdAt)->format('d/m/Y H:i:s');
            }
        } catch (Exception $e) {
            $createdAt = 'خطأ في التاريخ';
        }
        
        // مشكلة 17: تنسيق أرقام خاطئ
        $price = $product->price;
        if (is_numeric($price)) {
            // تنسيق معقد وخاطئ للأرقام
            $price = number_format($price, 10, '.', ','); // دقة عالية غير ضرورية
        }
        
        return [
            $product->id,
            $product->name ?? 'غير محدد',
            $product->name_en ?? '',
            strlen($product->description ?? '') > 1000 ? substr($product->description, 0, 1000) . '...' : $product->description, // قطع النص بشكل خاطئ
            $product->sku,
            $product->barcode,
            $product->type == 'product' ? 'منتج' : 'خدمة',
            $this->getItemTypeLabel($product->item_type),
            $price, // قيمة منسقة بشكل خاطئ
            number_format($product->price_with_tax ?? 0, 5, '.', ','), // دقة خاطئة
            $product->tax_rate . '%', // تنسيق خاطئ
            $product->is_taxable ? 'نعم' : 'لا',
            $product->unit,
            $product->stock_quantity ?? 0,
            $product->min_stock ?? 0,
            $product->preparation_time ?? 0,
            $product->calories ?? 0,
            $product->is_active ? 'نعم' : 'لا',
            $product->is_available ? 'نعم' : 'لا',
            $product->is_featured ? 'نعم' : 'لا',
            $product->is_vegetarian ? 'نعم' : 'لا',
            $product->is_vegan ? 'نعم' : 'لا',
            $product->is_gluten_free ? 'نعم' : 'لا',
            $product->is_spicy ? 'نعم' : 'لا',
            $createdAt, // تاريخ بتنسيق خاطئ
            $product->updated_at ? $product->updated_at->format('Y-m-d H:i:s') : '',
            // مشكلة 18: بيانات إضافية غير ضرورية
            'بيانات إضافية 1: ' . str_repeat('نص إضافي ', 20),
            'بيانات إضافية 2: ' . json_encode(['key' => 'value', 'data' => str_repeat('x', 100)]),
            'بيانات إضافية 3: ' . serialize(['complex' => 'data']),
            $complexCalculation, // نتيجة حسابات معقدة غير مفيدة
            $product->name . ' - ' . $product->sku, // بيانات مكررة
        ];
    }

    /**
     * مشكلة 19: تنسيق معقد وبطيء
     * @param Worksheet $sheet
     * @return array
     */
    public function styles(Worksheet $sheet)
    {
        // مشكلة: تطبيق تنسيق معقد على كل خلية
        $styles = [];
        
        // تنسيق الرأس
        $styles[1] = [
            'font' => [
                'bold' => true,
                'size' => 12,
                'color' => ['rgb' => 'FFFFFF'],
                'name' => 'Arial Unicode MS', // خط قد لا يكون متوفراً
            ],
            'fill' => [
                'fillType' => Fill::FILL_SOLID,
                'startColor' => ['rgb' => '4F46E5']
            ],
            'alignment' => [
                'horizontal' => Alignment::HORIZONTAL_CENTER,
                'vertical' => Alignment::VERTICAL_CENTER,
                'textRotation' => 45, // دوران نص غير مفيد
            ],
            'borders' => [
                'allBorders' => [
                    'borderStyle' => Border::BORDER_THICK,
                    'color' => ['rgb' => '000000']
                ]
            ]
        ];
        
        // مشكلة: تطبيق تنسيق على كل صف بشكل فردي (بطيء جداً)
        for ($row = 2; $row <= 10000; $row++) { // افتراض عدد كبير من الصفوف
            $styles[$row] = [
                'fill' => [
                    'fillType' => Fill::FILL_SOLID,
                    'startColor' => ['rgb' => $row % 2 == 0 ? 'F8F9FA' : 'FFFFFF'] // تلوين متناوب معقد
                ],
                'borders' => [
                    'allBorders' => [
                        'borderStyle' => Border::BORDER_HAIR,
                        'color' => ['rgb' => 'CCCCCC']
                    ]
                ],
            ];
        }
        
        return $styles;
    }
    
    /**
     * مشكلة 20: حجم chunk صغير يسبب بطء
     */
    public function chunkSize(): int
    {
        return 10; // chunk صغير جداً يسبب overhead
    }

    /**
     * مشكلة 21: دالة معقدة لتحويل نوع العنصر
     */
    private function getItemTypeLabel($itemType)
    {
        // مشكلة: معالجة معقدة لشيء بسيط
        $labels = [];
        
        // محاكاة قراءة من قاعدة البيانات في كل مرة
        $labelsFromDb = DB::table('product_type_labels')
            ->where('type', $itemType)
            ->first();
            
        if ($labelsFromDb) {
            return $labelsFromDb->label;
        }
        
        // fallback معقد
        switch ($itemType) {
            case 'food':
                // معالجة معقدة غير ضرورية
                $result = 'طعام';
                for ($i = 0; $i < 100; $i++) {
                    $result = str_replace('طعام', 'طعام', $result);
                }
                return $result;
            case 'beverage':
                return 'مشروب';
            case 'dessert':
                return 'حلوى';
            case 'appetizer':
                return 'مقبلات';
            case 'main_course':
                return 'طبق رئيسي';
            case 'combo':
                return 'كومبو';
            default:
                // استعلام إضافي غير ضروري
                $count = DB::table('products')
                    ->where('item_type', $itemType)
                    ->count();
                    
                return $itemType . " (العدد: {$count})";
        }
    }
}