<?php

namespace App\Exports;

use App\Models\Product;
use Maatwebsite\Excel\Concerns\FromQuery;
use Maatwebsite\Excel\Concerns\WithHeadings;
use Maatwebsite\Excel\Concerns\WithMapping;
use Maatwebsite\Excel\Concerns\WithStyles;
use Maatwebsite\Excel\Concerns\WithColumnWidths;
use Maatwebsite\Excel\Concerns\WithTitle;
use Maatwebsite\Excel\Concerns\ShouldAutoSize;
use PhpOffice\PhpSpreadsheet\Worksheet\Worksheet;
use PhpOffice\PhpSpreadsheet\Style\Alignment;
use PhpOffice\PhpSpreadsheet\Style\Font;
use PhpOffice\PhpSpreadsheet\Style\Fill;

class ProductsExport implements FromQuery, WithHeadings, WithMapping, WithStyles, WithColumnWidths, WithTitle, ShouldAutoSize
{
    protected $filters;

    public function __construct($filters = [])
    {
        $this->filters = $filters;
    }

    /**
     * Build the query for products based on filters
     */
    public function query()
    {
        $query = Product::query()->with(['category']);

        // Apply filters similar to ProductController@index
        if (!empty($this->filters['search'])) {
            $search = $this->filters['search'];
            $query->where(function ($q) use ($search) {
                $q->where('name', 'like', "%{$search}%")
                  ->orWhere('name_en', 'like', "%{$search}%")
                  ->orWhere('sku', 'like', "%{$search}%")
                  ->orWhere('barcode', 'like', "%{$search}%");
            });
        }

        if (!empty($this->filters['item_type'])) {
            $query->where('item_type', $this->filters['item_type']);
        }

        if (!empty($this->filters['status'])) {
            switch ($this->filters['status']) {
                case 'available':
                    $query->where('is_active', true)->where('is_available', true);
                    break;
                case 'unavailable':
                    $query->where('is_active', true)->where('is_available', false);
                    break;
                case 'inactive':
                    $query->where('is_active', false);
                    break;
            }
        }

        if (!empty($this->filters['category_id'])) {
            $query->where('category_id', $this->filters['category_id']);
        }

        if (!empty($this->filters['is_featured'])) {
            $query->where('is_featured', $this->filters['is_featured'] === 'true');
        }

        return $query->orderBy('name');
    }

    /**
     * Define the headings for the Excel file
     */
    public function headings(): array
    {
        return [
            'الرقم التسلسلي',
            'الاسم العربي',
            'الاسم الإنجليزي',
            'الرمز (SKU)',
            'الباركود',
            'نوع العنصر',
            'الفئة',
            'السعر',
            'تكلفة الوحدة',
            'الكمية المتوفرة',
            'الحد الأدنى للكمية',
            'الوزن (كجم)',
            'الوصف',
            'الحالة',
            'متاح للبيع',
            'منتج مميز',
            'صورة المنتج',
            'تاريخ الإنشاء',
            'تاريخ التحديث'
        ];
    }

    /**
     * Map each product to a row in the Excel file
     */
    public function map($product): array
    {
        return [
            $product->id,
            $product->name,
            $product->name_en ?? '',
            $product->sku,
            $product->barcode ?? '',
            $this->getItemTypeLabel($product->item_type),
            $product->category ? $product->category->name : '',
            number_format($product->price, 2),
            number_format($product->cost_price ?? 0, 2),
            $product->quantity ?? 0,
            $product->min_quantity ?? 0,
            $product->weight ?? 0,
            $product->description ?? '',
            $product->is_active ? 'نشط' : 'غير نشط',
            ($product->is_available ?? true) ? 'متاح' : 'غير متاح',
            ($product->is_featured ?? false) ? 'نعم' : 'لا',
            $product->image_url ?? '',
            $product->created_at ? $product->created_at->format('Y-m-d H:i:s') : '',
            $product->updated_at ? $product->updated_at->format('Y-m-d H:i:s') : '',
        ];
    }

    /**
     * Get item type label in Arabic
     */
    private function getItemTypeLabel($type)
    {
        $labels = [
            'food' => 'طعام',
            'beverage' => 'مشروب',
            'dessert' => 'حلوى',
            'product' => 'منتج',
            'service' => 'خدمة'
        ];

        return $labels[$type] ?? 'منتج';
    }

    /**
     * Apply styles to the worksheet
     */
    public function styles(Worksheet $sheet)
    {
        // Style the header row
        $sheet->getStyle('A1:S1')->applyFromArray([
            'font' => [
                'bold' => true,
                'size' => 12,
                'color' => ['rgb' => 'FFFFFF']
            ],
            'fill' => [
                'fillType' => Fill::FILL_SOLID,
                'startColor' => ['rgb' => '2563eb']
            ],
            'alignment' => [
                'horizontal' => Alignment::HORIZONTAL_CENTER,
                'vertical' => Alignment::VERTICAL_CENTER
            ]
        ]);

        // Apply general styles to all data
        $lastRow = $sheet->getHighestRow();
        $sheet->getStyle("A2:S{$lastRow}")->applyFromArray([
            'alignment' => [
                'horizontal' => Alignment::HORIZONTAL_RIGHT,
                'vertical' => Alignment::VERTICAL_CENTER
            ],
            'font' => [
                'size' => 10
            ]
        ]);

        // Center align certain columns
        $sheet->getStyle("A2:A{$lastRow}")->getAlignment()->setHorizontal(Alignment::HORIZONTAL_CENTER); // ID
        $sheet->getStyle("H2:H{$lastRow}")->getAlignment()->setHorizontal(Alignment::HORIZONTAL_CENTER); // Price
        $sheet->getStyle("I2:I{$lastRow}")->getAlignment()->setHorizontal(Alignment::HORIZONTAL_CENTER); // Cost
        $sheet->getStyle("J2:J{$lastRow}")->getAlignment()->setHorizontal(Alignment::HORIZONTAL_CENTER); // Quantity
        $sheet->getStyle("K2:K{$lastRow}")->getAlignment()->setHorizontal(Alignment::HORIZONTAL_CENTER); // Min Quantity
        $sheet->getStyle("L2:L{$lastRow}")->getAlignment()->setHorizontal(Alignment::HORIZONTAL_CENTER); // Weight

        // Add borders to all cells
        $sheet->getStyle("A1:S{$lastRow}")->applyFromArray([
            'borders' => [
                'allBorders' => [
                    'borderStyle' => \PhpOffice\PhpSpreadsheet\Style\Border::BORDER_THIN,
                    'color' => ['rgb' => 'CCCCCC']
                ]
            ]
        ]);

        return $sheet;
    }

    /**
     * Set column widths
     */
    public function columnWidths(): array
    {
        return [
            'A' => 8,   // ID
            'B' => 25,  // Arabic Name
            'C' => 25,  // English Name
            'D' => 15,  // SKU
            'E' => 15,  // Barcode
            'F' => 12,  // Item Type
            'G' => 20,  // Category
            'H' => 12,  // Price
            'I' => 12,  // Cost
            'J' => 10,  // Quantity
            'K' => 10,  // Min Quantity
            'L' => 10,  // Weight
            'M' => 30,  // Description
            'N' => 12,  // Status
            'O' => 12,  // Available
            'P' => 12,  // Featured
            'Q' => 30,  // Image URL
            'R' => 18,  // Created At
            'S' => 18,  // Updated At
        ];
    }

    /**
     * Set the worksheet title
     */
    public function title(): string
    {
        return 'قائمة المنتجات';
    }
}