<?php

namespace App\Exports;

use App\Models\InventoryMovement;
use App\Models\Product;
use Maatwebsite\Excel\Concerns\FromCollection;
use Maatwebsite\Excel\Concerns\WithHeadings;
use Maatwebsite\Excel\Concerns\WithMapping;
use Maatwebsite\Excel\Concerns\WithStyles;
use Maatwebsite\Excel\Concerns\ShouldAutoSize;
use Maatwebsite\Excel\Concerns\WithTitle;
use PhpOffice\PhpSpreadsheet\Worksheet\Worksheet;
use PhpOffice\PhpSpreadsheet\Style\Alignment;
use PhpOffice\PhpSpreadsheet\Style\Border;
use PhpOffice\PhpSpreadsheet\Style\Fill;

class InventoryReportExport implements FromCollection, WithHeadings, WithMapping, WithStyles, ShouldAutoSize, WithTitle
{
    protected $type;
    protected $dateFrom;
    protected $dateTo;
    protected $category;

    public function __construct($type = 'movements', $dateFrom = null, $dateTo = null, $category = null)
    {
        $this->type = $type;
        $this->dateFrom = $dateFrom;
        $this->dateTo = $dateTo;
        $this->category = $category;
    }

    /**
     * @return \Illuminate\Support\Collection
     */
    public function collection()
    {
        if ($this->type === 'stock') {
            return $this->getStockReport();
        } else {
            return $this->getMovementsReport();
        }
    }

    protected function getMovementsReport()
    {
        $query = InventoryMovement::with(['product', 'user'])
            ->orderBy('movement_date', 'desc');

        if ($this->dateFrom) {
            $query->whereDate('movement_date', '>=', $this->dateFrom);
        }

        if ($this->dateTo) {
            $query->whereDate('movement_date', '<=', $this->dateTo);
        }

        if ($this->category) {
            $query->whereHas('product', function($q) {
                $q->where('category', $this->category);
            });
        }

        return $query->get();
    }

    protected function getStockReport()
    {
        $query = Product::query();

        if ($this->category) {
            $query->where('category', $this->category);
        }

        return $query->get()->map(function($product) {
            return (object) [
                'product' => $product,
                'average_cost' => $product->getAverageCost(),
                'inventory_value' => $product->getCurrentInventoryValue(),
            ];
        });
    }

    /**
     * @return array
     */
    public function headings(): array
    {
        if ($this->type === 'stock') {
            return [
                'اسم المنتج',
                'رمز المنتج',
                'التصنيف',
                'الوحدة',
                'الكمية الحالية',
                'الحد الأدنى',
                'متوسط التكلفة',
                'قيمة المخزون',
                'حالة المخزون',
                'تاريخ آخر تحديث'
            ];
        } else {
            return [
                'التاريخ',
                'اسم المنتج',
                'رمز المنتج',
                'نوع الحركة',
                'الكمية',
                'تكلفة الوحدة',
                'التكلفة الإجمالية',
                'المرجع',
                'المستخدم',
                'ملاحظات'
            ];
        }
    }

    /**
     * @var mixed $row
     */
    public function map($row): array
    {
        if ($this->type === 'stock') {
            $product = $row->product;
            $isOutOfStock = $product->stock_quantity <= 0;
            $isLowStock = $product->stock_quantity <= $product->min_stock;
            
            $status = $isOutOfStock ? 'نافد' : ($isLowStock ? 'منخفض' : 'متوفر');

            return [
                $product->name,
                $product->sku,
                $product->category ?? '-',
                $product->unit,
                $product->stock_quantity,
                $product->min_stock,
                number_format($row->average_cost, 2),
                number_format($row->inventory_value, 2),
                $status,
                $product->updated_at->format('Y/m/d H:i')
            ];
        } else {
            $typeLabels = [
                'purchase' => 'شراء',
                'sale' => 'بيع',
                'adjustment_in' => 'تسوية إدخال',
                'adjustment_out' => 'تسوية إخراج',
                'return_in' => 'إرجاع إدخال',
                'return_out' => 'إرجاع إخراج',
                'transfer_in' => 'نقل إدخال',
                'transfer_out' => 'نقل إخراج'
            ];

            return [
                $row->movement_date->format('Y/m/d H:i'),
                $row->product->name,
                $row->product->sku,
                $typeLabels[$row->type] ?? $row->type,
                ($row->isInbound() ? '+' : '-') . number_format($row->quantity),
                $row->unit_cost ? number_format($row->unit_cost, 2) : '-',
                $row->total_cost ? number_format($row->total_cost, 2) : '-',
                $row->reference_type ? class_basename($row->reference_type) . ' #' . $row->reference_id : '-',
                $row->user->name,
                $row->notes ?? '-'
            ];
        }
    }

    /**
     * @param Worksheet $sheet
     * @return array
     */
    public function styles(Worksheet $sheet)
    {
        $highestRow = $sheet->getHighestRow();
        $highestColumn = $sheet->getHighestColumn();

        return [
            // تنسيق الرأس
            1 => [
                'font' => [
                    'bold' => true,
                    'color' => ['argb' => 'FFFFFF'],
                ],
                'fill' => [
                    'fillType' => Fill::FILL_SOLID,
                    'startColor' => ['argb' => '366092'],
                ],
                'alignment' => [
                    'horizontal' => Alignment::HORIZONTAL_CENTER,
                    'vertical' => Alignment::VERTICAL_CENTER,
                ],
            ],
            
            // تنسيق جميع الخلايا
            "A1:{$highestColumn}{$highestRow}" => [
                'borders' => [
                    'allBorders' => [
                        'borderStyle' => Border::BORDER_THIN,
                        'color' => ['argb' => '000000'],
                    ],
                ],
                'alignment' => [
                    'horizontal' => Alignment::HORIZONTAL_CENTER,
                    'vertical' => Alignment::VERTICAL_CENTER,
                ],
            ],
        ];
    }

    /**
     * @return string
     */
    public function title(): string
    {
        $titleMap = [
            'movements' => 'تقرير حركات المخزون',
            'stock' => 'تقرير حالة المخزون'
        ];

        return $titleMap[$this->type] ?? 'تقرير المخزون';
    }
}