<?php

namespace App\Exports;

use Maatwebsite\Excel\Concerns\FromArray;
use Maatwebsite\Excel\Concerns\WithHeadings;
use Maatwebsite\Excel\Concerns\WithStyles;
use Maatwebsite\Excel\Concerns\WithTitle;
use Maatwebsite\Excel\Concerns\WithColumnWidths;
use PhpOffice\PhpSpreadsheet\Worksheet\Worksheet;
use PhpOffice\PhpSpreadsheet\Style\Alignment;
use PhpOffice\PhpSpreadsheet\Style\Font;
use PhpOffice\PhpSpreadsheet\Style\Fill;
use PhpOffice\PhpSpreadsheet\Style\Border;

class IncomeStatementExport implements FromArray, WithHeadings, WithStyles, WithTitle, WithColumnWidths
{
    protected $data;
    protected $startDate;
    protected $endDate;
    protected $hasComparison;

    public function __construct($data, $startDate, $endDate, $hasComparison = false)
    {
        $this->data = $data;
        $this->startDate = $startDate;
        $this->endDate = $endDate;
        $this->hasComparison = $hasComparison;
    }

    public function array(): array
    {
        $rows = [];
        
        // معلومات التقرير
        $rows[] = ['قائمة الدخل (الأرباح والخسائر)'];
        $rows[] = ['الفترة: من ' . $this->startDate . ' إلى ' . $this->endDate];
        $rows[] = ['تاريخ الإنشاء: ' . now()->format('Y-m-d H:i:s')];
        $rows[] = [''];
        $rows[] = [''];

        // الإيرادات
        $rows[] = ['الإيرادات', '', $this->hasComparison ? '' : ''];
        foreach ($this->data['revenue_accounts'] as $account) {
            $row = [
                '   ' . $account['name'],
                number_format($account['amount'], 2)
            ];
            if ($this->hasComparison) {
                $row[] = number_format($account['comparison_amount'] ?? 0, 2);
            }
            $rows[] = $row;
        }
        
        $totalRevenueRow = [
            'إجمالي الإيرادات',
            number_format($this->data['total_revenue'], 2)
        ];
        if ($this->hasComparison) {
            $totalRevenueRow[] = number_format($this->data['comparison_data']['total_revenue'] ?? 0, 2);
        }
        $rows[] = $totalRevenueRow;
        $rows[] = [''];

        // تكلفة البضاعة المباعة
        if ($this->data['total_cogs'] > 0) {
            $rows[] = ['تكلفة البضاعة المباعة', '', $this->hasComparison ? '' : ''];
            foreach ($this->data['cogs_accounts'] as $account) {
                $row = [
                    '   ' . $account['name'],
                    number_format($account['amount'], 2)
                ];
                if ($this->hasComparison) {
                    $row[] = number_format($account['comparison_amount'] ?? 0, 2);
                }
                $rows[] = $row;
            }
            
            $totalCogsRow = [
                'إجمالي تكلفة البضاعة المباعة',
                number_format($this->data['total_cogs'], 2)
            ];
            if ($this->hasComparison) {
                $totalCogsRow[] = number_format($this->data['comparison_data']['total_cogs'] ?? 0, 2);
            }
            $rows[] = $totalCogsRow;
            
            $grossProfitRow = [
                'إجمالي الربح',
                number_format($this->data['gross_profit'], 2)
            ];
            if ($this->hasComparison) {
                $grossProfitRow[] = number_format($this->data['comparison_data']['gross_profit'] ?? 0, 2);
            }
            $rows[] = $grossProfitRow;
            $rows[] = [''];
        }

        // المصروفات
        $rows[] = ['المصروفات', '', $this->hasComparison ? '' : ''];
        foreach ($this->data['expense_accounts'] as $account) {
            $row = [
                '   ' . $account['name'],
                number_format($account['amount'], 2)
            ];
            if ($this->hasComparison) {
                $row[] = number_format($account['comparison_amount'] ?? 0, 2);
            }
            $rows[] = $row;
        }
        
        $totalExpensesRow = [
            'إجمالي المصروفات',
            number_format($this->data['total_expenses'], 2)
        ];
        if ($this->hasComparison) {
            $totalExpensesRow[] = number_format($this->data['comparison_data']['total_expenses'] ?? 0, 2);
        }
        $rows[] = $totalExpensesRow;
        $rows[] = [''];

        // النتيجة النهائية
        $rows[] = ['النتيجة النهائية', '', $this->hasComparison ? '' : ''];
        
        $totalRevenueResultRow = [
            'إجمالي الإيرادات',
            number_format($this->data['total_revenue'], 2)
        ];
        if ($this->hasComparison) {
            $totalRevenueResultRow[] = number_format($this->data['comparison_data']['total_revenue'] ?? 0, 2);
        }
        $rows[] = $totalRevenueResultRow;

        if ($this->data['total_cogs'] > 0) {
            $cogsResultRow = [
                'ناقص: تكلفة البضاعة المباعة',
                number_format($this->data['total_cogs'], 2)
            ];
            if ($this->hasComparison) {
                $cogsResultRow[] = number_format($this->data['comparison_data']['total_cogs'] ?? 0, 2);
            }
            $rows[] = $cogsResultRow;
            
            $grossProfitResultRow = [
                'إجمالي الربح',
                number_format($this->data['gross_profit'], 2)
            ];
            if ($this->hasComparison) {
                $grossProfitResultRow[] = number_format($this->data['comparison_data']['gross_profit'] ?? 0, 2);
            }
            $rows[] = $grossProfitResultRow;
        }

        $expensesResultRow = [
            'ناقص: إجمالي المصروفات',
            number_format($this->data['total_expenses'], 2)
        ];
        if ($this->hasComparison) {
            $expensesResultRow[] = number_format($this->data['comparison_data']['total_expenses'] ?? 0, 2);
        }
        $rows[] = $expensesResultRow;

        $netIncomeLabel = $this->data['net_income'] >= 0 ? 'صافي الربح' : 'صافي الخسارة';
        $netIncomeRow = [
            $netIncomeLabel,
            number_format($this->data['net_income'], 2)
        ];
        if ($this->hasComparison) {
            $netIncomeRow[] = number_format($this->data['comparison_data']['net_income'] ?? 0, 2);
        }
        $rows[] = $netIncomeRow;
        $rows[] = [''];

        // النسب المالية
        $rows[] = ['النسب المالية', '', $this->hasComparison ? '' : ''];
        
        $grossProfitMargin = $this->data['total_revenue'] > 0 
            ? (($this->data['gross_profit'] ?? 0) / $this->data['total_revenue']) * 100 
            : 0;
        
        $netProfitMargin = $this->data['total_revenue'] > 0 
            ? ($this->data['net_income'] / $this->data['total_revenue']) * 100 
            : 0;
        
        $expenseRatio = $this->data['total_revenue'] > 0 
            ? ($this->data['total_expenses'] / $this->data['total_revenue']) * 100 
            : 0;

        $rows[] = [
            'هامش الربح الإجمالي',
            number_format($grossProfitMargin, 2) . '%',
            $this->hasComparison ? '' : ''
        ];
        
        $rows[] = [
            'هامش الربح الصافي',
            number_format($netProfitMargin, 2) . '%',
            $this->hasComparison ? '' : ''
        ];
        
        $rows[] = [
            'نسبة المصروفات',
            number_format($expenseRatio, 2) . '%',
            $this->hasComparison ? '' : ''
        ];

        return $rows;
    }

    public function headings(): array
    {
        $headings = ['البيان', 'الفترة الحالية (ريال)'];
        
        if ($this->hasComparison) {
            $headings[] = 'فترة المقارنة (ريال)';
        }
        
        return $headings;
    }

    public function styles(Worksheet $sheet)
    {
        $styles = [
            // تنسيق العنوان الرئيسي
            1 => [
                'font' => ['bold' => true, 'size' => 18],
                'alignment' => ['horizontal' => Alignment::HORIZONTAL_CENTER]
            ],
            
            // محاذاة الأرقام إلى اليمين
            'B:C' => [
                'alignment' => ['horizontal' => Alignment::HORIZONTAL_RIGHT]
            ]
        ];

        // تنسيق عناوين الأقسام
        $sectionTitles = ['الإيرادات', 'تكلفة البضاعة المباعة', 'المصروفات', 'النتيجة النهائية', 'النسب المالية'];
        
        foreach ($sheet->getRowIterator() as $row) {
            $rowIndex = $row->getRowIndex();
            $cellValue = $sheet->getCell('A' . $rowIndex)->getValue();
            
            // تنسيق عناوين الأقسام
            if (in_array($cellValue, $sectionTitles)) {
                $styles['A' . $rowIndex . ':' . ($this->hasComparison ? 'C' : 'B') . $rowIndex] = [
                    'font' => ['bold' => true, 'size' => 14],
                    'fill' => [
                        'fillType' => Fill::FILL_SOLID,
                        'startColor' => ['rgb' => 'E3F2FD']
                    ],
                    'borders' => [
                        'outline' => ['borderStyle' => Border::BORDER_MEDIUM]
                    ]
                ];
            }
            
            // تنسيق الإجماليات
            if (strpos($cellValue, 'إجمالي') === 0 || strpos($cellValue, 'صافي') === 0) {
                $styles['A' . $rowIndex . ':' . ($this->hasComparison ? 'C' : 'B') . $rowIndex] = [
                    'font' => ['bold' => true],
                    'borders' => [
                        'top' => ['borderStyle' => Border::BORDER_THIN],
                        'bottom' => ['borderStyle' => Border::BORDER_THIN]
                    ]
                ];
            }
        }

        return $styles;
    }

    public function title(): string
    {
        return 'قائمة الدخل';
    }

    public function columnWidths(): array
    {
        $widths = [
            'A' => 40,
            'B' => 20
        ];
        
        if ($this->hasComparison) {
            $widths['C'] = 20;
        }
        
        return $widths;
    }
}