<?php

namespace App\Exports;

use App\Models\FiscalYear;
use Maatwebsite\Excel\Concerns\FromCollection;
use Maatwebsite\Excel\Concerns\WithHeadings;
use Maatwebsite\Excel\Concerns\WithMapping;
use Maatwebsite\Excel\Concerns\WithStyles;
use Maatwebsite\Excel\Concerns\WithTitle;
use Maatwebsite\Excel\Concerns\ShouldAutoSize;
use PhpOffice\PhpSpreadsheet\Worksheet\Worksheet;
use PhpOffice\PhpSpreadsheet\Style\Alignment;
use PhpOffice\PhpSpreadsheet\Style\Font;
use PhpOffice\PhpSpreadsheet\Style\Fill;

class FiscalYearExport implements FromCollection, WithHeadings, WithMapping, WithStyles, WithTitle, ShouldAutoSize
{
    protected $fiscalYears;
    protected $title;

    public function __construct($fiscalYears, $title = 'تقرير السنوات المالية')
    {
        $this->fiscalYears = $fiscalYears;
        $this->title = $title;
    }

    /**
     * @return \Illuminate\Support\Collection
     */
    public function collection()
    {
        return $this->fiscalYears;
    }

    /**
     * @return array
     */
    public function headings(): array
    {
        return [
            'الرقم',
            'اسم السنة المالية',
            'تاريخ البداية',
            'تاريخ النهاية',
            'عدد الأيام',
            'الأيام المنقضية',
            'الأيام المتبقية',
            'نسبة الإنجاز',
            'الحالة',
            'سنة حالية',
            'مغلقة',
            'الرصيد الافتتاحي',
            'الرصيد الختامي',
            'عدد القيود',
            'عدد الميزانيات',
            'تاريخ الإغلاق',
            'مُغلقة بواسطة',
            'ملاحظات',
            'تاريخ الإنشاء',
        ];
    }

    /**
     * @param mixed $fiscalYear
     * @return array
     */
    public function map($fiscalYear): array
    {
        return [
            $fiscalYear->id,
            $fiscalYear->name,
            $fiscalYear->start_date->format('Y-m-d'),
            $fiscalYear->end_date->format('Y-m-d'),
            $fiscalYear->total_days,
            $fiscalYear->elapsed_days,
            $fiscalYear->remaining_days,
            number_format($fiscalYear->completion_percentage, 2) . '%',
            $this->getStatusText($fiscalYear),
            $fiscalYear->is_current ? 'نعم' : 'لا',
            $fiscalYear->is_closed ? 'نعم' : 'لا',
            number_format($fiscalYear->opening_balance_total, 2),
            number_format($fiscalYear->closing_balance_total ?? 0, 2),
            $fiscalYear->journalEntries()->count(),
            $fiscalYear->budgets()->count(),
            $fiscalYear->closed_at ? $fiscalYear->closed_at->format('Y-m-d H:i') : '',
            $fiscalYear->closedBy->name ?? '',
            $fiscalYear->notes,
            $fiscalYear->created_at->format('Y-m-d H:i'),
        ];
    }

    /**
     * @param Worksheet $sheet
     * @return array
     */
    public function styles(Worksheet $sheet)
    {
        return [
            // تنسيق العنوان
            1 => [
                'font' => [
                    'bold' => true,
                    'size' => 12,
                    'color' => ['rgb' => 'FFFFFF'],
                ],
                'fill' => [
                    'fillType' => Fill::FILL_SOLID,
                    'startColor' => ['rgb' => '27AE60'],
                ],
                'alignment' => [
                    'horizontal' => Alignment::HORIZONTAL_CENTER,
                    'vertical' => Alignment::VERTICAL_CENTER,
                ],
            ],
            // تنسيق الخلايا
            'A:S' => [
                'alignment' => [
                    'horizontal' => Alignment::HORIZONTAL_CENTER,
                    'vertical' => Alignment::VERTICAL_CENTER,
                ],
            ],
            // تنسيق الأرقام والتواريخ
            'C:D' => [
                'alignment' => [
                    'horizontal' => Alignment::HORIZONTAL_RIGHT,
                ],
            ],
            'E:H' => [
                'alignment' => [
                    'horizontal' => Alignment::HORIZONTAL_RIGHT,
                ],
            ],
            'L:M' => [
                'alignment' => [
                    'horizontal' => Alignment::HORIZONTAL_RIGHT,
                ],
            ],
            'N:O' => [
                'alignment' => [
                    'horizontal' => Alignment::HORIZONTAL_RIGHT,
                ],
            ],
        ];
    }

    /**
     * @return string
     */
    public function title(): string
    {
        return $this->title;
    }

    /**
     * Get status text in Arabic
     */
    private function getStatusText($fiscalYear)
    {
        if ($fiscalYear->is_closed) {
            return 'مغلقة';
        } elseif ($fiscalYear->is_current) {
            return 'حالية';
        } elseif ($fiscalYear->start_date->isFuture()) {
            return 'مستقبلية';
        } elseif ($fiscalYear->end_date->isPast()) {
            return 'منتهية';
        } else {
            return 'نشطة';
        }
    }
}