<?php

namespace App\Exports;

use Maatwebsite\Excel\Concerns\FromCollection;
use Maatwebsite\Excel\Concerns\WithHeadings;
use Maatwebsite\Excel\Concerns\WithMapping;
use Maatwebsite\Excel\Concerns\WithStyles;
use Maatwebsite\Excel\Concerns\WithTitle;
use Maatwebsite\Excel\Concerns\WithColumnWidths;
use PhpOffice\PhpSpreadsheet\Worksheet\Worksheet;
use PhpOffice\PhpSpreadsheet\Style\Alignment;
use PhpOffice\PhpSpreadsheet\Style\Font;
use PhpOffice\PhpSpreadsheet\Style\Fill;
use PhpOffice\PhpSpreadsheet\Style\Border;

class CashTransactionsExport implements FromCollection, WithHeadings, WithMapping, WithStyles, WithTitle, WithColumnWidths
{
    protected $transactions;
    protected $filters;

    public function __construct($transactions, $filters = [])
    {
        $this->transactions = $transactions;
        $this->filters = $filters;
    }

    public function collection()
    {
        return $this->transactions;
    }

    public function map($transaction): array
    {
        return [
            $transaction->transaction_date->format('Y-m-d'),
            $transaction->transaction_number,
            $transaction->type === 'receipt' ? 'استلام نقدي' : 'دفعة نقدية',
            $transaction->cashAccount->code . ' - ' . $transaction->cashAccount->name,
            $transaction->description,
            $transaction->received_from ?? $transaction->paid_to ?? '',
            number_format($transaction->amount, 2),
            $transaction->payment_method ?? '',
            $transaction->reference ?? '',
            $transaction->creator ? $transaction->creator->name : '',
            $transaction->created_at->format('Y-m-d H:i'),
        ];
    }

    public function headings(): array
    {
        return [
            'التاريخ',
            'رقم المعاملة',
            'نوع المعاملة',
            'الحساب النقدي',
            'الوصف',
            'من/إلى',
            'المبلغ (ريال)',
            'طريقة الدفع',
            'المرجع',
            'المنشئ',
            'تاريخ الإنشاء',
        ];
    }

    public function styles(Worksheet $sheet)
    {
        return [
            // تنسيق الرأس
            1 => [
                'font' => ['bold' => true, 'size' => 12],
                'fill' => [
                    'fillType' => Fill::FILL_SOLID,
                    'startColor' => ['rgb' => '4472C4']
                ],
                'alignment' => [
                    'horizontal' => Alignment::HORIZONTAL_CENTER,
                    'vertical' => Alignment::VERTICAL_CENTER
                ],
                'borders' => [
                    'allBorders' => ['borderStyle' => Border::BORDER_THIN]
                ]
            ],
            
            // محاذاة الأرقام إلى اليمين
            'G:G' => [
                'alignment' => ['horizontal' => Alignment::HORIZONTAL_RIGHT]
            ],
            
            // محاذاة التواريخ إلى الوسط
            'A:A' => [
                'alignment' => ['horizontal' => Alignment::HORIZONTAL_CENTER]
            ],
            'K:K' => [
                'alignment' => ['horizontal' => Alignment::HORIZONTAL_CENTER]
            ],
            
            // محاذاة رقم المعاملة ونوع المعاملة إلى الوسط
            'B:B' => [
                'alignment' => ['horizontal' => Alignment::HORIZONTAL_CENTER]
            ],
            'C:C' => [
                'alignment' => ['horizontal' => Alignment::HORIZONTAL_CENTER]
            ],
            
            // محاذاة طريقة الدفع إلى الوسط
            'H:H' => [
                'alignment' => ['horizontal' => Alignment::HORIZONTAL_CENTER]
            ],
        ];
    }

    public function title(): string
    {
        return 'المعاملات النقدية';
    }

    public function columnWidths(): array
    {
        return [
            'A' => 12, // التاريخ
            'B' => 15, // رقم المعاملة
            'C' => 15, // نوع المعاملة
            'D' => 25, // الحساب النقدي
            'E' => 30, // الوصف
            'F' => 20, // من/إلى
            'G' => 15, // المبلغ
            'H' => 15, // طريقة الدفع
            'I' => 15, // المرجع
            'J' => 15, // المنشئ
            'K' => 18, // تاريخ الإنشاء
        ];
    }
}