<?php

namespace App\Exports;

use App\Models\Budget;
use Maatwebsite\Excel\Concerns\FromCollection;
use Maatwebsite\Excel\Concerns\WithHeadings;
use Maatwebsite\Excel\Concerns\WithMapping;
use Maatwebsite\Excel\Concerns\WithStyles;
use Maatwebsite\Excel\Concerns\WithTitle;
use Maatwebsite\Excel\Concerns\ShouldAutoSize;
use PhpOffice\PhpSpreadsheet\Worksheet\Worksheet;
use PhpOffice\PhpSpreadsheet\Style\Alignment;
use PhpOffice\PhpSpreadsheet\Style\Font;
use PhpOffice\PhpSpreadsheet\Style\Fill;

class BudgetExport implements FromCollection, WithHeadings, WithMapping, WithStyles, WithTitle, ShouldAutoSize
{
    protected $budgets;
    protected $title;

    public function __construct($budgets, $title = 'تقرير الميزانيات')
    {
        $this->budgets = $budgets;
        $this->title = $title;
    }

    /**
     * @return \Illuminate\Support\Collection
     */
    public function collection()
    {
        return $this->budgets;
    }

    /**
     * @return array
     */
    public function headings(): array
    {
        return [
            'الرقم',
            'اسم الميزانية',
            'السنة المالية',
            'مركز التكلفة',
            'الحساب',
            'نوع الفترة',
            'تاريخ البداية',
            'تاريخ النهاية',
            'المبلغ المخطط',
            'المبلغ الفعلي',
            'المبلغ المتبقي',
            'نسبة الاستهلاك',
            'نسبة الانحراف',
            'الحالة',
            'مستوى التنبيه',
            'تاريخ الإنشاء',
            'المنشئ',
        ];
    }

    /**
     * @param mixed $budget
     * @return array
     */
    public function map($budget): array
    {
        return [
            $budget->id,
            $budget->name,
            $budget->fiscalYear->name ?? '',
            $budget->costCenter->name ?? 'جميع المراكز',
            $budget->account->name ?? 'جميع الحسابات',
            $this->getPeriodTypeText($budget->period_type),
            $budget->start_date->format('Y-m-d'),
            $budget->end_date->format('Y-m-d'),
            number_format($budget->budgeted_amount, 2),
            number_format($budget->actual_amount, 2),
            number_format($budget->remaining_amount, 2),
            number_format($budget->consumption_percentage, 2) . '%',
            number_format($budget->variance_percentage, 2) . '%',
            $this->getStatusText($budget->status),
            $this->getAlertLevelText($budget),
            $budget->created_at->format('Y-m-d H:i'),
            $budget->creator->name ?? '',
        ];
    }

    /**
     * @param Worksheet $sheet
     * @return array
     */
    public function styles(Worksheet $sheet)
    {
        return [
            // تنسيق العنوان
            1 => [
                'font' => [
                    'bold' => true,
                    'size' => 12,
                    'color' => ['rgb' => 'FFFFFF'],
                ],
                'fill' => [
                    'fillType' => Fill::FILL_SOLID,
                    'startColor' => ['rgb' => '2E86C1'],
                ],
                'alignment' => [
                    'horizontal' => Alignment::HORIZONTAL_CENTER,
                    'vertical' => Alignment::VERTICAL_CENTER,
                ],
            ],
            // تنسيق الخلايا
            'A:Q' => [
                'alignment' => [
                    'horizontal' => Alignment::HORIZONTAL_CENTER,
                    'vertical' => Alignment::VERTICAL_CENTER,
                ],
            ],
            // تنسيق الأرقام
            'I:M' => [
                'alignment' => [
                    'horizontal' => Alignment::HORIZONTAL_RIGHT,
                ],
            ],
        ];
    }

    /**
     * @return string
     */
    public function title(): string
    {
        return $this->title;
    }

    /**
     * Get period type text in Arabic
     */
    private function getPeriodTypeText($periodType)
    {
        $types = [
            'monthly' => 'شهري',
            'quarterly' => 'ربع سنوي',
            'annual' => 'سنوي',
        ];

        return $types[$periodType] ?? $periodType;
    }

    /**
     * Get status text in Arabic
     */
    private function getStatusText($status)
    {
        $statuses = [
            'draft' => 'مسودة',
            'approved' => 'معتمد',
            'active' => 'نشط',
            'closed' => 'مغلق',
            'cancelled' => 'ملغي',
        ];

        return $statuses[$status] ?? $status;
    }

    /**
     * Get alert level text
     */
    private function getAlertLevelText($budget)
    {
        if ($budget->consumption_percentage >= $budget->critical_threshold) {
            return 'حرج';
        } elseif ($budget->consumption_percentage >= $budget->warning_threshold) {
            return 'تحذير';
        } else {
            return 'طبيعي';
        }
    }
}