<?php

namespace App\Exports;

use Maatwebsite\Excel\Concerns\FromArray;
use Maatwebsite\Excel\Concerns\WithHeadings;
use Maatwebsite\Excel\Concerns\WithStyles;
use Maatwebsite\Excel\Concerns\WithTitle;
use Maatwebsite\Excel\Concerns\WithColumnWidths;
use PhpOffice\PhpSpreadsheet\Worksheet\Worksheet;
use PhpOffice\PhpSpreadsheet\Style\Alignment;
use PhpOffice\PhpSpreadsheet\Style\Font;
use PhpOffice\PhpSpreadsheet\Style\Fill;
use PhpOffice\PhpSpreadsheet\Style\Border;

class BalanceSheetExport implements FromArray, WithHeadings, WithStyles, WithTitle, WithColumnWidths
{
    protected $data;
    protected $asOfDate;

    public function __construct($data, $asOfDate)
    {
        $this->data = $data;
        $this->asOfDate = $asOfDate;
    }

    public function array(): array
    {
        $rows = [];
        
        // معلومات التقرير
        $rows[] = ['الميزانية العمومية'];
        $rows[] = ['في تاريخ: ' . $this->asOfDate];
        $rows[] = ['تاريخ الإنشاء: ' . now()->format('Y-m-d H:i:s')];
        $rows[] = [''];
        $rows[] = [''];

        // الأصول
        $rows[] = ['الأصول', '', ''];
        $rows[] = [''];

        // الأصول المتداولة
        $rows[] = ['الأصول المتداولة', '', ''];
        foreach ($this->data['current_assets'] as $asset) {
            $rows[] = [
                '   ' . $asset['name'],
                number_format($asset['balance'], 2),
                ''
            ];
        }
        $rows[] = [
            'إجمالي الأصول المتداولة',
            number_format($this->data['total_current_assets'], 2),
            ''
        ];
        $rows[] = [''];

        // الأصول الثابتة
        if (!empty($this->data['fixed_assets'])) {
            $rows[] = ['الأصول الثابتة', '', ''];
            foreach ($this->data['fixed_assets'] as $asset) {
                $rows[] = [
                    '   ' . $asset['name'],
                    number_format($asset['balance'], 2),
                    ''
                ];
            }
            $rows[] = [
                'إجمالي الأصول الثابتة',
                number_format($this->data['total_fixed_assets'], 2),
                ''
            ];
            $rows[] = [''];
        }

        // إجمالي الأصول
        $rows[] = [
            'إجمالي الأصول',
            number_format($this->data['total_assets'], 2),
            ''
        ];
        $rows[] = [''];
        $rows[] = [''];

        // الخصوم وحقوق الملكية
        $rows[] = ['الخصوم وحقوق الملكية', '', ''];
        $rows[] = [''];

        // الخصوم المتداولة
        if (!empty($this->data['current_liabilities'])) {
            $rows[] = ['الخصوم المتداولة', '', ''];
            foreach ($this->data['current_liabilities'] as $liability) {
                $rows[] = [
                    '   ' . $liability['name'],
                    number_format($liability['balance'], 2),
                    ''
                ];
            }
            $rows[] = [
                'إجمالي الخصوم المتداولة',
                number_format($this->data['total_current_liabilities'], 2),
                ''
            ];
            $rows[] = [''];
        }

        // الخصوم طويلة الأجل
        if (!empty($this->data['long_term_liabilities'])) {
            $rows[] = ['الخصوم طويلة الأجل', '', ''];
            foreach ($this->data['long_term_liabilities'] as $liability) {
                $rows[] = [
                    '   ' . $liability['name'],
                    number_format($liability['balance'], 2),
                    ''
                ];
            }
            $rows[] = [
                'إجمالي الخصوم طويلة الأجل',
                number_format($this->data['total_long_term_liabilities'], 2),
                ''
            ];
            $rows[] = [''];
        }

        // إجمالي الخصوم
        $rows[] = [
            'إجمالي الخصوم',
            number_format($this->data['total_liabilities'], 2),
            ''
        ];
        $rows[] = [''];

        // حقوق الملكية
        $rows[] = ['حقوق الملكية', '', ''];
        foreach ($this->data['equity_accounts'] as $equity) {
            $rows[] = [
                '   ' . $equity['name'],
                number_format($equity['balance'], 2),
                ''
            ];
        }
        $rows[] = [
            '   صافي الدخل للفترة الحالية',
            number_format($this->data['retained_earnings'], 2),
            ''
        ];
        $rows[] = [
            'إجمالي حقوق الملكية',
            number_format($this->data['total_equity'], 2),
            ''
        ];
        $rows[] = [''];

        // إجمالي الخصوم وحقوق الملكية
        $rows[] = [
            'إجمالي الخصوم وحقوق الملكية',
            number_format($this->data['total_liabilities_and_equity'], 2),
            ''
        ];
        $rows[] = [''];

        // النسب المالية
        $rows[] = ['النسب المالية الرئيسية', '', ''];
        $rows[] = [''];
        
        $currentRatio = $this->data['total_current_liabilities'] > 0 
            ? $this->data['total_current_assets'] / $this->data['total_current_liabilities'] 
            : 0;
        
        $equityRatio = $this->data['total_assets'] > 0 
            ? ($this->data['total_equity'] / $this->data['total_assets']) * 100 
            : 0;
        
        $debtToEquityRatio = $this->data['total_equity'] > 0 
            ? $this->data['total_liabilities'] / $this->data['total_equity'] 
            : 0;

        $rows[] = [
            'نسبة السيولة الجارية',
            number_format($currentRatio, 2),
            'مرة'
        ];
        $rows[] = [
            'نسبة حقوق الملكية',
            number_format($equityRatio, 2),
            '%'
        ];
        $rows[] = [
            'نسبة الدين إلى حقوق الملكية',
            number_format($debtToEquityRatio, 2),
            'مرة'
        ];

        return $rows;
    }

    public function headings(): array
    {
        return [
            'البيان',
            'المبلغ (ريال)',
            'وحدة'
        ];
    }

    public function styles(Worksheet $sheet)
    {
        return [
            // تنسيق العنوان الرئيسي
            1 => [
                'font' => ['bold' => true, 'size' => 18],
                'alignment' => ['horizontal' => Alignment::HORIZONTAL_CENTER]
            ],
            
            // تنسيق عناوين الأقسام الرئيسية
            'A6:C6' => [ // الأصول
                'font' => ['bold' => true, 'size' => 14],
                'fill' => [
                    'fillType' => Fill::FILL_SOLID,
                    'startColor' => ['rgb' => 'E8F5E8']
                ],
                'borders' => [
                    'outline' => ['borderStyle' => Border::BORDER_MEDIUM]
                ]
            ],
            
            // تنسيق عناوين الأقسام الفرعية
            'A8:C8' => [ // الأصول المتداولة
                'font' => ['bold' => true, 'size' => 12],
                'fill' => [
                    'fillType' => Fill::FILL_SOLID,
                    'startColor' => ['rgb' => 'F0F8F0']
                ]
            ],
            
            // محاذاة الأرقام إلى اليمين
            'B:B' => [
                'alignment' => ['horizontal' => Alignment::HORIZONTAL_RIGHT]
            ],
            
            // محاذاة الوحدات إلى الوسط
            'C:C' => [
                'alignment' => ['horizontal' => Alignment::HORIZONTAL_CENTER]
            ],
            
            // تنسيق الإجماليات
            'A' => function($sheet) {
                foreach ($sheet->getRowIterator() as $row) {
                    $cellValue = $sheet->getCell('A' . $row->getRowIndex())->getValue();
                    if (strpos($cellValue, 'إجمالي') === 0) {
                        $sheet->getStyle('A' . $row->getRowIndex() . ':C' . $row->getRowIndex())
                              ->getFont()->setBold(true);
                        $sheet->getStyle('A' . $row->getRowIndex() . ':C' . $row->getRowIndex())
                              ->getBorders()->getTop()->setBorderStyle(Border::BORDER_THIN);
                    }
                }
            }
        ];
    }

    public function title(): string
    {
        return 'الميزانية العمومية';
    }

    public function columnWidths(): array
    {
        return [
            'A' => 45,
            'B' => 20,
            'C' => 10
        ];
    }
}